package com.bokesoft.yigo.meta.datamigration;

import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.LinkedList;

import com.bokesoft.yes.common.struct.StringLinkedHashMap;
import com.bokesoft.yes.meta.datamigration.calculate.MetaMigrationParas;
import com.bokesoft.yigo.meta.base.AbstractMetaObject;
import com.bokesoft.yigo.meta.base.IMetaEnv;
import com.bokesoft.yigo.meta.base.KeyPairMetaObject;
import com.bokesoft.yigo.meta.base.MetaException;
import com.bokesoft.yigo.meta.datamigration.error.MetaDataMigrationErrorInfo;
import com.bokesoft.yigo.meta.datamigration.error.MetaDataMigrationErrorInfoCollection;
import com.bokesoft.yigo.meta.factory.IMetaFactory;
import com.bokesoft.yigo.meta.form.IPropertyMerger;
import com.bokesoft.yigo.meta.form.MetaForm;
import com.bokesoft.yigo.meta.intf.IMetaProject;

/**
 * MetaDataMigration定义数据迁移的关系，为数据迁移元数据的主类，主要定义如下条件
 * <ul>
 * <li>迁移的状态字段标识，是迁移发生所依据的条件字段的标识，通常情况下是单据的状态字段(Status)
 * <li>迁移状态字段的阀值，当状态字段的值大于或等于该值时，迁移发生，当状态字段的值小于该时，取消迁移
 * <li>迁移的数据条件，当迁移发生时或取消时，只对满足该条件的数据进行处理
 * </ul>
 * <p>
 * 迁移关系同时数据定义迁移的源和目标定义，其中源可以来源于数据对象中的多个表，目标只能来源于数据对象中的一个表。数据来源的定义详见
 * {@link MetaDMSourceTableCollection} ，数据目标的定义说明见
 * {@link MetaDMTargetTableCollection}。
 * </p>
 * 
 * @author 王元和
 * @since YES 1.0
 * @see MetaDMSourceTableCollection
 * @see MetaDMTargetTableCollection
 */
public class MetaDataMigration extends KeyPairMetaObject implements IPropertyMerger<MetaDataMigration> {
	/** 迁移的标识 */
	private String key = "";
	/** 迁移的名称 */
	private String caption = "";
	/** 迁移的描述 */
	private String description = "";
	/** 源数据对象标识 */
	private String srcDataObjectKey = "";
	/** 目标数据对象标识 */
	private String tgtDataObjectKey = "";

	/** 迁移的状态字段的标识，用于控制整个数据对象的条件 */
	private String statusFieldKey = "";
	/*重新迁移使用的Sql，一般用于多表关系的sql片段 */
	private String fromPartInSql = "";

	/** 迁移发生的状态字段的条件值(字符串表示) */
	private String statusValue = "";

	/** 迁移的数据条件，用于数据单条源数据的条件，表达式定义 */
	private String condition = "";
	
	/** 迁移的总条件，用于整个迁移表的条件，表达式定义 */
	private String headCondition = "";

	/** 迁移的数据来源表集合 */
	private MetaDMSourceTableCollection sourceTableCollection = null;

	/** 迁移的数据目标表集合 */
	private MetaDMTargetTableCollection targetTableCollection = null;

	/** 解析以后的一些迁移参数 */
	private MetaMigrationParas migrationParas = null;
	
	/**自定义异常集合*/
	private MetaDataMigrationErrorInfoCollection errorInfoCollection = null;
	
	/** 迁移所在的工程*/
	private IMetaProject project = null;
	/** 继承的数据迁移标识 */
	private String extend = "";

	private String mergeToSourceMapKey = null;
	@Override
	public void getChildMetaObjects(LinkedList<Object> list) {
		list.add(sourceTableCollection);
		list.add(targetTableCollection);
		list.add(errorInfoCollection);
	}

	public static final String TAG_NAME = "DataMigration";

	@Override
	public String getTagName() {
		return TAG_NAME;
	}

	@Override
	public AbstractMetaObject createChildMetaObject(IMetaEnv env, Object doc, Object node, String tagName, Object flag, int runType) throws MetaException {
		AbstractMetaObject resultMetaObject = null;
		if (MetaDMSourceTableCollection.TAG_NAME.equals(tagName)) {
			this.sourceTableCollection = new MetaDMSourceTableCollection();
			resultMetaObject = sourceTableCollection;
		} else if (MetaDMTargetTableCollection.TAG_NAME.equals(tagName)) {
			this.targetTableCollection = new MetaDMTargetTableCollection();
			resultMetaObject = targetTableCollection;
		}else if (MetaDataMigrationErrorInfoCollection.TAG_NAME.equals(tagName)) {
			this.errorInfoCollection = new MetaDataMigrationErrorInfoCollection();
			resultMetaObject = errorInfoCollection;
		}
		return resultMetaObject;
	}

	/**
	 * 设置迁移的标识
	 * 
	 * @param key
	 *            迁移的标识
	 */
	public void setKey(String key) {
		this.key = key;
	}

	/**
	 * 取得迁移的标识
	 * 
	 * @return 迁移的标识
	 */
	@Override
	public String getKey() {
		return this.key;
	}

	public String getMergeToSourceMapKey() {
		return mergeToSourceMapKey;
	}

	public void setMergeToSourceMapKey(String mergeToSourceMapKey) {
		this.mergeToSourceMapKey = mergeToSourceMapKey;
	}

	/**
	 * 设置迁移的名称
	 * 
	 * @param caption
	 *            迁移的名称
	 */
	public void setCaption(String caption) {
		this.caption = caption;
	}

	/**
	 * 取得迁移的名称
	 * 
	 * @return 迁移的名称
	 */
	public String getCaption() {
		return this.caption;
	}

	/**
	 * 设置迁移的描述
	 * 
	 * @param description
	 *            迁移的描述
	 */
	public void setDescription(String description) {
		this.description = description;
	}

	/**
	 * 取得迁移的描述
	 * 
	 * @return 迁移的描述
	 */
	public String getDescription() {
		return this.description;
	}

	/**
	 * 设置迁移状态字段的标识
	 * 
	 * @param statusFieldKey
	 *            数据对象中字段的标识
	 */
	public void setStatusFieldKey(String statusFieldKey) {
		this.statusFieldKey = statusFieldKey;
	}

	/**
	 * 取得迁移状态字段的标识
	 * 
	 * @return 字段标识
	 */
	public String getStatusFieldKey() {
		return this.statusFieldKey;
	}

	/**
	 * 设置迁移状态值
	 * 
	 * @param statusValue
	 *            状态值
	 */
	public void setStatusValue(String statusValue) {
		this.statusValue = statusValue;
	}

	/**
	 * 取得迁移状态值
	 * 
	 * @return 状态值
	 */
	public String getStatusValue() {
		return this.statusValue;
	}

	/**
	 * 设置迁移发生的数据条件
	 * 
	 * @param condition
	 *            条件表达式
	 */
	public void setCondition(String condition) {
		this.condition = condition;
	}

	/**
	 * 取得迁移发生的数据条件
	 * 
	 * @return 条件表达式
	 */
	public String getCondition() {
		return this.condition;
	}

	/**
	 * 设置迁移发生的总数据条件
	 * 
	 * @param headCondition
	 *            条件表达式
	 */
	public void setHeadCondition(String headCondition) {
		this.headCondition = headCondition;
	}

	/**
	 * 取得迁移发生的总数据条件
	 * 
	 * @return 条件表达式
	 */
	public String getHeadCondition() {
		return this.headCondition;
	}
	
	/**
	 * 设置来源表集合
	 * 
	 * @param sourceTableCollection
	 *            表集合
	 */
	public void setSourceTableCollection(MetaDMSourceTableCollection sourceTableCollection) {
		this.sourceTableCollection = sourceTableCollection;
	}

	/**
	 * 取得来源表集合
	 * 
	 * @return 来源表集合
	 */
	public MetaDMSourceTableCollection getSourceTableCollection() {
		return this.sourceTableCollection;
	}

	/**
	 * 设置目标表集合
	 * 
	 * @param targetTableCollection
	 *            表集合
	 */
	public void setTargetTableCollection(MetaDMTargetTableCollection targetTableCollection) {
		this.targetTableCollection = targetTableCollection;
	}

	/**
	 * 取得目标表集合
	 * 
	 * @return 表集合
	 */
	public MetaDMTargetTableCollection getTargetTableCollection() {
		return this.targetTableCollection;
	}

	public String getFromPartInSql() {
		return fromPartInSql;
	}

	public void setFromPartInSql(String fromPartInSql) {
		this.fromPartInSql = fromPartInSql;
	}

	/**
	 * 设置源数据对象标识
	 * 
	 * @param srcDataObjectKey
	 *            源数据对象标识
	 */
	public void setSrcDataObjectKey(String srcDataObjectKey) {
		this.srcDataObjectKey = srcDataObjectKey;
	}

	/**
	 * 取得源数据对象标识
	 * 
	 * @return 源数据对象标识
	 */
	public String getSrcDataObjectKey() {
		return this.srcDataObjectKey;
	}

	/**
	 * 设置目标数据对象标识
	 * 
	 * @param tgtDataObjectKey
	 *            目标数据对象标识
	 */
	public void setTgtDataObjectKey(String tgtDataObjectKey) {
		this.tgtDataObjectKey = tgtDataObjectKey;
	}

	/**
	 * 取得目标数据对象标识
	 * 
	 * @return 目标数据对象标识
	 */
	public String getTgtDataObjectKey() {
		return this.tgtDataObjectKey;
	}

	public String getExtend() {
		return extend;
	}

	public void setExtend(String extend) {
		this.extend = extend;
	}

	@Override
	public AbstractMetaObject clone() {
		MetaDataMigration newMigration = new MetaDataMigration();
		newMigration.setKey(key);
		newMigration.setCaption(caption);
		newMigration.setDescription(description);
		newMigration.setSrcDataObjectKey(srcDataObjectKey);
		newMigration.setFromPartInSql(fromPartInSql);
		newMigration.setTgtDataObjectKey(tgtDataObjectKey);
		newMigration.setStatusFieldKey(statusFieldKey);
		newMigration.setStatusValue(statusValue);
		newMigration.setCondition(condition);
		newMigration.setExtend(extend);
		newMigration.setHeadCondition(headCondition);
		newMigration.setSourceTableCollection(sourceTableCollection == null ? null : (MetaDMSourceTableCollection) sourceTableCollection.clone());
		newMigration.setTargetTableCollection(targetTableCollection == null ? null : (MetaDMTargetTableCollection) targetTableCollection.clone());
		newMigration.setErrorInfoCollection(errorInfoCollection == null ? null : (MetaDataMigrationErrorInfoCollection)errorInfoCollection.clone()); 
		return newMigration;
	}

	@Override
	public AbstractMetaObject newInstance() {
		return new MetaDataMigration();
	}

	public MetaMigrationParas getMigrationParas(IMetaFactory factory) throws Throwable {
		if (migrationParas == null) {
			synchronized (this) {
				if (migrationParas == null)
					migrationParas = new MetaMigrationParas(this, factory);
			}
		}

		return migrationParas;
	}

	public MetaDataMigrationErrorInfoCollection getErrorInfoCollection() {
		return errorInfoCollection;
	}

	public void setErrorInfoCollection(MetaDataMigrationErrorInfoCollection errorInfoCollection) {
		this.errorInfoCollection = errorInfoCollection;
	}

	public IMetaProject getProject() {
		return project;
	}

	public void setProject(IMetaProject project) {
		this.project = project;
	}

	@Override
	public void merge(MetaDataMigration metaDataMigration) {
		if (this.fromPartInSql == null){
			this.fromPartInSql = metaDataMigration.getFromPartInSql();
		}
		if (metaDataMigration.sourceTableCollection != null && metaDataMigration.sourceTableCollection.size() > 0) {
			if (this.sourceTableCollection == null || this.sourceTableCollection.size() == 0) {
				this.sourceTableCollection = metaDataMigration.sourceTableCollection;
			} else {
				for (MetaDMSourceTable metaDMSourceTable : metaDataMigration.sourceTableCollection) {
					String key = metaDMSourceTable.getKey();
					MetaDMSourceTable metaSourceTable = this.sourceTableCollection.get(key);
					// 如果不包括当前表则整体添加
					if (metaSourceTable == null) {
						this.sourceTableCollection.add(metaDMSourceTable);
					} else {
						for (MetaDMSourceField metaDMSourceField : metaDMSourceTable) {
							//如果包含即不插入
							if (metaSourceTable.containsKey(metaDMSourceField.getKey())) {
								throw new RuntimeException("合并SourceField 时错误，原因：数据迁移马甲" + metaDataMigration.key +
										"中定义的与源单重复,重复的Key为：" + metaDMSourceField.getKey());
							}
							metaSourceTable.add(metaDMSourceField);
						}
					}
				}
				this.sourceTableCollection.merge(metaDataMigration.sourceTableCollection);
			}
		}
		if (metaDataMigration.targetTableCollection != null && metaDataMigration.targetTableCollection.size() > 0) {
			if (this.targetTableCollection == null || this.targetTableCollection.size() == 0) {
				this.targetTableCollection = metaDataMigration.targetTableCollection;
			} else {
				for (MetaDMTargetTable metaDMTargetTable : metaDataMigration.targetTableCollection) {
					if (this.targetTableCollection.containsKey(metaDMTargetTable.getKey())) {
						MetaDMTargetTable metaDMTargetTable1 = this.targetTableCollection.get(metaDMTargetTable.getKey());
						for (MetaDMTargetField metaDMTargetField : metaDMTargetTable) {
							if (!metaDMTargetTable1.containsKey(metaDMTargetField.getKey())) {
								metaDMTargetTable1.add(metaDMTargetField);
							}
						}
					} else {
						this.targetTableCollection.add(metaDMTargetTable);
					}
				}
			}
		}
		if (metaDataMigration.errorInfoCollection != null && metaDataMigration.errorInfoCollection.size() > 0) {
			if (this.errorInfoCollection == null || this.errorInfoCollection.size() == 0) {
				this.errorInfoCollection = metaDataMigration.errorInfoCollection;
			} else {
				for (MetaDataMigrationErrorInfo metaDataMigrationErrorInfo : metaDataMigration.errorInfoCollection) {

					if (this.errorInfoCollection.containsKey(metaDataMigrationErrorInfo.getKey())) {
						throw new RuntimeException("合并DataMigrationErrorInfo 时错误，原因：数据迁移马甲" + metaDataMigration.key +
								"中定义的与源单重复,重复的Key为：" + metaDataMigrationErrorInfo.getKey());
					} else {
						this.errorInfoCollection.add(metaDataMigrationErrorInfo);
					}
				}
			}
		}
	}

}
