import ContainerPanel from "../../../common/component/container/panel/ContainerPanel";
import Custom from "../../../common/component/control/Custom";
import TableLayout from "../../../common/dom/element/TableLayout";
import SpanElement from "../../../common/dom/element/SpanElement";
import ActionNames from "../../../common/event/ActionNames";
import IEventListener from "../../../common/listener/IEventListener";
import SelectionChangeListener from "../../../common/listener/SelectionChangeListener";
import SelectionModel from "../../../common/selection/SelectionModel";
import Paras from "../../../common/struct/Paras";
import TableCellLocation from "../../../common/struct/TableCellLocation";
import IXmlEntity from "../../../common/xml/IXmlEntity";
import ITagNode from "../../../common/xml/node/ITagNode";
import TagNode from "../../../common/xml/node/TagNode";
import XmlEntity from "../../../common/xml/XmlEntity";
import { IPropertyContainer } from "../../../design/plugin/property/base/PropertyDefine";
import DeleteCmd from "./cmd/DeleteCmd";
import NewAreaCmd from "./cmd/NewAreaCmd";
import ResizeAreaCmd from "./cmd/ResizeAreaCmd";
import GroupArea from "./view/GroupArea";
import AreaConstants from "./property/AreaConstants";
import ShadowArea from "./view/ShadowArea";
import AreaStateDelegate from "./state/AreaStateDelegate";
import AreaHandleControl from './view/AreaHandleControl';
import MoveCmd from "./cmd/MoveCmd";
import { EExtraLayoutType } from "../../../common/enum/Enums";
import ChangeAreaCaptionCmd from "./cmd/ChangeAreaCaptionCmd";
import PropertyIO from "../../../design/plugin/property/base/PropertyIO";
import MathUtil from "../../../common/util/MathUtil";
import GroupAreaContainer from "./view/GroupAreaContainer";
import IXElement from "../../../common/dom/xelement/IXElement";
import SizeInfo from "../../../common/struct/SizeInfo";

export default class AreaDesignView extends ContainerPanel implements IEventListener {

    public static D_WIDTH: number = 1000;

    public static D_HEIGHT: number = 760;

    private tableElement: TableLayout;

    private xmlEntity: XmlEntity;

    private areaStateDelegate: AreaStateDelegate;

    private gridPanel: YIUI.Panel.GridLayoutPanel;

    private selectionListener?: SelectionChangeListener;

    private shadowArea: ShadowArea;

    private groupAreaContainer: GroupAreaContainer;

    constructor(gridPanel: YIUI.Panel.GridLayoutPanel, propertyContainer: IPropertyContainer<any>) {
        super();
        this.addClass("area");
        this.gridPanel = gridPanel;
        this.xmlEntity = new XmlEntity();
        this.shadowArea = new ShadowArea();
        this.addComponent(this.shadowArea);
        this.areaStateDelegate = new AreaStateDelegate(this);
        this.prepareXmlEntity(this.xmlEntity, gridPanel);
       
        var widths = this.getRealWidths(gridPanel);
        this.tableElement = new TableLayout(widths);
        this.tableElement.addClass("layout");
        // 根据layout的高度数组的长度 生成相同大小的数组 值全部为30
        let heights = new Array(gridPanel.heights.length).fill(30);
        this.tableElement.initRows(heights);
        //this.addComponent(new Custom(this.tableElement));
        this.groupAreaContainer = new GroupAreaContainer(this.tableElement);
        this.initComponents(this.tableElement, this.gridPanel);
        this.addComponent(this.groupAreaContainer);

        this.loadViewByXmlEntity(this.xmlEntity);

        this.selectionListener = new SelectionChangeListener( propertyContainer, new PropertyIO(this.xmlEntity));
        this.xmlEntity.addEventListener(this);
        this.xmlEntity.getSelectionModel().select(this.xmlEntity.getTagRoot());
        this.initEvents();
        this.initKeyEvent(this.getEl());

        this.getEl().setAttribute('tabIndex', '0');
        this.getEl().focus();
        this.setStyle('outline', 'none');

        this.doSelectRoot();
    }

    /**
     * 事件注册
     */
    private initEvents() {
        this.addEventListener('mousedown', (e: MouseEvent) => {
            this.areaStateDelegate.getCurrentState().mousedown(e);
        });

        this.addEventListener('mousemove', (e: MouseEvent) => {
            this.areaStateDelegate.getCurrentState().mousemove(e);
        });

        this.addEventListener('mouseup', (e: MouseEvent) => {
            this.areaStateDelegate.getCurrentState().mouseup(e);
        });
    }

    getXmlEntity(): IXmlEntity {
        return this.xmlEntity;
    }

    getSelectionModel(): SelectionModel {
        return this.xmlEntity.getSelectionModel();
    }
    
    private prepareXmlEntity(xmlEntity: XmlEntity, gridPanel: YIUI.Panel.GridLayoutPanel) {
        var extralayout = gridPanel.getMetaObj().extraLayout;
        var extraLayoutTag = xmlEntity.getRoot().ensureChild(AreaConstants.NODE_ExtraLayout);
        var type = extralayout ? extralayout.type : 'Collapsible';
        extraLayoutTag.setAttributeValue(AreaConstants.ATTR_Type, type);
        if (!extralayout || !extralayout.groupCollection || extralayout.groupCollection.length == 0) return;
        var groupCollectionTag = new TagNode(AreaConstants.NODE_GroupCollection);
        extraLayoutTag.addChild(groupCollectionTag);
        for (let group of extralayout.groupCollection) {
            var groupTag = new TagNode(AreaConstants.NODE_Group);
            groupTag.setAttributeValue(AreaConstants.ATTR_Key, group.key);
            groupTag.setAttributeValue(AreaConstants.ATTR_Caption, group.caption);
            groupTag.setAttributeValue(AreaConstants.ATTR_StartRow, String(group.startRow));
            groupTag.setAttributeValue(AreaConstants.ATTR_EndRow, String(group.endRow));
            groupTag.setAttributeValue(AreaConstants.ATTR_Expanded, group.expanded ? 'true' : 'false');
            groupCollectionTag.addChild(groupTag);

            var handle = group.handle;
            if (handle) {
                var handleTag = new TagNode(AreaConstants.NODE_Handle);
                handleTag.setAttributeValue(AreaConstants.ATTR_Type, handle.type);
                handleTag.setAttributeValue(AreaConstants.ATTR_X, String(handle.x));
                handleTag.setAttributeValue(AreaConstants.ATTR_Y, String(handle.y));
                handleTag.setAttributeValue(AreaConstants.ATTR_XSpan, String(handle.xSpan));
                handleTag.setAttributeValue(AreaConstants.ATTR_YSpan, String(handle.ySpan));

                var format = handle.format;
                if (format) {
                    var tagFormat = handleTag.ensureChild(AreaConstants.NODE_Format);
                    format.foreColor && tagFormat.setAttributeValue(AreaConstants.ATTR_ForeColor, format.foreColor)
                    var font = format.font;
                    if (font) {
                        var tagFont = tagFormat.ensureChild(AreaConstants.NODE_Font);
                        tagFont.setAttributeValue(AreaConstants.ATTR_FONT_Name,font.name ? font.name : '', '');
                        tagFont.setAttributeValue(AreaConstants.ATTR_FONT_Bold, (font.bold == undefined || font.bold) ? 'true' : 'false');
                        tagFont.setAttributeValue(AreaConstants.ATTR_FONT_Italic, font.italic ? 'true' : 'false', 'false');
                        tagFont.setAttributeValue(AreaConstants.ATTR_FONT_Size, font.size ? String(font.size) : '', '');
                        tagFont.setAttributeValue(AreaConstants.ATTR_FONT_Underline, font.underline ? 'true' : 'false', 'false');
                        tagFont.setAttributeValue(AreaConstants.ATTR_FONT_UnderlineColor, font.underlineColor ? font.underlineColor : '', '');
                    }
                }
                groupTag.addChild(handleTag);
            }
        }
    }

    private getRealWidths(gridPanel: YIUI.Panel.GridLayoutPanel): number[] {
        var widths: number[] = [];
        var totalWidth: number = 0;
        for (var width of gridPanel.widths) {
            widths.push(MathUtil.isPercentage(width) ? 200: SizeInfo.getNumFromPX(width)); //如果是百分比就直接给200宽度
            totalWidth += MathUtil.isPercentage(width) ? 200: SizeInfo.getNumFromPX(width);
        }

        var scale = AreaDesignView.D_WIDTH / totalWidth;
        for (var index = 0; index < widths.length; index ++) {
            widths[index] = Math.floor(widths[index] * scale) - 2;
        }
        return widths;
    }

    private initComponents(tableElement: TableLayout, gridPanel: YIUI.Panel.GridLayoutPanel): void {
        for (let item of gridPanel.items) {
            if (item.caption) {
                var span = new SpanElement(item.caption);
                span.addClass('area_span_text');
                span.addClass('area_mirror_text');
                item.type != YIUI.ControlType.LABEL && span.addClass('area_mirror_input');
                tableElement.setCellElement(item.meta.y, item.meta.x, span);
            }
        }
    }

    drawMoveShadow(offsetX: number, offsetY: number): void {
        var selectable = this.getXmlEntity().getSelectionModel().getSingleSeletionItem();
        var tagNode = selectable?.getTagNode();
        if (!tagNode) return;
        switch (tagNode.getTagName()) {
            case AreaConstants.NODE_Group:
                this.drawMoveAreaShadow(tagNode, offsetX, offsetY);
                break;
            case AreaConstants.NODE_Handle:
                this.drawMoveHandleShadow(tagNode, offsetX, offsetY);
                break;
        }
    }

    drawMoveHandleShadow(tagNode: ITagNode, offsetX: number, offsetY: number) {
        var handle = this.groupAreaContainer.findHandleControl(tagNode);
        if (!handle) return;
        var newLeft = this.getLeft(handle.getX()) + offsetX;
        var newTop = this.getTop(handle.getY()) + offsetY;
        this.shadowArea.setLocation(newLeft, newTop, handle.getWidth().getValue(), handle.getHeight().getValue());
    }

    drawMoveAreaShadow(tagNode: ITagNode, offsetX: number, offsetY: number) {
        var area = this.groupAreaContainer.findArea(tagNode);
        if (!area) return false;
        var startRow =  area.getStartRow();
        var newTop = this.getTop(startRow) + offsetY;
        this.shadowArea.setLocation(0, newTop, area.getWidth().getValue(), area.getHeight().getValue());
    }

    drawAreaShadow(startRow: number, endRow:number): void {
        var top = this.tableElement.getTop(startRow);
        var width = this.tableElement.getWidth();
        var height = this.tableElement.getHeightBetween(startRow, endRow);
        var selectedArea = this.getSelectedArea();
        if (selectedArea) {
            var bChanged = false;
            if (this.canResize(startRow, selectedArea)) {
                this.shadowArea.setStartRow(startRow);
                bChanged = true;
            }
            if (this.canResize(endRow, selectedArea)) {
                this.shadowArea.setEndRow(endRow);
                bChanged = true;
            }
            bChanged && this.shadowArea.setLocation(0, top, width, height);
        }
    }

    canResize(row: number, curArea: GroupArea): boolean {
        var findArea = this.groupAreaContainer.findAreaByRow(row);
        if (!findArea) return true;           // 如果行位置未被使用， 则当前行位置可用
        if (curArea == findArea) return true; // 如果行位置被自身使用， 则当前行位置可用
        return false;
    }

    hideShadow(): void {
        this.shadowArea.hide();
    }

    addView(tagNode: ITagNode) {
        switch (tagNode.getTagName()) {
            case AreaConstants.NODE_Group:
                this.addArea(tagNode);
                break;
            case AreaConstants.NODE_Handle:
                this.addHandle(tagNode);
                break;
        }
    }

    addHandle(handleTag: ITagNode): void {
        var areaTag = handleTag.getParent();
        if (!areaTag) return;

        var areaView = this.groupAreaContainer.findArea(areaTag);
        var handleControl = new AreaHandleControl("", handleTag, this.areaStateDelegate);
        if (handleControl) {
            this.addComponent(handleControl);
        }
        areaView?.setHandleControl(handleControl);
        this.updateArea(areaView);
        this.xmlEntity.getSelectionModel().select(handleTag);
        this.remarkAll();
    }

    addArea(areaTag: ITagNode, bLoading: boolean = false) {
        var area = new GroupArea(areaTag, this.areaStateDelegate);
        area.addEventListener("click", (e: MouseEvent) => {
            this.xmlEntity.getSelectionModel().select(areaTag);
        });
        !bLoading && this.checkArea(area.getStartRow(), area.getEndRow());
        this.groupAreaContainer.addGroupArea(area);
        this.updateArea(this.groupAreaContainer.findArea(areaTag));
        this.xmlEntity.getSelectionModel().select(areaTag);
    }

    /**
     * 更新视图，此处参数中的tagNode可能是发生变化的某个子节点，需要调用 findOptableTagNode查找到所在视图进行更新
     */
    updateView(tagNode: ITagNode): void {
        var optableTagNode = this.findOptableTagNode(tagNode);
        if (!optableTagNode) return;

        switch (optableTagNode.getTagName()) {
            case AreaConstants.NODE_Group:
                this.updateArea(this.groupAreaContainer.findArea(optableTagNode));
                break;
            case AreaConstants.NODE_Handle:
                this.updateHandle(this.groupAreaContainer.findHandleControl(optableTagNode));
                break;
        }
        this.xmlEntity.getSelectionModel().select(optableTagNode);
    }

    private findOptableTagNode(tagNode: ITagNode | undefined): ITagNode | null {
        if (!tagNode) return null;
        if (tagNode.getTagName() == AreaConstants.NODE_Group || tagNode.getTagName() == AreaConstants.NODE_Handle) {
            return tagNode;
        }
        return this.findOptableTagNode(tagNode.getParent());
    }

    updateHandle(handleControl: AreaHandleControl | null): void {
        if (!handleControl) return;
        var tagArea = handleControl.getTagNode().getParent();
        var caption = tagArea ? tagArea.getAttributeValue(AreaConstants.ATTR_Caption, '') : '';
        handleControl.updateUI(caption);
        var handleLeft = this.getLeft(handleControl.getX());
        var handleTop = this.getTop(handleControl.getY());
        var width = this.getColumnWidth(handleControl.getX());
        var height = this.getRowHeight(handleControl.getY());
        handleControl.setLocation(handleLeft + 1, handleTop + 1, width - 2, height - 2);
        this.remarkAll();
    }

    updateArea(area: GroupArea | null): void {
        if (!area) return;
        var startRow = area.getStartRow();
        var endRow = area.getEndRow();
        var top = this.tableElement.getTop(startRow);
        var width = this.tableElement.getWidth();
        var height = this.tableElement.getHeightBetween(startRow, endRow);
        area.setLocation(3, top + 3, width - 6, height - 6);
        this.updateHandle(area.getHandleControl());
        this.remarkAll();
        this.display();
    }

    remarkAll(): void {
        this.groupAreaContainer.remarkAll();
    }

    /**
     * 取left偏移量
     */
    getLeft(posX: number): number {
        return this.tableElement.getLeft(posX);
    }

    /**
     * 取top偏移量
     */
    getTop(posY: number): number {
        return this.tableElement.getTop(posY);
    }

    /**
     * 取单元格位置列宽度
     */
    getColumnWidth(posX: number): number {
        return this.tableElement.getColumnWidth(posX);
    }

    /**
     * 取单元格位置行高度
    */
    getRowHeight(y: number): number {
        return this.tableElement.getRowHeight(y);
    }

    removeView(tagNode: ITagNode): void {
        switch (tagNode.getTagName()) {
            case AreaConstants.NODE_Group:
                this.removeArea(tagNode);
                break;
            case AreaConstants.NODE_Handle:
                this.removeHandle(tagNode);
                break;
        }
    }

    removeHandle(tagNode: ITagNode): void {
        var handleControl =  this.groupAreaContainer.findHandleControl(tagNode);
        handleControl && this.groupAreaContainer.removeComponent(handleControl);
        this.remarkAll();
        this.display();
        this.doSelectRoot();
        this.areaStateDelegate.setNormalState();
    }

    removeArea(areaTag: ITagNode): void {
        var area = this.groupAreaContainer.findArea(areaTag);
        if (!area) return;
        this.groupAreaContainer.removeGroupArea(area);
        this.remarkAll();
        this.display();
        this.doSelectRoot();
        this.areaStateDelegate.setNormalState();
    }

    findArea(tagNode: ITagNode): GroupArea | null {
        return this.groupAreaContainer.findArea(tagNode)
    }

    findHandleControl(tagNode: ITagNode): AreaHandleControl | null {
        return this.groupAreaContainer.findHandleControl(tagNode)
    }

    doAddAreaCmd(startRow: number, endRow: number) {
        this.checkAreaHandlePosition(startRow);
        var key = `${AreaConstants.NODE_Group}_${startRow}_${endRow}`;
        var cmd = new NewAreaCmd(this.xmlEntity, key, startRow, endRow);
        this.xmlEntity.getCmdQueue().doCmd(cmd);
    }

    doAddAreaCmdByXY(startX: number, startY: number, endX: number, endY: number) {
        var loc1 = this.hitLocationByXY(startX, startY);
        var loc2 = this.hitLocationByXY(endX, endY);
        if (!loc1 || !loc2) return;
    
        var startRow = Math.min(loc1.getRow(), loc2.getRow());
        var endRow = Math.max(loc1.getRow(), loc2.getRow());
        // Grid Layout 中第一行为系统默认行，不计入行序号范围
        this.doAddAreaCmd(startRow, endRow);
    }

    doDeleteCmd(): void {
        var cmd = new DeleteCmd(this.xmlEntity);
        this.xmlEntity.getCmdQueue().doCmd(cmd);
    }

    doMoveCmd(offsetX: number, offsetY: number): void {
        var cmd = new MoveCmd(this, offsetX, offsetY);
        this.xmlEntity.getCmdQueue().doCmd(cmd);
    }

    doResizeCmd(): void {
        var selectedArea = this.getSelectedArea();
        if (!selectedArea) return;

        var startRow = this.shadowArea.getStartRow();
        if (!selectedArea.isHandleRowChanged()) {
            this.checkAreaHandlePosition(startRow);
        }
        var endRow = this.shadowArea.getEndRow();
        var cmd = new ResizeAreaCmd(this.xmlEntity, selectedArea.getTagNode(), startRow, endRow);
        this.xmlEntity.getCmdQueue().doCmd(cmd);
        this.remarkAll();
    }

    doChangeAreaCaptionCmd(caption: string): void {
        var cmd = new ChangeAreaCaptionCmd(this.xmlEntity, caption);
        this.xmlEntity.getCmdQueue().doCmd(cmd);
    }

    getSelectedArea(): GroupArea | null {
        var selectable = this.getXmlEntity().getSelectionModel().getSingleSeletionItem();
        if (selectable) {
            return this.groupAreaContainer.findArea(selectable.getTagNode());
        }
        return null;
    }

    hitLocation(el: HTMLElement): TableCellLocation | null {
        return this.tableElement.hitCellByEl(el);
    }

    hitLocationByXY(x: number, y: number): TableCellLocation | null {
        return this.tableElement.hitCell(x, y);
    } 

    doSelect(x: number, y: number): void {
        var hitTagNode = this.groupAreaContainer.hitTagNode(x, y);
        this.xmlEntity.getSelectionModel().select(hitTagNode ? hitTagNode : this.xmlEntity.getTagRoot());
    }

    doSelectByTagNode(tagNode: ITagNode): void {
        this.groupAreaContainer.unselectAll();
        switch (tagNode.getTagName()) {
            case AreaConstants.NODE_Group:
                this.groupAreaContainer.findArea(tagNode)?.markSelected(true);
                break;
            case AreaConstants.NODE_Handle:
                this.groupAreaContainer.findHandleControl(tagNode)?.markSelected(true);
                break;
        }
    }

    doSelectRoot(): void {
        this.xmlEntity.getSelectionModel().selectRoot();
    }

    checkArea(startRow: number, endRow: number): void {
        var endCol = this.tableElement.getLastColumnIndex();
        for (var rowIndex = startRow; rowIndex <= endRow; rowIndex ++) {
            for (var colIndex = 0; colIndex <= endCol; colIndex ++) {
                if (this.tableElement.isMarkedCell(rowIndex, colIndex)) {
                    throw new Error("区域之间不能有交集!");
                }
            }
        }
    }

    checkPosition(row: number, col: number): void {
        if (row < 0 || row > this.tableElement.getLastRowIndex()
            || col < 0 || col > this.tableElement.getLastColumnIndex()) {
            throw new Error("超出范围!");
        }
    }

    checkAreaMove(startRow: number, endRow: number): void {
        if (startRow < 0 || endRow > this.tableElement.getLastRowIndex()) {
            throw new Error("超出范围!");
        }
        var curArea = this.getSelectedArea();
        var endCol = this.tableElement.getLastColumnIndex();
        for (var rowIndex = startRow; rowIndex <= endRow; rowIndex ++) {
            if (curArea?.containRow(rowIndex) || curArea?.getHandleControl()?.getY() == rowIndex) {
                continue;
            }
            for (var colIndex = 0; colIndex <= endCol; colIndex ++) {
                if (this.tableElement.isMarkedCell(rowIndex, colIndex)) {
                    throw new Error("区域之间不能有交集!");
                }
            }
        }
    }

    checkAreaHandlePosition(startRow: number): void {
        if (this.getType() == EExtraLayoutType.Collapsible && (startRow == 0 || this.groupAreaContainer.findAreaByRow(startRow - 1))) {
            throw new Error("起始行前需预留一行空白区域，用于放置控制点!");
        } 
    }

    getType(): string {
        return this.getXmlEntity().getTagRoot().getAttributeValue(AreaConstants.ATTR_Type);
    }

    refreshSelection(): void {
        if (!this.xmlEntity) return;
        var selectable = this.xmlEntity.getSelectionModel().getSingleSeletionItem();
        this.xmlEntity.getSelectionModel().reset();
        this.xmlEntity.getSelectionModel().select(selectable);
    }

    notifyEvent(cmd: string, paras: Paras) {
        if (!this.xmlEntity) return;

        switch (cmd) {
            case ActionNames.model_node_update_select:
                if (!this.selectionListener) break;
                if (paras.getElement()) {
                    this.doSelectByTagNode(paras.getElement())
                    this.selectionListener.changeSelection(paras.getElement());
                } else {
                    this.selectionListener.changeSelection(this.xmlEntity.getTagRoot());
                }
                break;
            case ActionNames.model_node_add:
                var tagNode = <ITagNode> paras.getElement();
                this.addView(tagNode);
                break;
            case ActionNames.model_node_update: 
                var tagNode = <ITagNode> paras.getElement();
                this.updateView(tagNode);
                break;
            case ActionNames.model_node_remove:
                var tagNode = <ITagNode> paras.getElement();
                this.removeView(tagNode);
                break;
        }
    }

    toXml(bSimple: boolean): string {
        if (this.groupAreaContainer.isEmpty()) return '';
        if (!this.xmlEntity) return '';
        return this.xmlEntity.getXml(bSimple);
    }

    loadViewByXmlEntity(xmlEntity: IXmlEntity) {
        var extraLayoutTag = xmlEntity.getRoot().getChild(AreaConstants.NODE_ExtraLayout);
        if (!extraLayoutTag) return;
        var groupCollectionTag = extraLayoutTag.getChild(AreaConstants.NODE_GroupCollection);
        if (!groupCollectionTag) return;

        var groups = groupCollectionTag.getChildren();
        for (let group of groups) {
            var groupTag = <TagNode> group;
            this.addArea(groupTag, true);
        }
    }

    getLayoutView(): TableLayout {
        return this.tableElement;
    }

    updateAllView(): void {
        var extraLayoutTag = this.xmlEntity.getRoot().getChild(AreaConstants.NODE_ExtraLayout);
        if (!extraLayoutTag) return;
        var groupCollectionTag = extraLayoutTag.getChild(AreaConstants.NODE_GroupCollection);
        if (!groupCollectionTag) return;
        var groups = groupCollectionTag.getChildren();
        for (let group of groups) {
            var groupTag = <TagNode> group;
            this.updateView(groupTag);
        }
    }

    private initKeyEvent(el: HTMLElement): void {
        el.addEventListener('keyup', (e: any) => {
            e.stopPropagation();
            if (e.key === 'Delete') {
                this.doDeleteCmd();
            } else if (e.ctrlKey && e.key === 'z') {
                this.getXmlEntity().undo();
            } else if (e.ctrlKey && e.key === 'y') {
                this.getXmlEntity().redo();
            } 
        }, true);
    }
}
