import { IPropertyGroup, IPropertyIO } from "../base/PropertyDefine";
import PropertyUIControl from "./PropertyUIControl";
import PropertyContants from "../base/PropertyContants";
import IEventListener from "../../../../common/listener/IEventListener";
import Events from "../../../../common/event/Events";
import Paras from "../../../../common/struct/Paras";
import GroupElement from "../../../../common/dom/element/GroupElement";
import AbstractControl from "../../../../common/component/control/AbstractControl";
import IOperatioExecutor from "../../../../common/operation/IOperatioExecutor";
import TypeUtil from "../../../../common/util/TypeUtil";
import PropertyStateDelegate from "./state/PropertyStateDelegate";

export default class PropertyUIGroup extends GroupElement {

    private propertyGroup: IPropertyGroup;

    private callback: IPropertyIO;

    private eventListener?: IEventListener;

    private optDispatcher?: IOperatioExecutor;

    private stateDelegate: PropertyStateDelegate

    private propertyControls: PropertyUIControl[] = [];

    constructor(propertyGroup: IPropertyGroup, callback: IPropertyIO, stateDelegate: PropertyStateDelegate, optDispatcher?: IOperatioExecutor) {
        super(propertyGroup.caption, propertyGroup.expanded);
        this.addClass('property-group');
        this.propertyGroup = propertyGroup;
        this.callback = callback;
        this.optDispatcher = optDispatcher;
        this.stateDelegate = stateDelegate;
        this.isExpanded() && this.displayGroup();
    }

    setOnEventListener(eventListener: IEventListener): void {
        this.eventListener = eventListener;
    }

    unfocus() {
        var control: PropertyUIControl;
        for(let child of this.getChildren()) {
            if (child != this.title) {
                control = <PropertyUIControl> child;
                control.unfocus();
            }
        }
    }

    getPropertyControl(key: string): PropertyUIControl | undefined {
        for (let propertyControl of this.propertyControls) {
            if (propertyControl.getPropertyItem().key == key) {
                return propertyControl;
            }
        }
        return undefined;
    }

    protected displayGroup() {
        this.propertyControls = [];
        for(let propertyItem of this.propertyGroup.properties) {
            var propertyControl: PropertyUIControl = new PropertyUIControl(propertyItem, this.callback, this.stateDelegate, this.optDispatcher);
            this.propertyControls.push(propertyControl);
            propertyControl.addEventListener("click", (e: MouseEvent) => {
                e.preventDefault();
                var paras = Paras.newInstance().setElement(e.target);
                this.eventListener?.notifyEvent(Events.Focus, paras);
                } ,true);
            
            propertyControl.setOnPropertyValueChanged((control: AbstractControl<any>) => {
                var paras = Paras.newInstance().setElement(control);
                this.eventListener?.notifyEvent(Events.VauleChanged, paras);
            });
        }
        this.updateVisiableAndEditable();
    }

    public async updateVisiableAndEditable() {
        this.clearChildren();
        this.title && this.addChild(this.title);
        for(let propertyControl of this.propertyControls) {
            var propertyItem = propertyControl.getPropertyItem();
            var visiable = this.optDispatcher ? await this.optDispatcher.asyncEval(propertyItem.visible) : TypeUtil.toBoolean(propertyItem.visible);
            if (visiable) {
                propertyControl.resize(this.curWidth, PropertyContants.CONTROL_HEIGHT);
                this.addChild(propertyControl);
            }
        }
        this.resetHeight();
    }

    private curWidth: number = 0;
    
    resize(width:number, height:number) {
        this.curWidth = width;
        for( let child of this.getChildren()) {
            child.resize(width, PropertyContants.CONTROL_HEIGHT);
        }
    }

    resizeByRatio(ratio: number): void {
        for( let child of this.getChildren()) {
            if (child instanceof PropertyUIControl) {
                (<PropertyUIControl> child).resizeByRatio(ratio);
            }
        }
    }

    getInnerHeight(): number {
        var total = PropertyContants.CONTROL_HEIGHT; //分组头高度
        for( let child of this.getChildren()) {
            if (child instanceof PropertyUIControl) {
                total += PropertyContants.CONTROL_HEIGHT;
            }
        }
        return total;
    }

    resetHeight(): void {
        this.stateDelegate.getView().resetHeight();
    }
}