import IPropertyItem, { IPropertyIO } from "../base/PropertyDefine";
import AbstractDomElement from "../../../../common/dom/AbstractDomElement";
import AbstractControl from "../../../../common/component/control/AbstractControl";
import ColorEditor from "../../../../common/component/control/ColorEditor";
import Span from "../../../../common/component/control/Span";
import ControlFactory from "../../../../common/component/factory/ControlFactory";
import PropertyContants from "../base/PropertyContants";
import SizeInfo from "../../../../common/struct/SizeInfo";
import GlobalVariable from "../../../GlobalVariable";
import TypeUtil from "../../../../common/util/TypeUtil";
import IOperatioExecutor from "../../../../common/operation/IOperatioExecutor";
import YigoControl from "../../../../yiui/design/control/YigoControl";
import PropertyUIPanel from "./PropertyUIPanel";
import PropertyStateDelegate from "./state/PropertyStateDelegate";
import { EDesignMode } from "../../../../common/enum/Enums";
import {CommonConstant} from "../../../datamap/base/CommonConstant";
import GlobalConstants from "../../../../common/struct/GlobalConstants";

export default class PropertyUIControl extends AbstractDomElement {

    private propertyItem: IPropertyItem;

    private callback: IPropertyIO;

    private labelControl?: Span;

    private spanResizer?: Span;

    private editorControl?: AbstractControl<any>;

    private textControl?: Span;

    private focused: boolean = false;

    private fnOnPropertyValueChange: any;

    private optDispatcher?: IOperatioExecutor;

    private stateDelegate: PropertyStateDelegate

    private curTotalWidth: number = 0;

    private curHeight: number = 0;

    constructor(propertyItem: IPropertyItem, callback: IPropertyIO, stateDelegate: PropertyStateDelegate, optDispatcher?: IOperatioExecutor) {
        super();
        this.addClass('property-control-line');
        this.setStyle("line-height", PropertyContants.CONTROL_HEIGHT + "px");
        this.propertyItem = propertyItem;
        this.callback = callback;
        this.stateDelegate = stateDelegate;
        this.optDispatcher = optDispatcher;
        this.init(propertyItem);
    }

    getPropertyItem(): IPropertyItem {
        return this.propertyItem;
    }

    protected createEl(): HTMLElement {
        return document.createElement("li");
    }

    public updateUI(): void {
        this.init(this.propertyItem);
    }

    private init(propertyItem: IPropertyItem) {
        this.labelControl = new Span(propertyItem.caption);
        this.labelControl.setStyle("display", "inline-block");
        this.labelControl.addClass("property-control-base");
        this.labelControl.addClass("property-label");
        this.addChild(this.labelControl.getDomElement());

        this.spanResizer = new Span("");
        this.spanResizer.setStyle("position", "absolute");
        this.spanResizer.setStyle("display", "inline-block");
        this.spanResizer.addClass("property-control-base");
        this.spanResizer.addClass("property-label");
        this.spanResizer.addClass("property-resize-ltr");
        this.addChild(this.spanResizer.getDomElement());

        this.spanResizer.getDomElement().hover(() => {
            this.stateDelegate.setResizeState();
        },() => {
            if (!this.stateDelegate.getCurrentState().isLeftMouseButtonDowning()) {
                this.stateDelegate.setNormalState();
            }
        });

        this.textControl = new Span("");
        var editorControl = ControlFactory.createControl(this.propertyItem.control, {key: propertyItem.key, component: propertyItem.component}, this.optDispatcher);
        this.initControl(editorControl).then(() => {
            this.setEditControlValue(editorControl, propertyItem).then((value) => {
                var text = editorControl.getShowText();
                if (this.textControl) {
                    this.textControl.setText((!text && value) ? value : text);//取显示值，例如字典、下拉框
                    this.textControl.setStyle("display", "inline-block");
                    this.textControl.addClass("property-control-base");
                    this.textControl.addClass("property-label");
                    if (propertyItem.control == 'ColorEditor') {
                        this.textControl.setStyle('background-color', value);
                    }
                }
            });
        });

        this.addChild(this.textControl.getDomElement());

        var bEditable = this.isEditableItem(this.propertyItem);
        if ((this.optDispatcher &&  !this.optDispatcher.isEditable()) || !bEditable) {
            this.labelControl.addClass("property-disable");
            this.textControl.addClass("property-disable");
        }
    }

    // 目前没有遇到要支持公式的情况，引入公式，就需要引入异步，代码结构冲击较大，暂不做公式处理
    private isEditableItem(propertyItem: IPropertyItem): boolean {
        var bContextEditable = true;
        if (this.optDispatcher) {
            bContextEditable = this.optDispatcher.isEditable();
        }
        return bContextEditable && TypeUtil.toBoolean(propertyItem.editable);
    }

    public setFocused(bFocused: boolean): void {
        if (bFocused == this.focused) return;
        bFocused ? this.focus() : this.unfocus();
    }

    public isFocused(): boolean {
        return this.focused;
    }

    getValue(): any {
        return this.editorControl ? this.editorControl.getValue() : this.callback.getValue(this.propertyItem);
    }

    public focus() {
        if (this.focused) return;
        this.focused = true;
        this.labelControl && this.labelControl.removeClass('property-unfocused');
        this.labelControl && this.labelControl.addClass('property-focused');

        var bEditable = this.isEditableItem(this.propertyItem);
        if (!bEditable) return;

        var height = this.textControl ? this.textControl.getDomElement().getHtmlClientHeight() : 0;
        var width = this.textControl ? this.textControl.getDomElement().getHtmlClientWidth() - 5 : 0;
        this.textControl && this.removeChild(this.textControl.getDomElement());
        this.editorControl && this.removeChild(this.editorControl.getDomElement());

        // 渲染yigo组件
        if(this.propertyItem.component) {
            throw new Error("异常方法调用，相关代码需修复")
            //(<YIUI.Component> this.propertyItem.component).doRender($(this.getEl()));
            //(<YIUI.Component> this.propertyItem.component).getEl().removeClass("ui-txted");
            this.editorControl = new YigoControl(<YIUI.Component> this.propertyItem.component);
        } else {
            this.editorControl = ControlFactory.createControl(this.propertyItem.control, {key: this.propertyItem.key, component: this.propertyItem.component}, this.optDispatcher);
        }

        this.editorControl.setOnValueChanged((control: AbstractControl<any>) => {
            this.fnOnPropertyValueChange(this);
        });
        
        this.editorControl.addClass("property-control-base");
        this.editorControl.addClass("property-input");
        this.editorControl.setWidth(SizeInfo.valueOfPX(width)); 
        this.editorControl.setHeight(SizeInfo.valueOfPX(height - 1));

        this.editorControl.setOnButtonClick(() => {
            var clickEvent = this.propertyItem.onbuttonclick;
            if (clickEvent) {
                clickEvent = clickEvent.replace(GlobalVariable.V_SELF_KEY, this.propertyItem.key);
                clickEvent = clickEvent.replace(GlobalVariable.V_SELF_VALUE, this.editorControl ? this.editorControl.getValue(): "");
            }
            clickEvent && this.optDispatcher && this.optDispatcher.eval(clickEvent);
        });

        this.initControl(this.editorControl).then(() => {
            if (this.editorControl) {
                this.setEditControlValue(this.editorControl, this.propertyItem).then((value) => {
                    this.editorControl && this.editorControl.caretLast();
                }) 
            }
        });
        this.addChild(this.editorControl.getDomElement());
    }

    private isUserMode(): boolean {
        if (!this.optDispatcher) return false;
        return this.optDispatcher.getMode() == EDesignMode.User;
    }

    private async setEditControlValue(editorControl: AbstractControl<any>, propertyItem: IPropertyItem): Promise<any> {
        var value = this.callback.getValue(propertyItem);
        editorControl.setValue(value);
        if (TypeUtil.isStr(value) && this.optDispatcher && this.isUserMode()) {
            var formulaID = await this.optDispatcher.queryFormulaDictID(value);
            if (formulaID > 0) {
                var formulaName = await this.optDispatcher.queryFormulaDictName(formulaID);
                if (!this.optDispatcher.isDevFormulaID(formulaID) && propertyItem.key != GlobalConstants.S_FormKey) {
                    formulaName && editorControl.setShowText(formulaName);
                }
            }
        } else {
            // 部分属性根据公式计算结果，显示已设置或未设置
            var defaultvalue = propertyItem.defaultvalue;
            if (defaultvalue && GlobalVariable.isFormula(defaultvalue) && this.optDispatcher) {
                var formula = GlobalVariable.getFormula(defaultvalue);
                defaultvalue = await this.optDispatcher.eval(formula);
            } else if(GlobalVariable.hasVarialble(defaultvalue)) {
                defaultvalue && (defaultvalue = GlobalVariable.getVarialble(defaultvalue));
            }else{
                defaultvalue = value;
            }
            defaultvalue && editorControl.setShowText(defaultvalue);
        }
        editorControl.caretLast();
        return value;
    }

    setPropertyValue(value: any, text?: string) {
        this.callback.setValue(this.propertyItem, value);
        this.editorControl?.setValue(value);
        text && this.editorControl?.setShowText(text);
        if (this.textControl) {
            text ? this.textControl.setShowText(text) : this.textControl.setShowText(value);
        }
    }

    setOnPropertyValueChanged(fn: Function): void {
        this.fnOnPropertyValueChange = fn;
    }

    private async initControl(editor: AbstractControl<any>): Promise<void> {
        if (!editor) return;
        var oninitialize = this.propertyItem.oninitialize;
        if (oninitialize && GlobalVariable.isFormula(oninitialize) && this.optDispatcher) {
            var formula = GlobalVariable.getFormula(oninitialize);
            oninitialize = await this.optDispatcher.eval(formula);
        } else {
            oninitialize && (oninitialize = GlobalVariable.getVarialble(oninitialize));
        }
        oninitialize && editor.initControl(oninitialize);
    }

    public unfocus() {
        if (!this.focused) return;
        this.focused = false;
        this.labelControl && this.labelControl.removeClass('property-focused');
        this.labelControl && this.labelControl.addClass('property-unfocused');

        var bEditable = this.isEditableItem(this.propertyItem);
        if (!bEditable || !this.editorControl)  return;

        this.textControl && this.removeChild(this.textControl.getDomElement());
        var text = this.editorControl.getShowText();
        this.removeChild(this.editorControl.getDomElement());
        this.textControl?.setText(text);
        if (this.editorControl instanceof ColorEditor) {
            this.textControl?.setStyle('background-color', this.editorControl.getValue());
        }
        var value = this.editorControl.getValue();
        this.textControl && this.addChild(this.textControl.getDomElement());
        this.callback.setValue(this.propertyItem, value);
    }

    private updateUnfocusControl() {

    }

    public resize(width:number, height:number) {
        if (width == 0) return;

        this.curTotalWidth = width;
        this.curHeight = height;
        var leftWidth: number = width * PropertyUIPanel.PROP_RATIO - 6;
        var rightWidth: number = width * (1 - PropertyUIPanel.PROP_RATIO) - 6;
        var curHeight: number = height - 1;
        var leftWidthInfo = SizeInfo.valueOfPX(leftWidth);
        var rightWidthInfo = SizeInfo.valueOfPX(rightWidth);
        var heightInfo = SizeInfo.valueOfPX(curHeight);

        this.labelControl && this.labelControl.setWidth(leftWidthInfo); 
        this.labelControl && this.labelControl.setHeight(heightInfo)

        this.spanResizer && this.spanResizer.setHeight(heightInfo);
        //this.spanResizer && this.spanResizer.setStyle("margin-left", widthInfo.toString());
        this.textControl && this.textControl.setWidth(rightWidthInfo);
        this.textControl && this.textControl.setHeight(heightInfo);
        this.editorControl && this.editorControl.setWidth(rightWidthInfo);
        this.editorControl && this.editorControl.setHeight(heightInfo);
        this.editorControl && this.editorControl.fireResized(rightWidthInfo, heightInfo);
        this.editorControl && this.editorControl.lostFocus();
    }

    resizeByRatio(ratio: number): void {
        this.resize(this.curTotalWidth, this.curHeight);
    }
}