import AbstractDomElement from "../../../common/dom/AbstractDomElement";
import {CommonConstant} from "../base/CommonConstant";
import SizeInfo from "../../../common/struct/SizeInfo";
import Span from "../../../common/component/control/Span";
import TextEditor from "../../../common/component/control/TextEditor";
import InputElement from "../../../common/dom/element/InputElement";

export class DataMapTableLayout extends AbstractDomElement{
    private tableBody!: TableBody;
    private width!: number;
    private height!: number;
    private container! : Container;

    constructor(width : number, height : number){
        super();
        this.init(width,height);
    }

    init(width : number, height : number){
        this.clearChildren();
        this.width = width;
        this.height = height;
        this.tableBody = new TableBody(width , height);
        this.addChild(this.tableBody);
    }

    addField(name: string): Field {
        let row = new Row();
        row.setAttr(CommonConstant.TYPE_SOURCE, name);
        let field = new Field(null);
        row.addField(field);
        this.tableBody.addChild(row);
        return field;
    }

    addFeedbackField(name: string , index: number): Field {
        let row = new Row();
        row.setAttr(CommonConstant.TYPE_SOURCE, name);
        row.setAttr(CommonConstant.TYPE_FeedbackIndex, index.toString());
        let field = new Field(null);
        row.addField(field);
        this.tableBody.addChild(row);
        return field;
    }

    addSpan(): Container{
        let row = new Row()
        let container = new Container(null);
        this.container = container;
        row.addSpan(container)
        this.tableBody.addChild(row);
        return container;
    }

    protected createEl(tag: string): HTMLElement {
        return document.createElement("table");
    }

    getWidth() {
        return this.width;
    }

    getHeight() {
        return this.height;
    }

    hitRow(x: number, y: number): number | null{
        this.resetRowsTop();
        let hitRowIndex = -1;
        for(let rowIndex = 0; rowIndex < this.tableBody.getRowCount(); rowIndex++){
            let row = this.tableBody.getRow(rowIndex);
            if(y >= row.getTop() && y < row.getTop() + row.getHeight()){
                hitRowIndex = rowIndex;
                break;
            }
        }
        if(hitRowIndex == -1) return null;
        return hitRowIndex;
    }

    getRow(cellLocation: number): Field {
        return <Field>this.tableBody.getRow(cellLocation).getChildAt(0);
    }

    resetRowsTop(): void {
        this.tableBody.resetRowsTop();
    }
}

class TableBody extends AbstractDomElement{
    private width : number;
    private height : number;
    constructor(width : number, height : number){
        super();
        this.width = width;
        this.height = height;
    }

    protected createEl(tag: string): HTMLElement {
        return document.createElement("tbody");
    }

    /**
     * 获取行
     * @param index
     * @returns
     */
    getRow(index: number): Row {
        return <Row>this.getChildAt(index); // 第一行不算
    }

    resetRowsTop() {
        let top = 0;
        for (let rowIndex = 0; rowIndex < this.getRowCount(); rowIndex++) {
            let row = this.getRow(rowIndex);
            row.setTop(top);
            top += row.getHeight();
        }
    }

    /**
     * 返回行数
     * @returns
     */
    getRowCount(): number {
        return this.getChildCount(); // 第一行不算
    }
}

class Field extends AbstractDomElement {
    constructor(el: HTMLElement | null) {
        super(el);
        this.addClass("source_fieldArea")
    }

    protected createEl(tag: string): HTMLElement {
        return document.createElement("td");
    }

    getContent(): AbstractDomElement | null {
        if (this.getChildCount() == 0) return null;
        return this.getChildAt(0);
    }

}

class Row extends AbstractDomElement {
    private top: number = 0;

    constructor() {
        super();
        this.setStyleHeight(SizeInfo.valueOfPX(CommonConstant.ROW_HEIGHT));

    }

    addField(cell: Field) {
        this.addChild(cell);
    }

    addSpan(item: Container) {
        this.addChild(item);
    }

    protected createEl(tag: string): HTMLElement {
        return document.createElement("tr");
    }

    getTop() {
        return this.top;
    }

    setTop(top: number) {
        this.top = top;
    }

    getHeight(): number {
        return this.getHtmlClientHeight(); // 此处不用配置中的行height，是因为erp中有多个控件放在一个grid cell中的用法，会将row高度撑大
    }

}

export class Container extends AbstractDomElement{
    private selectIcon: Span;
    private textEditor : TextEditor;
    private input : Span;
    private addButton : InputElement;

    constructor(el: HTMLElement | null){
        super(el);
        this.addClass("source_fieldArea");
        this.setStyle("border-width", "thin");
        let span = new Span("单击选择表单字段");
        span.addClass("textArea");
        this.input = span;

        this.textEditor = new TextEditor();
        this.textEditor.setStyle("display","none");

        let selectSpan = new Span("");
        selectSpan.addClass("selectImg");
        selectSpan.setStyle("background","url("+ CommonConstant.PREFIX + CommonConstant.SELECT_FIELD_URL+ ") center no-repeat");
        this.selectIcon = selectSpan;

        let inputElement = new InputElement();
        inputElement.addClass("addButton_Head");
        inputElement.setAttr("type","button");
        inputElement.setAttr("name","button");
        inputElement.setAttr("value","...");
        this.addButton = inputElement;

        this.appendChild(span);
        this.appendChild(selectSpan);
        this.appendChild(inputElement);
        this.appendChild(this.textEditor);
    }

    appendChild(any: any) {
        this.getEl().appendChild(any.getEl());
    }
    setContent(element: AbstractDomElement): void {
        this.addChild(element);
    }

    getContent(): AbstractDomElement | null {
        if (this.getChildCount() == 0) return null;
        return this.getChildAt(0);
    }

    getIcon(): Span {
        return this.selectIcon;
    }

    getAddButton(): InputElement {
        return this.addButton;
    }

    getInput(): Span {
        return this.input;
    }
    getTextEditor(): TextEditor {
        return this.textEditor;
    }

    protected createEl(tag: string): HTMLElement {
        return document.createElement("td");
    }

}

