import AbstractWorkSpace from "../base/AbstractWorkSpace";
import Toolbar from "../../common/component/toolbar/Toolbar";
import {EDesignMode} from "../../common/enum/Enums";
import ComplexDesignView from "../base/ComplexDesignView";
import OperationDispatcher from "../../common/operation/OperationDispatcher";
import AbstractDesignView from "../base/AbstractDesignView";
import {DataMapDesignView} from "./view/DataMapDesignView";
import {DataMapGraphicView} from "./view/DataMapGraphicView";
import XmlEntity from "../../common/xml/XmlEntity";
import {CommonFunction} from "./base/CommonFunction";
import {IPropertyIO} from "../plugin/property/base/PropertyDefine";
import {DataMapProperty} from "./property/DataMapProperty";
import Button from "../../common/component/control/Button";
import AbstractDomElement from "../../common/dom/AbstractDomElement";
import {CommonConstant} from "./base/CommonConstant";
import ITagNode from "../../common/xml/node/ITagNode";
import SizeInfo from "../../common/struct/SizeInfo";
import Events from "../../common/event/Events";
import Menu from "../../common/component/menu/Menu";
import {DataMapNormalState} from "./state/DataMapNormalState";
import KeyUpEvent = JQuery.KeyUpEvent;
import {DataMapSourceTableView} from "./view/DataMapSourceTableView";
import {DataMapTargetTableView} from "./view/DataMapTargetTableView";
import GlobalVariable from "../GlobalVariable";


export default class DataMapWorkSpace extends AbstractWorkSpace<DataMapGraphicView, Toolbar> {
    private mune: Menu | undefined;
    private path?: string;

    constructor(mode: EDesignMode, path: string) {
        super(mode, new Toolbar());
        this.path = path;
        this.addClass("dataMap-workspace");
    }

    createDesignView(viewContainer: ComplexDesignView<any, any>, operationDispatcher: OperationDispatcher): AbstractDesignView<any> {
        return new DataMapDesignView(viewContainer, operationDispatcher);
    }

    createGraphicView(mode: EDesignMode) {
        let dataMapGraphicView = new DataMapGraphicView();
        let stcView = dataMapGraphicView.getStcView();
        let ttcView = dataMapGraphicView.getTtcView();
        this.initSourceEvent(stcView.getDomElement());
        this.initTargetEvent(ttcView.getDomElement());
        let domElement = dataMapGraphicView.getGridBorder().getDomElement();
        let paper = dataMapGraphicView.getPaper();
        domElement.addEventListener('contextmenu', function (e: any) {
            e.preventDefault();
        })
        domElement.addEventListener('mousemove', (e: MouseEvent) => this.getGraphicView().getOptStateDelegate().getCurrentState().mousemove(e));
        domElement.addEventListener('mouseup', (e: MouseEvent) => this.getGraphicView().getOptStateDelegate().getCurrentState().mouseup(e));
        paper.mousedown(event => {
            let currentState = this.getGraphicView().getOptStateDelegate().getCurrentState();
            if (currentState instanceof DataMapNormalState) {
                currentState.setOperationDispatcher(this.operationDispatcher);
            }
            currentState.mousedown(event);
        });
        return dataMapGraphicView;
    }

    initSourceEvent(stcViewDom: AbstractDomElement) {
        stcViewDom.addEventListener('mousedown', (e: MouseEvent) => {
            let target = <HTMLElement>e.target;
            if (target.nodeName == CommonConstant.Node_Type_Span) {
                if (target.className.indexOf("selectImg") != -1) {
                    this.getSourceDataBySelectImg(target);
                } else if (target.className.indexOf("textArea") != -1) {
                    this.addInputSourceField(target);
                }
            } else if (target.nodeName == CommonConstant.Node_Type_Input && target.className.indexOf("addButton_Head") != -1) {
                let attribute = target.getAttribute(CommonConstant.NODE_SrcFormKey);
                if (!attribute) {
                    attribute = target.getAttribute(CommonConstant.NODE_Table_Key);
                }
                if(!attribute && target.getAttribute(CommonConstant.ATTR_DataSource)){
                    attribute = target.getAttribute(CommonConstant.ATTR_DataSource);
                }
                let sourceViews = this.getGraphicView().getSourceViews();
                let sourceView = sourceViews.get(attribute!);
                if (sourceView) {
                    this.getGraphicView().setSelectSourceView(sourceView);
                    var isTitle = sourceView.isTableTitle();
                    let tableKey = "";
                    let formKey;
                    let dataSource = "";
                    if (isTitle) {
                        let value = target.getAttribute(CommonConstant.NODE_SrcFormKey);
                        formKey = value;
                    } else {
                        let value = target.getAttribute(CommonConstant.NODE_Table_Key);
                        if(value){
                            formKey = value!.split("|")[0];
                            tableKey = value!.split("|")[1];
                        }else{
                            value = target.getAttribute(CommonConstant.ATTR_DataSource);
                            formKey = value!.split("|")[0];
                            dataSource = value!.split("|")[1].replace(/'/g, '"');
                        }

                    }
                    let formula = `Macro_ShowFieldList('${isTitle}', '${formKey}' ,'${tableKey}', '${dataSource}',  'source')`;
                    this.operationDispatcher.eval(formula);
                }
            }
        });

    }

    initTargetEvent(ttcViewDom: AbstractDomElement) {
        ttcViewDom.addEventListener('mousedown', (e: MouseEvent) => {
            let target = <HTMLElement>e.target;
            if (target.nodeName == CommonConstant.Node_Type_Span) {
                if (target.className.indexOf("selectImg") != -1) {
                    this.getTgtDataBySelectImg(target);
                } else if (target.className.indexOf("textArea") != -1) {
                    this.addInputTargetField(target);
                }
            } else if (target.nodeName == CommonConstant.Node_Type_Input && target.className.indexOf("addButton_Head") != -1) {
                let attribute = target.getAttribute(CommonConstant.NODE_TgtFormKey);
                if (!attribute) {
                    attribute = target.getAttribute(CommonConstant.NODE_Table_Key);
                }
                let targetViews = this.getGraphicView().getTargetViews();
                let targetView = targetViews.get(attribute!);
                if (targetView) {
                    this.getGraphicView().setSelectTargetView(targetView);
                    var isTitle = targetView.isTableTitle();
                    let tableKey = "";
                    let formKey;
                    let dataSource = "";
                    if (isTitle) {
                        let value = target.getAttribute(CommonConstant.NODE_TgtFormKey);
                        formKey = value;
                    } else {
                        let value = target.getAttribute(CommonConstant.NODE_Table_Key);
                        formKey = value!.split("|")[0];
                        tableKey = value!.split("|")[1];
                    }
                    let formula = `Macro_ShowFieldList('${isTitle}', '${formKey}' ,'${tableKey}', '${dataSource}', 'target')`;
                    this.operationDispatcher.eval(formula);
                }
            }
        });
    }


    getSourceDataBySelectImg(target: HTMLElement) {
        let value, formula;
        if (target.getAttribute(CommonConstant.NODE_SrcFormKey)) {
            value = target.getAttribute(CommonConstant.NODE_SrcFormKey);
            formula = `Macro_GetFormAllField('${value}','','')`;
        } else {
            value = target.getAttribute(CommonConstant.NODE_Table_Key);
            formula = `Macro_GetTableFields('${value!.split("|")[0]}','${value!.split("|")[1]}', '','')`;
        }
        let sourceViews = this.getGraphicView().getSourceViews();
        let sourceView = sourceViews.get(value!);
        if (!sourceView) return;
        this.getGraphicView().setSelectSourceView(sourceView);
        let menu = sourceView!.getMenu();
        menu.addEventListener("focus", (e: any) => {
            this.getGraphicView().getOptStateDelegate().setInputFieldState()
        });
        var self = this;
        menu.addEventListener("blur", (e: any) => {
            self.getGraphicView().getOptStateDelegate().setNormalState()
        });
        this.parseFieldDataToSourceMenu(menu, formula, this, sourceView,target);
    }

    getTgtDataBySelectImg(target: HTMLElement) {
        let value, formula;
        if (target.getAttribute(CommonConstant.NODE_TgtFormKey)) {
            value = target.getAttribute(CommonConstant.NODE_TgtFormKey);
            formula = `Macro_GetFormAllField('${value}','target','')`;
        } else {
            value = target.getAttribute(CommonConstant.NODE_Table_Key);
            formula = `Macro_GetTableFields('${value!.split("|")[0]}','${value!.split("|")[1]}','target','')`;
        }
        let tgtViews = this.getGraphicView().getTargetViews();
        let tgtView = tgtViews.get(value!);
        if (!tgtView) return;
        this.getGraphicView().setSelectTargetView(tgtView);

        let menu = tgtView!.getMenu();
        menu.addEventListener("focus", (e: any) => {
            this.getGraphicView().getOptStateDelegate().setInputFieldState()
        });
        var self = this;
        menu.addEventListener("blur", (e: any) => {
            self.getGraphicView().getOptStateDelegate().setNormalState()
        });
        this.parseFieldDataToTgtMenu(menu, formula, this, tgtView, target);
    }


    addInputSourceField(target: HTMLElement) {
        let graphicView = this.getGraphicView();
        // title 和 table的输入框获取的attribute不一样
        let attribute = target.getAttribute(CommonConstant.NODE_SrcFormKey);
        if (!attribute) {
            attribute = target.getAttribute(CommonConstant.NODE_Table_Key);
        }
        let sourceViews = graphicView.getSourceViews();
        var tableView = sourceViews.get(attribute!);
        if (!tableView) return;
        // 原有dom的隐藏
        let inputPrompt = tableView.getInputPromptSpan();
        inputPrompt!.setStyle("display", "none");
        tableView.getAddIcon().setStyle("display", "none");
        tableView.getInputButton()?.setStyle("display", "none");
        // 输入框显示
        let addItemInput = tableView.getInputTextEditor();
        if (addItemInput) {
            var height = SizeInfo.valueOfPX(Number($(target).parent().height()));
            var width = SizeInfo.valueOfPX(Number($(target).parent().width()));
            addItemInput.setStyle("display", "block");
            addItemInput.setStyle("width", width);
            addItemInput.setStyle("height", "100%")
            addItemInput.setStyle("min-height", height)
            addItemInput.setStyle("disabled", "true");
            setTimeout(function () {
                addItemInput!.getEl().focus();
            }, 10);
        }
        var self = this;


        // 进行additemInput的事件初始化
        if (!tableView.getInitTextEditor()){
            let addItemInputFocusFun = function () {
                self.getGraphicView().getOptStateDelegate().setInputFieldState();
                self.getGraphicView().setSelectSourceView(tableView!);
                var isTitle = tableView!.isTableTitle();
                let formula;
                if (isTitle) {
                    let value = target.getAttribute(CommonConstant.NODE_SrcFormKey);
                    formula = `Macro_GetFormAllField('${value}','','')`;
                } else {
                    let value = target.getAttribute(CommonConstant.NODE_Table_Key);
                    formula = `Macro_GetTableFields('${value!.split("|")[0]}','${value!.split("|")[1]}','','')`;
                }
                let menu = tableView!.getMenu();
                self.parseFieldDataToSourceMenu(menu, formula, self, tableView!, target);
            }

            addItemInput!.addEventListener(Events.Focus, (e: any) => addItemInputFocusFun());

            var addItemInputBlurFun = function (){
                self.getGraphicView().getOptStateDelegate().setNormalState();
                setTimeout(function () {
                    tableView!.getMenu().hide();
                }, 10)
                tableView!.getAddIcon().setStyle("display", "block");
                addItemInput!.setStyle("display", "none");
                tableView!.getInputButton()?.setStyle("display", "block");
                inputPrompt!.setStyle("display", "block");
            }

            addItemInput!.addEventListener(Events.Blur, (e: any) => addItemInputBlurFun());

            addItemInput!.addEventListener('keyup', (e: KeyUpEvent) => {
                // ctrl v 可以搜索
                if ((!e.ctrlKey && e.code != "ControlLeft" && e.code != "ControlRight") || (e.ctrlKey && e.code == "KeyV")) {
                    if (e.code != "ArrowDown" && e.code != "ArrowUp" && e.code != "ArrowLeft" && e.code != "ArrowRight") {
                        var isTitle = tableView!.isTableTitle();
                        let formula;
                        if (isTitle) {
                            let value = target.getAttribute(CommonConstant.NODE_SrcFormKey);
                            formula = `Macro_GetFormAllField('${value}','','${e.target.value}')`;
                        } else {
                            let value = target.getAttribute(CommonConstant.NODE_Table_Key);
                            formula = `Macro_GetTableFields('${value!.split("|")[0]}','${value!.split("|")[1]}','','${e.target.value}')`;
                        }
                        let menu = tableView!.getMenu();
                        this.parseFieldDataToSourceMenu(menu, formula, self, tableView!, target);
                    }

                }
            });

            tableView.setInitTextEditor(true);

        }
    }

    addInputTargetField(target: HTMLElement) {
        let graphicView = this.getGraphicView();
        // title 和 table的输入框获取的attribute不一样
        let attribute = target.getAttribute(CommonConstant.NODE_TgtFormKey);
        if (!attribute) {
            attribute = target.getAttribute(CommonConstant.NODE_Table_Key);
        }
        let targetViews = graphicView.getTargetViews();
        var targetView = targetViews.get(attribute!);
        if (!targetView) return;
        let inputPrompt = targetView.getInputPromptSpan();
        inputPrompt!.setStyle("display", "none");
        targetView.getAddIcon().setStyle("display", "none");
        targetView.getInputButton()?.setStyle("display", "none");
        let addItemInput = targetView.getInputTextEditor();
        if (addItemInput) {
            var height = SizeInfo.valueOfPX(Number($(target).parent().height()));
            var width = SizeInfo.valueOfPX(Number($(target).parent().width()));
            addItemInput.setStyle("display", "block");
            addItemInput.setStyle("width", width);
            addItemInput.setStyle("height", "100%")
            addItemInput.setStyle("min-height", height)
            addItemInput.setStyle("disabled", "true");
            setTimeout(function () {
                addItemInput!.getEl().focus();
            }, 10);
        }
        var self = this;

        // 进行additemInput的事件初始化
        if (!targetView.getInitTextEditor()) {
            let addItemInputFocusFun = function () {
                self.getGraphicView().getOptStateDelegate().setInputFieldState();
                self.getGraphicView().setSelectTargetView(targetView!);
                var isTitle = targetView!.isTableTitle();
                let formula;
                if (isTitle) {
                    let value = target.getAttribute(CommonConstant.NODE_TgtFormKey);
                    formula = `Macro_GetFormAllField('${value}','target','')`;
                } else {
                    let value = target.getAttribute(CommonConstant.NODE_Table_Key);
                    formula = `Macro_GetTableFields('${value!.split("|")[0]}','${value!.split("|")[1]}','target','')`;
                }
                let menu = targetView!.getMenu();
                self.parseFieldDataToTgtMenu(menu, formula, self, targetView!, target);
            }

            addItemInput!.addEventListener(Events.Focus, (e: any) => addItemInputFocusFun());

            var addItemInputBlurFun = function () {
                self.getGraphicView().getOptStateDelegate().setNormalState();
                setTimeout(function () {
                    targetView!.getMenu().hide();
                }, 10)
                targetView!.getAddIcon().setStyle("display", "block");
                addItemInput!.setStyle("display", "none");
                targetView!.getInputButton()?.setStyle("display", "block");
                inputPrompt!.setStyle("display", "block");
            }

            addItemInput!.addEventListener(Events.Blur, (e: any) => addItemInputBlurFun());

            addItemInput!.addEventListener('keyup', (e: KeyUpEvent) => {
                // ctrl v 可以搜索
                if ((!e.ctrlKey && e.code != "ControlLeft" && e.code != "ControlRight") || (e.ctrlKey && e.code == "KeyV")) {
                    if (e.code != "ArrowDown" && e.code != "ArrowUp" && e.code != "ArrowLeft" && e.code != "ArrowRight") {
                        console.log("执行了清除")
                        let menu = targetView!.getMenu();
                        menu.clearChildren();
                        var isTitle = targetView!.isTableTitle();
                        let formula;
                        if (isTitle) {
                            let value = target.getAttribute(CommonConstant.NODE_TgtFormKey);
                            formula = `Macro_GetFormAllField('${value}','target','${e.target.value}')`;
                        } else {
                            let value = target.getAttribute(CommonConstant.NODE_Table_Key);
                            formula = `Macro_GetTableFields('${value!.split("|")[0]}','${value!.split("|")[1]}','target','${e.target.value}')`;
                        }
                        this.parseFieldDataToTgtMenu(menu, formula, self, targetView!, target);
                    }
                }
            });

            targetView.setInitTextEditor(true);
        }

    }


    parseFieldDataToSourceMenu (menu: Menu, formula:string, self:DataMapWorkSpace, sourceView:DataMapSourceTableView, target: HTMLElement) {

        this.operationDispatcher.eval(formula).then((res) => {
            if(this.mune) {
                this.mune.clearChildren();
            }
            if (!res) return;
            var r = res.toString();
            let strings = r.split(";");
            this.mune = menu;
            for (let string of strings) {
                let keyAndCaption = string.split(",");
                let menuItem = menu.addMenuItem("FieldItem", keyAndCaption[1]);
                menuItem.getDomElement().setAttr("Value", keyAndCaption[0]);
                menu.setStyle("overflow", "auto");
                menuItem.addEventListener(Events.MouseDown, (event: any) => {
                    let selectSourceView = this.getGraphicView().getSelectSourceView();
                    if (selectSourceView) {
                        let tagNode = selectSourceView.getTagNode();
                        let value = menuItem.getDomElement().getAttr("value");
                        let keyAndValue = value!.split("|");
                        this.getGraphicView().addSourceFieldCmd(this.getGraphicView(), keyAndValue[0], keyAndValue[1], tagNode);
                    }
                    menu.hide();
                })

            }
            // @ts-ignore
            let top = target.parentElement.parentElement.offsetParent.offsetTop
            menu.show(0, top + 60, sourceView!.getWidth().getValue(), 150, sourceView!.getEl())
            if(target.className.indexOf("selectImg") != -1) {
                menu.getEl().focus();
            }
        })
    }

    parseFieldDataToTgtMenu (menu: Menu, formula:string, self:DataMapWorkSpace, tgtView:DataMapTargetTableView, target: HTMLElement) {
        this.operationDispatcher.eval(formula).then((res) => {
            if (!res) return;
            if(this.mune){
                this.mune.clearChildren();
            }
            var r = res.toString();
            let strings = r.split(";");
            this.mune = menu;
            for (let string of strings) {
                let keyAndCaption = string.split(",");
                let menuItem = menu.addMenuItem("FieldItem", keyAndCaption[1]);
                menuItem.getDomElement().setAttr("Value", keyAndCaption[0]);
                menu.setStyle("overflow", "auto");
                menuItem.addEventListener(Events.MouseDown, (event: any) => {
                    let selectTargetView = this.getGraphicView().getSelectTargetView();
                    if (selectTargetView) {
                        let tagNode = selectTargetView.getTagNode();
                        let value = menuItem.getDomElement().getAttr("value");
                        let keyAndValue = value!.split("|");
                        this.getGraphicView().addTargetFieldCmd(this.getGraphicView(), keyAndValue[0], keyAndValue[1], tagNode);
                    }
                    menu.hide();
                })

            }
            // @ts-ignore
            let top = target.parentElement.parentElement.offsetParent.offsetTop
            menu.show(0, top + 60, tgtView!.getWidth().getValue(), 150, tgtView!.getEl());
            if (target.className.indexOf("selectImg") != -1) {
                menu.getEl().focus();
            }
        })
    }

    deleteElement(): void {
    }

    protected prepareToolbar(toolbar: Toolbar): void {
        let btnSave = new Button("保存");
        btnSave.setOnClickListener((e: MouseEvent) => {
            this.save();
        });
        toolbar.addItem(btnSave);

        let btnDelete = new Button("删除");
        btnDelete.setOnClickListener((e: MouseEvent) => {
            this.delete();
        });
        toolbar.addItem(btnDelete);
        var btnUndo = new Button("撤销");
        btnUndo.setOnClickListener((e: MouseEvent) => this.viewContainer.undo());
        toolbar.addItem(btnUndo);
        var btnRedo = new Button("重做");
        btnRedo.setOnClickListener((e: MouseEvent) => this.viewContainer.redo());
        toolbar.addItem(btnRedo);
        // 分隔符
        toolbar.addSeparator();
        var btnSwitch = new Button("源码");
        btnSwitch.setOnClickListener((e: MouseEvent) => {
            this.viewContainer.toggleSourceVisible();
            btnSwitch.setCaption(this.viewContainer.isSourceVisible() ? "视图" : "源码");
            let xmlEntity = this.getGraphicView().getXmlEntity();
            xmlEntity.select(xmlEntity.getTagRoot());
        });
        toolbar.addItem(btnSwitch);
        toolbar.addSeparator();
        var btnAddFeed = new Button("添加反填表单");
        btnAddFeed.setOnClickListener((e: MouseEvent) => {
            var exp = "ShowModal('D_AddFeedBackForm',{FormKey:{GetDataMapRootAttrValue('Key')}});"
            this.operationDispatcher.eval(exp);
        });
        toolbar.addItem(btnAddFeed);
    }

    save(): void {
        let tagRoot = this.getXmlEntity().getTagRoot();
        if (!tagRoot) return;
        let exp = `SaveExcel()`;
        this.operationDispatcher.eval(exp);
    }

    delete(): void {
        var exp = `DeleteBpm()`;
        this.operationDispatcher.eval(exp);
    }

    load(xml: string) {
        super.load(xml);
        let xmlEntity = XmlEntity.parse(xml);
        let rootChild = xmlEntity.getRoot().getChild("Map");
        if (!rootChild) return;
        if(rootChild.hasAttribute(CommonConstant.ATTR_Extend)){
            var exp =`Macro_GetExtendDataMapLinks('${rootChild.getAttributeValue(CommonConstant.ATTR_Extend)}')`;
            this.operationDispatcher.eval(exp).then((res) =>{
                let sourceToTargetLinks = res.sourceToTarget;
                let targetToFeedbackLinks = res.targetToFeedback;
                let serviceSourceTable = res.serviceSourceTable;
                this.getGraphicView().setExtendSourceToTargetLinks(sourceToTargetLinks);
                this.getGraphicView().setExtendTargetToFeedbackLinks(targetToFeedbackLinks);
                this.getGraphicView().setServiceSourceTable(serviceSourceTable);
            });
        }
        CommonFunction.GetAllFields(rootChild, this.operationDispatcher, this.getGraphicView(), this.getXmlEntity(), this.path!);
    }

    public getOperationDispatcher() {
        return this.operationDispatcher;
    }

    protected createPropertyIO(): IPropertyIO {
        return new DataMapProperty(this.getXmlEntity(), this.getGraphicView());
    }

}