import BorderPanel from "../../common/component/container/panel/BorderPanel";
import JComponent from "../../common/component/JComponent";
import {EDesignMode} from "../../common/enum/Enums";
import ActionNames from "../../common/event/ActionNames";
import IEventListener from "../../common/listener/IEventListener";
import SelectionChangeListener from "../../common/listener/SelectionChangeListener";
import IOperatioExecutor from "../../common/operation/IOperatioExecutor";
import OperationDispatcher from "../../common/operation/OperationDispatcher";
import SelectionModel from "../../common/selection/SelectionModel";
import Paras from "../../common/struct/Paras";
import SizeInfo from "../../common/struct/SizeInfo";
import IWorkSpace from "../../common/workspace/IWorkSpace";
import IXmlEntity from "../../common/xml/IXmlEntity";
import XmlEntity from "../../common/xml/XmlEntity";
import {IPropertyContainer, IPropertyIO, IPropertyPlugin} from "../plugin/property/base/PropertyDefine";
import PropertyIO from "../plugin/property/base/PropertyIO";
import AbstractDesignView from "./AbstractDesignView";
import AbstractGraphicView from "./AbstractGraphicView";
import ComplexDesignView from "./ComplexDesignView";

export default abstract class AbstractWorkSpace<T extends AbstractGraphicView<any>, B extends JComponent<any>> extends BorderPanel implements IWorkSpace, IEventListener {

    private designView: AbstractDesignView<any>;

    private curPropertyContainer: IPropertyContainer<any>;

    protected operationDispatcher: OperationDispatcher;

    protected selectionListener?: SelectionChangeListener;

    private xmlEntity!: IXmlEntity;

    private mode: EDesignMode = EDesignMode.User;

    protected viewContainer!: ComplexDesignView<T, B>;
    private toolbar: B;

    constructor(mode: EDesignMode, toolbar: B) {
        super(0, 0, 0, 0);
        this.mode = mode;
        this.toolbar = toolbar;
        this.operationDispatcher = new OperationDispatcher();
        var graphicView = this.createGraphicView(mode);
        this.viewContainer = new ComplexDesignView(graphicView, toolbar);
        this.designView = this.createDesignView(this.viewContainer, this.operationDispatcher);
        this.prepareToolbar(toolbar);
        this.curPropertyContainer = this.designView;
        this.setCenter(this.designView);

        this.initKeyEvent(this.getEl());
    }

    getToolBar(): B {
        return this.toolbar
    }

    getMode(): EDesignMode {
        return this.mode;
    }

    isUserModel(): boolean {
        return this.mode == EDesignMode.User;
    }

    getSelectionModel(): SelectionModel {
        return this.getXmlEntity().getSelectionModel();
    }

    setOperationExecutor(operatioExecutor: IOperatioExecutor): void {
        this.operationDispatcher.setExecutor(operatioExecutor);
    }

    getXmlEntity(): IXmlEntity {
        return this.xmlEntity;
    }

    setEditable(editable: boolean) {
        this.operationDispatcher.setEditable(editable);
        this.viewContainer.setEditable(editable);
    }

    isSourceView(): boolean {
        return this.viewContainer.isSourceVisible()
    }

    reloadSource(): void {
        this.viewContainer.reloadSource();
    }

    loadView(xmlEntity: IXmlEntity): void {
        this.viewContainer.loadXml(xmlEntity);
    }

    getGraphicView(): T {
        return this.viewContainer.getGraphicView();
    }

    getPropertyPlugin(): IPropertyPlugin {
        return this.curPropertyContainer.getPropertyUIContainer().getPropertyPlugin();
    }

    private parent?: HTMLElement;

    doLayout(parent: HTMLElement): void {
        this.parent = parent;
        if (!parent.contains(this.getEl())) {
            parent.appendChild(this.getEl());
        }
        this.updateLayout();
        window.addEventListener("resize", (e: any) => {
            this.updateLayout();
            this.resizeDesignView();
        });
    }

    private resizeDesignView() {
        this.designView.resize(this.getEl().clientWidth, this.getEl().clientHeight);
    }

    updateLayout() {
        this.parent && this.reLayout(SizeInfo.valueOfPX(this.parent.clientWidth), SizeInfo.valueOfPX(this.parent.clientHeight));
        this.resizeDesignView();
    }

    load(xml: string) {
        this.xmlEntity = XmlEntity.parse(xml);
        this.loadView(this.xmlEntity);
        this.selectionListener = new SelectionChangeListener(this.curPropertyContainer, this.createPropertyIO()); //this.devDesignView.getPropertyPlugin(), this.devDesignView.getPropertyContainer().getEl()); 
        this.xmlEntity.addEventListener(this);
        this.xmlEntity.getSelectionModel().select(this.xmlEntity.getTagRoot());
    }

    toXml(bSimple: boolean): string {
        return this.getXmlEntity().getXml(bSimple);
    }

    private initKeyEvent(el: HTMLElement): void {
        el.addEventListener('keyup', (e) => {
            if (!this.operationDispatcher.isEditable()) return;
            if (this.isSourceView()) return;

            if (e.key === 'Delete') {
                this.deleteElement();
            } else if (e.ctrlKey && e.key === 'z') {
                this.getXmlEntity().undo();
            } else if (e.ctrlKey && e.key === 'y') {
                this.getXmlEntity().redo();
            } else if (e.key === 'F2') {
                //this.viewContainer.toggleSourceViewVisible();
            }
        });
    }

    notifyEvent(cmd: string, paras: Paras) {
        if (!this.selectionListener) return;
        switch (cmd) {
            case ActionNames.model_node_update_select:
                if (paras.getElement()) {
                    this.selectionListener.changeSelection(paras.getElement());
                } else {
                    this.selectionListener.changeSelection(this.getXmlEntity().getTagRoot());
                }
                break;
            case ActionNames.model_node_properties_commit:
                this.designView.getPropertyUIContainer().getPropertyPlugin().commit();
                break;
        }
    }

    protected createPropertyIO(): IPropertyIO {
        return new PropertyIO(this.getXmlEntity());
    }

    abstract createDesignView(viewContainer: ComplexDesignView<T, B>, operationDispatcher: OperationDispatcher): AbstractDesignView<any>

    abstract createGraphicView(mode: EDesignMode): T;

    abstract deleteElement(): void

    protected abstract prepareToolbar(toolbar: B): void;
}


/*this.getEl().oncontextmenu = function(e) {
    //取消默认的浏览器自带右键
    e.preventDefault();
};
/*window.oncontextmenu=function(e){
    //取消默认的浏览器自带右键 很重要！！
    e.preventDefault();
    //获取我们自定义的右键菜单
    var menu=document.querySelector("#menu");
    //根据事件对象中鼠标点击的位置，进行定位
    menu.style.left=e.clientX+'px';
    menu.style.top=e.clientY+'px';
    //改变自定义菜单的宽，让它显示出来
    menu.style.width='100px';
    menu.style.height='auto';
}*/