import { ENodeType } from "../../enum/Enums";
import StringBuilder from "../../util/StringBuilder";
import TypeUtil from "../../util/TypeUtil";
import Attribute from "../attribute/Attribute";
import IAttribute from "../attribute/IAttribute";
import IXmlNodeVisitor from "../IXmlNodeVisitor";
import INodeOutput from "../output/INodeOutput";
import XmlUtil from "../util/XmlUtil";
import ContentNode from "./ContentNode";
import INode from "./INode";
import ITagNode from "./ITagNode";
import GlobalConstants from '../../struct/GlobalConstants';
import Var from "../../struct/Var";
import VarIndexUtil from "../../util/VarIndexUtil";
import BaseConstants from "../../../design/base/BaseConstants";

export default class TagNode implements ITagNode {
	// 行号	
	private startLine: number = -1;

	private endLine: number = -1;
	
	private tagName: string;

	private parent?: ITagNode;
	
	private children: INode[] = [];
	
	private attributes: IAttribute[] = [];

	private selected = false;

	public constructor(tagName: string) {
		this.tagName = tagName;
	}

	/**
	 * 通过行号查找当前的TagNode
	 * @param line 
	 * @returns 
	 */
	findNodeByLine(line: number): INode | null {
		if (line == this.startLine) {
			return this;
		}
		for (let node of this.children) {
			var result = node.findNodeByLine(line);
			if (result) return result;
		}
		return null;
	}

	getStartLine(): number {
		return this.startLine;
	}

	getEndLine(): number {
		return this.endLine;
		/* var nextSibling = this.nextSibling();
		if (nextSibling) {
			return nextSibling.getLine() - 1;
		}
		if (this.parent) {
			return this.parent.getEndLine() - 1;
		}
		return -1; */
	}

	/**
	 * 取下一个兄弟节点
	 */
	nextSibling(): INode | null {
		var parent = this.getParent();
		if (parent) {
			var siblings = parent.getChildren();
			var index = siblings.indexOf(this);
			if (index >=0 && (index + 1) < siblings.length) {
				return siblings[index + 1];
			}
		}
		return null;
	}

	/**
	 * 通过关键属性值，查找某个节点
	 * @param tag 
	 * @param primaryKey 
	 * @param primaryValue 
	 * @returns 
	 */
	find(tag: string, primaryKey: string, primaryValue: string): ITagNode | undefined {
		if (this.tagName == tag && this.getAttributeValue(primaryKey) == primaryValue) {
			return <ITagNode> this;
		}
		for (let child of this.children) {
			if (child.getNodeType() == ENodeType.TAG) {
				var result = (<ITagNode> child).find(tag, primaryKey, primaryValue);
				if (result) return result;
			}
		}
		return undefined;
	}

	/**
	 * 是否是用户模式下定义的节点
	 * 
	 * @returns 
	 */
    isUserDefine(): boolean {
        return this.getAttributeValue(GlobalConstants.S_IsUserDefine) == 'true';
    }

	setContent(content: string): void {
	}

	setCDATAValue(value: string): void {
		var cdataNode = this.ensureCDataNode();
		if (value) {
			cdataNode.setContent(`<![CDATA[${value}]]>`);
		} else {
			this.removeChild(cdataNode);
		}
	}

	getCDATAValue(): string {
		var cdata = this.getCDATA();
		return cdata.length > 10 ?  cdata.substring(9, cdata.length - 3) : "";
	}

	getCDATA(): string {
		return this.ensureCDataNode().getContent();
	}

	removeAttribute(key: string): void {
		var index = this.indexOfAttribute(key);
		if (index > -1) {
			this.attributes.splice(index, 1);
		}
	}

	setSelected(b: boolean): void {
		this.selected = b;
	}

	isSelected(): boolean {
		return this.selected;
	}

	setParent(parent: ITagNode): void {
		this.parent = parent;
	}

	getParent(): ITagNode | undefined {
		return this.parent;
	}

	hasAttribute(key: string): boolean {
		let attr = this.getAttribute(key);
		return attr != null;
	}

	setAttributeValue(key: string, value: string, defaultValue?: string): void {
		if (value == defaultValue) {
			this.removeAttribute(key);
			return;
		}

		let attr = this.getAttribute(key);
		if (attr) {
			attr.setValue(value);
		} else {
			this.addAttribute(key, value);	
		}
	}

	getAttributeValue(key: string, defaultValue: string = ''): string {
		let attr = this.getAttribute(key);
		if (attr) {
			return attr.getValue();
		} else {
			return defaultValue;
		}
	}

	getAttributeValueInt(key: string, defaultValue: number = 0): number {
		return this.hasAttribute(key) ? parseInt(this.getAttributeValue(key)) : defaultValue;
	}

	getAttributeValueBoolean(key: string, defaultValue: boolean): boolean {
		return this.hasAttribute(key) ? TypeUtil.toBoolean(this.getAttributeValue(key)) : defaultValue;
	}

	getNodeType(): ENodeType {
		return ENodeType.TAG;
	}

	getTagName(): string {
		return this.tagName;
	}

	getChildren(): INode[] {
		for (let child of this.children) {
			if(child.getNodeType() && (child.getNodeType() == "comment" || child.getNodeType() == "text")) {
				let index = this.children.indexOf(child)
				this.children.splice(index, 1);
			}
		}
		return this.children;
	}

	/**
	 * 删除子节点
	 * onlyClearUserDefine： 是否仅删除用户模式下定义的节点
	 * 
	 */
	clearChildren(tag?: string, onlyClearUserDefine?: boolean,  attrkey?: string, attrValue?: string): void {
		var tmpChildren = [];
		if (tag) {
			for (let child of this.children) {
				if (child instanceof TagNode) {
					var tagNode = <TagNode> child;
					var value = attrkey ? tagNode.getAttributeValue(attrkey) : "";
					attrValue = attrValue ? attrValue : "";
					if (tagNode.tagName != tag || value != attrValue) {
						tmpChildren.push(tagNode);
						//onlyClearUserDefine && tagNode.isUserDefine() && tmpChildren.push(tagNode);
					} else if (onlyClearUserDefine && !tagNode.isUserDefine()) {
						tmpChildren.push(tagNode);
					}
				} else {
					tmpChildren.push(child);
				}
			}
		}
		this.children = tmpChildren;
	}

	getChild(tag: string): ITagNode | null {
		for (let child of this.children) {
			if (child.getNodeType() == ENodeType.TAG) {
				var tagNode = <ITagNode> child;
				if (tagNode.getTagName() == tag) {
					return tagNode;
				}
			} 
		}
		return null;
	}

	ensureChild(tags: string): ITagNode {
		if (!tags) return this;
		
		var pos = tags.indexOf('|')
		if (pos <= 0) {
			return this.ensureSingleChild(tags);
		} 
		var tag = tags.substring(0, pos);
		var child = this.ensureSingleChild(tag);
		return child.ensureChild(tags.substring(pos + 1));
	}

	ensureSingleChild(tag: string): ITagNode {
		var child = this.getChild(tag);
		if (child) return child;
		child = new TagNode(tag);
		this.addChild(child);
		return child;
	}

	public getAttributes(): IAttribute[] {
		return this.attributes;
	}

	public getContent(): string {
		return toString();
	}

	public addChild(child: INode): INode {
		child.setParent(this);
		this.children.push(child);
		return child;
	}

	removeChild(child: INode): void {
        var index = this.children.indexOf(child); 
        if (index>= 0) { 
            this.children.splice(index, 1); 
        }
	}

	addAttribute(key: string, value: string): IAttribute {
		var attribute = new Attribute(key, value);
		this.attributes.push(attribute)
		return  attribute;
	}

	getAttribute(key: string): IAttribute | null {
		for (let attr of this.attributes) {
			if (attr.getKey() == key) {
				return attr;
			}
		}
		return null;
	}

	hasChild(): boolean {
		return this.children.length > 0;
	}

	toXml(bSimple: boolean): string {
		let sb = new StringBuilder();
		this.writeXml(this.getLevel(), sb, bSimple);
		return sb.toString();
	}

	getLevel(): number {
		var tmpTag: ITagNode | undefined = this;
		var level = -2;
		while (tmpTag) {
			level ++;
			tmpTag = tmpTag.getParent();
		}
		return level;
	}

	toJSONString(): string {
		let sb = new StringBuilder();
		this.writeJSON(sb);
		return sb.toString();
	}

	writeJSON(sb: StringBuilder) {
		sb.append("{tag:").append(this.tagName);
		var lenAttr = this.attributes.length;
		if (lenAttr > 0) {
			var index = 0;
			sb.append(",");
			sb.append(GlobalConstants.S_Attributes).append(":{")
			for (let attr of this.attributes) {
				sb.append(attr.getKey()).append(":'").append(attr.getValue()).append("'");
				(index < lenAttr - 1) && sb.append(",");
				index ++;
			}
			sb.append("}");
		}

		if (this.hasChild()) {
			if (lenAttr > 0) {
				sb.append(",");
			}
			var index = 0;
			var size = this.children.length;
			sb.append(GlobalConstants.S_ChildNodes).append(":[")
			for (let node of this.children) {
				node.writeJSON(sb);
				(index < size - 1) && sb.append(",");
				index ++;
			}
			sb.append("]");
		}
		sb.append("}");
	}

	writeXml(level: number, sb: StringBuilder, bSimple: boolean): boolean {
		var beginIndex = sb.getLastIndex();
		if (!bSimple) {
			XmlUtil.indentLine(level, sb);
		}

		var linebreak = bSimple ? "" : "\r\n";
		
		var bNullTag: boolean = (this.tagName == null || this.tagName == "");
		if (!bNullTag) {
			sb.append("<").append(this.tagName);
			for (let attr of this.attributes) {
				sb.append(" ").append(attr.toString());
			}
			if (!this.hasChild()) {
				// 移除 <TagName/> 这种空节点
				if (this.attributes.length > 0 || this.tagName == BaseConstants.Node_Cell|| this.tagName == BaseConstants.Node_Rows) {
					sb.append("/>");
					return true;
				} else {
					sb.rollbackTo(beginIndex);
					return false;
				}
			} else {
				sb.append(">");
			}
		}

		if (this.hasChild()) {
			var bFirstLine: boolean = bNullTag;
			for (let node of this.children) {
				if (bFirstLine) {
					bFirstLine = false;
					node.writeXml(level + 1, sb, bSimple);
				} else {
					sb.append(linebreak);
					var result = node.writeXml(level + 1, sb, bSimple);
					if (!result) sb.deleteLastChar(); // 若未写入，那么删除掉换行符
				}
			}
		}
		
		if (!bNullTag) {
			sb.append(linebreak);
			if (!bSimple) {
				XmlUtil.indentLine(level, sb);
			}
			sb.append("</").append(this.tagName).append(">");
		}
		return true;
	}

	resetLineIndex(varIndex: Var<number>): void {
		if (!this.hasChild() && this.attributes.length == 0 && this.tagName != BaseConstants.Node_Cell&& this.tagName != BaseConstants.Node_Rows) {
			this.startLine = -1;
			this.endLine = -1;
			return;
		}
		this.startLine = VarIndexUtil.incVarIndex(varIndex);
		if (this.hasChild()) {
			for (let node of this.children) {
				node.resetLineIndex(varIndex);
			}
		} else {
			this.endLine = this.startLine;
			return;
		}
		// 结束节点标志
		this.endLine = VarIndexUtil.incVarIndex(varIndex);
	}

	outputLevel(level: number, outputHandler: INodeOutput): void {
		outputHandler.out(level, this, true);
		for (let node of this.children) {
			node.outputLevel(level + 1, outputHandler);
		}

		if (this.hasChild()) {
			outputHandler.out(level, this, false);
		}
	}

	output(outputHandler: INodeOutput): void {
		var bNullTag: boolean = (this.tagName == null || this.tagName == "");
		if (bNullTag) {
			for (let node of this.children) {
				node.outputLevel(0, outputHandler);
			}
		}
	}

	getMaxNodeID(): number {
		var id = this.getAttributeValueInt(GlobalConstants.S_ID);
		for (let child of this.children) {
			id = Math.max(id, child.getMaxNodeID());;
		}
		return id;
	}

	getTagNode(): ITagNode {
		return this;
	}

	accept(visitor: IXmlNodeVisitor) {
        visitor.visit(this);
        for (let child of this.children) {
            child.accept(visitor);
        }
    }

   private indexOfAttribute(key: string): number {
		var index = 0;
		for (let attr of this.attributes) {
			if (key == attr.getKey()) {
				return index;
			}
			index ++;
		}
		return -1;
	}

	private ensureCDataNode(): INode {
		for (let node of this.children) {
			if (node.getNodeType() == ENodeType.CDATA) {
				return node;
			}
		}
		var node = new ContentNode(ENodeType.CDATA, "");
		this.addChild(node);
		return node;
	}
}
