import CmdQueue from "../cmd/CmdQueue";
import {ENodeType} from "../enum/Enums";
import ActionNames from "../event/ActionNames";
import IEventListener from "../listener/IEventListener";
import ISelectable from "../selection/ISelectable";
import SelectionModel from "../selection/SelectionModel";
import Paras from "../struct/Paras";
import IXmlEntity from "./IXmlEntity";
import DefaultNodeFactory from "./node/DefaultNodeFactory";
import INode from "./node/INode";
import INodeFactory from "./node/INodeFactory";
import ITagNode from "./node/ITagNode";
import NodeFactoryWithoutComment from "./node/NodeFactoryWithoutComment";
import RootNode from "./node/RootNode";
import XmlParser from "./XmlParser";

export default class XmlEntity implements IXmlEntity {

    private static nodeFactoryWidthoutComment = new NodeFactoryWithoutComment();

    private static defaultNodeFactory = new DefaultNodeFactory();

    private cmdQueue: CmdQueue = new CmdQueue();

    private maxNodeID = -1;

    //private selectionChangeListeners: ISelectionChangedListener[] = [];

    private eventListeners: IEventListener[] = [];

    private root: ITagNode = new RootNode();

    private withoutComment: boolean = false;

    private selectionModel: SelectionModel;

    constructor(withoutComment: boolean = false) {
        this.withoutComment = withoutComment;
        this.selectionModel = new SelectionModel(this);
    }

    find(tag: string, primaryKey: string, primaryValue: string): ITagNode | undefined {
        return this.root.find(tag, primaryKey, primaryValue);
    }

    undo(): void {
        this.cmdQueue.undoCmd();
    }

    redo(): void {
        this.cmdQueue.redoCmd();
    }

    getCmdQueue(): CmdQueue {
        return this.cmdQueue;
    }

    unselect(selectable: ISelectable) {
        selectable.setSelected(false);
        this.updateSelect(selectable.getTagNode());
    }

    unSelectAll() {
        // 切换选择前需先提交属性修改
        var paras = Paras.newInstance().setElement(this.selectionModel.getSelectionItems());
        this.fireEvent(ActionNames.model_node_properties_commit, paras); 
        var tagRoot = this.getTagRoot();
        this.unSelectChildren(tagRoot.getChildren());
    }

    private unSelectChildren(children: INode[]) {
        for (let child of children) {
            if (child.getNodeType() == ENodeType.TAG) {
                var tagNode = <ITagNode>child;
                if (tagNode.isSelected()) {
                    tagNode.setSelected(false);
                    this.updateSelect(tagNode);
                }
                this.unSelectChildren(tagNode.getChildren());
            }
        }
    }

    select(selectable: ISelectable | undefined): void {
        if (selectable) {
            selectable.setSelected(true);
            this.updateSelect(selectable.getTagNode());
        } else {
            this.updateSelect(undefined);
        }
    }

    getSelectionModel(): SelectionModel {
        return this.selectionModel;
    }

    addEventListener(listener: IEventListener): void {
        this.eventListeners.push(listener);
    }

    getTagNode(): ITagNode {
        return this.root;
    }

    getRoot(): ITagNode {
        return this.root;
    }

    addNode(node: ITagNode, parentNode?: ITagNode): ITagNode {
        var parent = parentNode ? parentNode : this.getTagRoot()
        parent.addChild(node);
        var paras = Paras.newInstance().setElement(node);
        this.fireEvent(ActionNames.model_node_add, paras);
        return node;
    }

    removeNode(node: ITagNode, fireEvent: boolean = true) {
        node.getParent()?.removeChild(node);
        var paras = Paras.newInstance().setElement(node);
        this.fireEvent(ActionNames.model_node_remove, paras);
    }

    removeSelectNode(): void {
        var item = this.selectionModel.getSingleSeletionItem();
        if (item) {
            this.removeNode(item.getTagNode());
        }
    }

    ensureNode(tagName: string, parent: ITagNode): ITagNode {
        var node = parent.getChild(tagName);
        if (!node) {
            node = this.getNodeFactory().createTagNode(tagName);
            this.addNode(node, parent);
        }
        return node;
    }

    getNodeFactory(): INodeFactory {
        return this.withoutComment ? XmlEntity.nodeFactoryWidthoutComment : XmlEntity.defaultNodeFactory;
    }

    getXml(bSimple: boolean): string {
        return this.root.toXml(bSimple);
    }

    getTagRoot(): ITagNode {
        for (let child of this.root.getChildren()) {
            if (child.getNodeType() == ENodeType.TAG) {
                return <ITagNode>child;
            }
        }
        return this.root;
    }

    update(tagNode: ITagNode | undefined, bUpdateAttrOnly: boolean = false): void {
        var paras = Paras.newInstance().setElement(tagNode).add(Paras.P_UPDATE_ATTR_ONLY, bUpdateAttrOnly);
        this.fireEvent(ActionNames.model_node_update, paras);
    }


    updateSelect(tagNode: ITagNode | undefined) {
        var paras = Paras.newInstance().setElement(tagNode);
        if (this.selectionModel.getBMultiSelection()) {
            this.fireEvent(ActionNames.model_node_update_multi_select, paras);
        } else {
            this.fireEvent(ActionNames.model_node_update_select, paras);

        }
    }

    fireEvent(action: string, paras: Paras) {
        for (let listener of this.eventListeners) {
            listener.notifyEvent(action, paras);
        }
    }

    reload(xml: string): void {
        this.root.clearChildren();
        let parser: XmlParser = new XmlParser();
        parser.parse(xml, this);
        this.cmdQueue.clear();
    }

    findNodeByLine(line: number): INode | null {
        return this.root.findNodeByLine(line);
    }

    genNextNodeID(): number {
        if (this.maxNodeID == -1) {
            this.maxNodeID = this.getTagRoot().getMaxNodeID();
        }
        return ++this.maxNodeID;
    }

    fireCommit(): void {
        var paras = Paras.newInstance().setElement(this.selectionModel.getSelectionItems());
        this.fireEvent(ActionNames.model_node_properties_commit, paras);
    }

    public static parse(xml: string, withoutComment: boolean = false): XmlEntity {
        let parser: XmlParser = new XmlParser();
        var xmlEntity = new XmlEntity(withoutComment);
        parser.parse(xml, xmlEntity);
        return xmlEntity;
    }

    public static parseTagNode(xml: string): ITagNode {
        let parser: XmlParser = new XmlParser();
        var xmlEntity = new XmlEntity(true);
        parser.parse(xml, xmlEntity);
        return xmlEntity.getTagRoot();
    }

    public static parseTagNodeCollection(xml: string): INode[] {
        var xmlWithRoot = `<ROOT>${xml}</ROOT>`;
        var entity = XmlEntity.parse(xmlWithRoot);
        var root = entity.getTagRoot();
        return root.getChildren();
    }
}
