import IPropertyItem, { IPropertyGroup, IPropertyIO } from "../base/PropertyDefine";
import AbstractDomElement from "../../../../common/dom/AbstractDomElement";
import PropertyUIGroup from "./PropertyUIGroup";
import IEventListener from "../../../../common/listener/IEventListener";
import Paras from "../../../../common/struct/Paras";
import Events from "../../../../common/event/Events";
import PropertyUIControl from "./PropertyUIControl";
import DomUtil from "../../../../common/dom/DomUtil";
import IOperatioExecutor from "../../../../common/operation/IOperatioExecutor";
import PropertyStateDelegate from "./state/PropertyStateDelegate";
import IPropertyView from "./IPropertyView";
import SizeInfo from "../../../../common/struct/SizeInfo";

export default class PropertyUIPanel extends AbstractDomElement implements IEventListener, IPropertyView {

    public static PROP_RATIO: number = 0.5;

    private optDispatcher?: IOperatioExecutor;

    private propertyGroups: IPropertyGroup[];

    private callback: IPropertyIO;

    private stateDelegate: PropertyStateDelegate;

    constructor(propertyGroups: IPropertyGroup[], callback: IPropertyIO, optDispatcher?: IOperatioExecutor) {
        super();
        this.propertyGroups = propertyGroups;
        this.callback = callback;
        this.optDispatcher = optDispatcher;
        this.stateDelegate = new PropertyStateDelegate(this);
        this.addClassArray(this.styleFactory.getPropertyPanel());
        this.setStyle("position","relative");
        this.init(propertyGroups);
        this.initEvents();
    }

    init(propertyGroups: IPropertyGroup[]): void {
        this.clearChildren();
        if (propertyGroups) {
            for (let group of propertyGroups) {
                var uiGroup: PropertyUIGroup = new PropertyUIGroup(group, this.callback, this.stateDelegate, this.optDispatcher);
                this.addChild(uiGroup);
                uiGroup.setOnEventListener(this);
            }
            this.resetHeight();
        }
    }
    
    /**
     * 事件注册
     */
    private initEvents() {
        this.addEventListener('mousedown', (e: MouseEvent) => {
            this.stateDelegate.getCurrentState().mousedown(e);
        });

        this.addEventListener('mousemove', (e: MouseEvent) => {
            this.stateDelegate.getCurrentState().mousemove(e);
        });

        this.addEventListener('mouseup', (e: MouseEvent) => {
            this.stateDelegate.getCurrentState().mouseup(e);
        });
    }
    
    notifyEvent(cmd: string, paras: Paras): void {
        switch (cmd) {
            case Events.Focus:
                this.focus(paras.getElement());
                break;
            case Events.VauleChanged:
                this.fireValueChanged(paras.getElement());
                break;
        }
    }

    focus(el: HTMLElement): void {
        var lineEl: HTMLElement | null =  DomUtil.findParentElByClassName(el, "property-control-line");
        var propertyLine: PropertyUIControl | null = <PropertyUIControl> this.findElement(lineEl);
        if (!propertyLine || propertyLine.isFocused()) {
            return;
        }
        this.unfocus();
        propertyLine.setFocused(true);
    }

    setPropertyValue(key: string, value: any, text?: string): void {
        var group: PropertyUIGroup;
        for (let child of this.getChildren()) {
            group = <PropertyUIGroup> child;
            var uiControl = group.getPropertyControl(key);
            uiControl?.setPropertyValue(value, text);
        }
    }

    setFocus(key: string): void {
        var group: PropertyUIGroup;
        for (let child of this.getChildren()) {
            group = <PropertyUIGroup> child;
            var uiControl = group.getPropertyControl(key);
            uiControl?.setFocused(true);
        }
    }

    unfocus() {
        var group: PropertyUIGroup;
        for (let child of this.getChildren()) {
            group = <PropertyUIGroup> child;
            group.unfocus();
        }
        // 焦点发生变化后，重新显示属性栏，因为某些属性项发生变化后可能会影响其他属性项
        //this.updateVisiableAndEditable();
    }

    private fireValueChanged(propertyUIControl: PropertyUIControl) {
        //var key = control.getKey();
        if (!propertyUIControl) return;
        var propItem = propertyUIControl.getPropertyItem(); //this.getPropertyItem(key);
        if (propItem) {
            var parentEl = this.getEl().parentElement;
            var top = parentEl?.scrollTop;
            this.callback.setValue(propItem, propertyUIControl.getValue());
            //if (propItem.onvaluechanged) {
                //var exp = String(propItem.onvaluechanged);
                //exp = exp.replace(GlobalVariable.V_SELF_KEY, key);
                //exp = exp.replace(GlobalVariable.V_SELF_VALUE, control.getValue());
                //this.optDispatcher && await this.optDispatcher.eval(exp);
            //}
            this.updateVisiableAndEditable();
            if (parentEl != null && top) {
                parentEl.scrollTop = top;
            }
        }
    }

    private async updateVisiableAndEditable() {
        for (var child of this.getChildren()) {
            if (child instanceof PropertyUIGroup) {
                if ((<PropertyUIGroup> child).isExpanded()) {
                   await child.updateVisiableAndEditable();
                }
            }
        }
    }

    resize(width:number, height:number) {
        this.setStyleWidth(SizeInfo.valueOfPX(width));
        this.setStyleHeight(SizeInfo.valueOfPX(height));
        this.resetHeight();
    }

    getRatio(): number {
        return PropertyUIPanel.PROP_RATIO;
    }

    setRatio(ratio: number): void {
        PropertyUIPanel.PROP_RATIO = ratio;
        for( let child of this.getChildren()) {
            if (child instanceof PropertyUIGroup) {
                (<PropertyUIGroup> child).resizeByRatio(ratio);
            }
        }
    }

    getInnerHeight(): number {
        var total = 0;
        for( let child of this.getChildren()) {
            if (child instanceof PropertyUIGroup) {
                var uiGroup = <PropertyUIGroup> child;
                total += uiGroup.getInnerHeight();
            }
        }
        return total;
    }

    resetHeight(): void {
        var realHeight = this.getInnerHeight() + 10;
        this.setStyleHeight(SizeInfo.valueOfPX(realHeight));
        this.resetChildSize();
    }

    private resetChildSize(): void {
        var parentEl =  this.getEl().parentElement;
        parentEl && this.setStyleWidth(SizeInfo.valueOfPX(parentEl.clientWidth));
        for( let child of this.getChildren()) {
            if (parentEl) {
                child.resize(parentEl.clientWidth, this.getEl().clientHeight);
            }
        }
    }
}
