import SpanElement from "../../../common/dom/element/SpanElement";
import TableCellLocation from "../../../common/struct/TableCellLocation";
import AbstractDomElement from "../../../common/dom/AbstractDomElement";
import SizeInfo from "../../../common/struct/SizeInfo";
import DomElement from "../../../common/dom/element/DomElement";
import TextEditor from "../../../common/component/control/TextEditor";
import DownArrow from "../../../common/component/control/DownArrow";
import AbstractControl from "../../../common/component/control/AbstractControl";
import Span from "../../../common/component/control/Span";
import EXCEL from "../base/EXCEL";
import ITagNode from "../../../common/xml/node/ITagNode";
import BaseConstants from "../../base/BaseConstants";
import ExcelConstants from "../base/ExcelConstants";

export default class ExcelTableLayout extends AbstractDomElement {

    private tBody!: TBody;
    private thead!: tHead;
    private tColgroup!: tColgroup;

    constructor(widths: number[], lineWidth: number = 1,create :boolean) {
        super();
        if (create){
            this.init(widths, lineWidth);
        }
    }

    init(widths: number[], lineWidth: number = 1) {
        this.clearChildren();
        this.thead = new tHead(widths, lineWidth);
        this.addChild(this.thead);
        this.tColgroup = new tColgroup(widths, lineWidth);
        this.addChild(this.tColgroup);
        this.tBody = new TBody(widths, lineWidth);
        this.addChild(this.tBody);

    }

    protected createEl(tag: string): HTMLElement {
        return document.createElement("table");
    }

    initRows(heights: number[]): void {
        for (let i = 0; i < heights.length; i++) {
            this.addRow(heights[i], i, EXCEL.D_ROW_Type);
        }

    }

    resetRowsTop(): void {
        this.tBody.resetRowsTop();
    }

    setCellElement(row: number, col: number, element: AbstractDomElement): void {
        var cell = this.tBody.getRow(row).getCell(col);
        cell.addChild(element);
    }

    getCell(cellLocation: TableCellLocation): Cell {
        return this.tBody.getRow(cellLocation.getRow()).getCell(cellLocation.getCol());
    }

    getCellAt(row: number, col: number): Cell {
        return this.tBody.getRow(row).getCell(col);
    }

    getThead(): tHead {
        return this.thead;
    }

    getColgroupAt(col: number): any {
        return this.tColgroup.getChildAt(col);
    }

    getColgroups(): any {
        return this.tColgroup.getChildren();
    }

    getTheadThAt(col: number): any {
        return this.thead.getRow(0).getCell(col + 1);
    }

    getBodyThAt(row: number): any {
        return this.tBody.getRow(row).getTh();
    }

    addRow(height: number, index: number, type: String): Row {
        return this.tBody.addRow(height, index, type);
    }

    getLastColumnIndex(): number {
        return this.tBody.getLastColumnIndex();
    }

    getLastRowIndex(): number {
        return this.tBody.getLastRowIndex();
    }

    hitCell(x: number, y: number): TableCellLocation | null {
        this.resetRowsTop();
        var hitRowIndex = -1;
        for (var rowIndex = 0; rowIndex < this.tBody.getRowCount(); rowIndex++) {
            var row = this.tBody.getRow(rowIndex);
            if (y > row.getTop() && y < row.getTop() + row.getHeight()) {
                hitRowIndex = rowIndex;
                break;
            }
        }

        var hitColIndex = -1;
      this.thead.resetColumnsLefts(this.thead.getChildren()[0].getChildren())
        var newColumns: any[] = new Array();
        var columns = this.thead.getColumns();
        for (let i = 1; i < columns.length; i++) {
            newColumns.push(columns[i]);
        }
        for (var colIndex = 0; colIndex < newColumns.length; colIndex++) {
            var column = newColumns[colIndex];
            if (x > column.getLeft() && x < column.getLeft() + column.getWidth()) {
                hitColIndex = colIndex;
                break;
            }
        }


        /*   for (var colIndex = 0; colIndex < this.tBody.getColumnCount(); colIndex++) {
               var column = this.tBody.getColumn(colIndex);
               if (x > column.getLeft() && x < column.getLeft() + column.getWidth()) {
                   hitColIndex = colIndex;
                   break;
               }
           }*/

        if (hitRowIndex == -1 || hitColIndex == -1) return null;
        return new TableCellLocation(hitRowIndex, hitColIndex);
    }

    hitCellByEl(el: HTMLElement): TableCellLocation | null {
        this.resetRowsTop();
        for (var rowIndex = 0; rowIndex < this.tBody.getRowCount(); rowIndex++) {
            var row = this.tBody.getRow(rowIndex);
            for (var colIndex = 0; colIndex < this.tBody.getColumnCount(); colIndex++) {
                if (row.getCell(colIndex).getEl() == el) {
                    return new TableCellLocation(rowIndex, colIndex);
                }
            }
        }
        return null;
    }

    /**
     * 某列的左边线位置
     * @param colIndex
     * @returns
     */
    getLeft(colIndex: number): number {
        return this.tBody.getColumn(colIndex).getLeft();
    }

    /**
     * 某列的右边线位置
     * @param colIndex
     * @returns
     */
    getRight(colIndex: number): number {
        var column = this.tBody.getColumn(colIndex);
        return column.getLeft() + column.getWidth();
    }

    getColumnWidth(colIndex: number): number {
        var column = this.tBody.getColumn(colIndex);
        return column.getWidth();
    }

    /**
     * 某行的上边线位置
     *
     * @param rowIndex
     * @returns
     */
    getTop(rowIndex: number): number {
        this.resetRowsTop();
        return this.tBody.getRow(rowIndex).getTop();
    }

    /**
     * 某行的下边线位置
     * @param rowIndex
     * @returns
     */
    getBottom(rowIndex: number): number {
        this.resetRowsTop();
        var row = this.tBody.getRow(rowIndex);
        return row.getTop() + row.getHeight();
    }

    getRowHeight(rowIndex: number): number {
        var row = this.tBody.getRow(rowIndex);
        return row.getHeight();
    }

    getHeightBetween(row1: number, row2: number) {
        if (row1 > row2) {
            // 交换行序号
            var tmp = row1;
            row1 = row2;
            row1 = tmp;
        }
        this.resetRowsTop();
        return this.getBottom(row2) - this.getTop(row1);
    }

    getWidth(): number {
        var lastCol = this.tBody.getLastColumnIndex();
        return this.getRight(lastCol);
    }

    getHeight(): number {
        this.resetRowsTop();
        var lastRow = this.tBody.getRowCount() - 1;
        return this.getBottom(lastRow);
    }

    isMarkedCell(row: number, col: number): boolean {
        var cell = this.tBody.getRow(row).getCell(col);
        return cell.isMarked();
    }

    markRange(startRow: number, endRow: number) {
        var endCol = this.getLastColumnIndex();
        for (var rowIndex = startRow; rowIndex <= endRow; rowIndex++) {
            var row = this.tBody.getRow(rowIndex);
            for (var colIndex = 0; colIndex <= endCol; colIndex++) {
                var cell = row.getCell(colIndex);
                cell.setMarked(true);
            }
        }
    }

    markCell(row: number, col: number): void {
        this.tBody.getRow(row).getCell(col).setMarked(true);
    }

    clearMarked(): void {
        var endRow = this.getLastRowIndex();
        var endCol = this.getLastColumnIndex();
        for (var rowIndex = 0; rowIndex <= endRow; rowIndex++) {
            var row = this.tBody.getRow(rowIndex);
            for (var colIndex = 0; colIndex <= endCol; colIndex++) {
                var cell = row.getCell(colIndex);
                cell.setMarked(false);
            }
        }
    }
}

class TBody extends AbstractDomElement {

    private lineWidth = 1;

    private columns: Column[] = [];

    constructor(widths: number[], lineWidth: number) {
        super();
        this.lineWidth = lineWidth;
        for (let width of widths) {
            var column = new Column(width);
            this.columns.push(column);
        }
        this.resetColumnsLeft();
        // var row = this.addRow(0.5);
        // row.addClass('first');
        // this.initFirstRow(row);
    }

    resetColumnsLeft() {
        var left = 0;
        for (let i = 0; i < this.columns.length; i++) {
            left += this.lineWidth;
            this.columns[i].setLeft(left);
            left += this.columns[i].getWidth();
        }
    }

    resetRowsTop() {
        var top = 0;
        for (var rowIndex = 0; rowIndex < this.getRowCount(); rowIndex++) {
            var row = this.getRow(rowIndex);
            row.setTop(top);
            top += row.getHeight();
        }
    }

    private initFirstRow(row: Row) {
        for (var i = 0; i < this.columns.length; i++) {
            var span = new SpanElement("");
            span.setStyleWidth(SizeInfo.valueOfPX(this.columns[i].getWidth()));
            span.setStyle('display', 'block');
            var cell = row.getCell(i);
            cell.setStyleWidth(SizeInfo.valueOfPX(this.columns[i].getWidth()));
            cell.addChild(span);
        }
    }

    protected createEl(tag: string): HTMLElement {
        return document.createElement("tbody");
    }

    addRow(height: number, index: number, type: String): Row {
        var row = new Row(height);
        var block = new Block();
        block.setText(String(index + 1));
        block.setAttr("rowindex", String(index))
        block.addClass("excel-th");
        block.setStyle("position", "relative");
        let rowspan = new Span("");//行拖拽
        rowspan.addClass("excel-dragRow");
        rowspan.setStyle("font-size", "0");
        block.addChild(rowspan.getDomElement());
        row.appendChild(block);
        if (type != "Fix") {
            if (type == "Detail") {
                block.setStyle("background", 'url("' + EXCEL.PREFIX + '/css/img/detail_x8.png") left 15px center no-repeat');

            } else if (type == "Head") {
                block.setStyle("background", 'url("' + EXCEL.PREFIX + '/css/img/head_x8.png") left 15px center no-repeat');

            } else if (type == "Group") {
                block.setStyle("background", 'url("' + EXCEL.PREFIX + '/css/img/group_x8.png") left 15px center no-repeat');

            } else if (type == "DetailHead") {
                block.setStyle("background", 'url("' + EXCEL.PREFIX + '/css/img/detailhead_x8.png") left 15px center no-repeat');

            }
        }
        for (let i = 0; i < this.columns.length; i++) {
            var cell = new Cell(null);
            row.addCell(cell);
        }
        this.addChild(row);
        return row;
    }

    /**
     * 获取行
     * @param index
     * @returns
     */
    getRow(index: number): Row {
        return <Row>this.getChildAt(index);
    }

    getColumns(): Column[] {
        return this.columns;
    }

    /**
     * 返回行数
     * @returns
     */
    getRowCount(): number {
        return this.getChildCount();
    }

    getColumn(index: number): Column {
        return this.columns[index];
    }

    getColumnCount(): number {
        return this.columns.length;
    }

    getLastColumnIndex(): number {
        return this.getColumnCount() - 1;
    }

    getLastRowIndex(): number {
        return this.getRowCount() - 1;
    }
}

class tHead extends AbstractDomElement {

    private lineWidth = 1;

    private columns: Column[] = [];

    constructor(widths: number[], lineWidth: number) {
        super();
        this.lineWidth = lineWidth;

        var columnTh = new Column(EXCEL.D_TH_COLUMN_HEIGHT);
        this.columns.push(columnTh);
        for (let i = 1; i < widths.length; i++) {
            var column = new Column(widths[i]);
            this.columns.push(column);
        }
        this.columns.push(new Column(EXCEL.D_COLUMN_WIDTH));//是否添加
        this.resetColumnsLeft();
        var row = this.addRow(EXCEL.D_TH_COLUMN_WIDTH);
        row.addClass('first');
        this.initFirstRow(row);
    }

    getColumns(): Column[] {
        return this.columns;
    }

    setColumns(columns: Column[]): void {
        this.columns = columns;
    }

    resetColumnsLeft() {
        var left = 0;
        for (let column of this.columns) {
            left += this.lineWidth;
            column.setLeft(left);
            left += column.getWidth();
        }
    }

    resetColumnsLefts(newColumns: any[]) {
        var left = 0;
        for (let i = 0; i < newColumns.length; i++) {
            var never = newColumns[i];
            var column = this.columns[i];
            left += this.lineWidth;
            column.setLeft(left);
            var offsetWidth = never.getEl().offsetWidth;
            column.setWidth(offsetWidth);
            left += column.getWidth();

        }
    }

    resetRowsTop() {
        var top = 0;
        for (var rowIndex = 0; rowIndex < this.getRowCount(); rowIndex++) {
            var row = this.getRow(rowIndex);
            row.setTop(top);
            top += row.getHeight();
        }
    }

    private initFirstRow(row: Row) {
        for (var i = 1; i < this.columns.length; i++) {
            var code;
            if (i - 1 - 26 >= 0) {
                code = String.fromCharCode(65) + String.fromCharCode(i - 1 - 26 + 65)
            } else {
                code = String.fromCharCode(65 + i - 1);
            }
            var cell = row.getCell(i);
            cell.setStyle("position", "relative");

            cell.setText(code);
            cell.setStyleWidth(SizeInfo.valueOfPX(this.columns[i].getWidth()));
            cell.setAttr("colindex", String(i));
            let columnSpan = new Span("");//行拖拽
            columnSpan.addClass("excel-dragColumn");
            columnSpan.setStyle("font-size", "0");
            cell.addChild(columnSpan.getDomElement());
        }
    }

    protected createEl(tag: string): HTMLElement {
        return document.createElement("thead");
    }

    addRow(height: number): Row {
        var row = new Row(height);
        var block = new Block();
        block.setStyleWidth(SizeInfo.valueOfPX(60));
        block.setStyle("background-color", "#F4F4F4");
        row.addTh(block);
        for (let i = 1; i < this.columns.length; i++) {
            var cell = new Block();
            row.addTh(cell);

        }
        this.addChild(row);
        return row;
    }

    /**
     * 获取行
     * @param index
     * @returns
     */
    getRow(index: number): Row {
        return <Row>this.getChildAt(index);
    }

    /**
     * 返回行数
     * @returns
     */
    getRowCount(): number {
        return this.getChildCount();
    }

    getColumn(index: number): Column {
        return this.columns[index];
    }

    getColumnCount(): number {
        return this.columns.length;
    }

    getLastColumnIndex(): number {
        return this.getColumnCount() - 1;
    }

    getLastRowIndex(): number {
        return this.getRowCount() - 1;
    }
}

class tColgroup extends AbstractDomElement {

    private lineWidth = 1;

    private columns: Column[] = [];

    constructor(widths: number[], lineWidth: number) {
        super();
        this.lineWidth = lineWidth;
        var col = new Col();
        //  col.setStyleWidth(SizeInfo.valueOfPX(50));
        this.addChild(col)

        col.addClass('colgroup');
        for (let i = 0; i < widths.length; i++) {
            var col1 = new Col();
            //   col1.setStyleWidth(SizeInfo.valueOfPX(50));
            this.addChild(col1)
        }
        for (let width of widths) {
            var column = new Column(width);
            this.columns.push(column);

        }
        // this.resetColumnsLeft();

    }

    resetColumnsLeft() {
        var left = 0;
        for (let column of this.columns) {
            left += this.lineWidth;
            column.setLeft(left);
            left += column.getWidth();
        }
    }

    resetRowsTop() {
        var top = 0;
        for (var rowIndex = 0; rowIndex < this.getRowCount(); rowIndex++) {
            var row = this.getRow(rowIndex);
            row.setTop(top);
            top += row.getHeight();
        }
    }

    protected createEl(tag: string): HTMLElement {
        return document.createElement("colgroup");
    }

    /**
     * 获取行
     * @param index
     * @returns
     */
    getRow(index: number): Row {
        return <Row>this.getChildAt(index);
    }

    /**
     * 返回行数
     * @returns
     */
    getRowCount(): number {
        return this.getChildCount();
    }

    getColumn(index: number): Column {
        return this.columns[index];
    }

    getColumnCount(): number {
        return this.columns.length;
    }

    getLastColumnIndex(): number {
        return this.getColumnCount() - 1;
    }

    getLastRowIndex(): number {
        return this.getRowCount() - 1;
    }
}

class Column {
    private left: number = 0;

    private width: number;

    constructor(width: number) {
        this.width = width;
    }

    setLeft(left: number) {
        this.left = left;
    }

    getLeft() {
        return this.left;
    }

    getWidth(): number {
        return this.width;
    }

    setWidth(width: number) {
        this.width = width;
    }

}

class Row extends AbstractDomElement {

    private top: number = 0;

    private th: any

    constructor(height: number) {
        super();
        //this.height = height;
        if (height > 0) this.setStyleHeight(SizeInfo.valueOfPX(height));
    }

    protected createEl(tag: string): HTMLElement {
        return document.createElement("tr");
    }

    getTh(): any {
        return this.th;
    }

    setTop(top: number) {
        this.top = top;
    }

    getTop() {
        return this.top;
    }

    getHeight(): number {
        return this.getEl().clientHeight; // 此处不用配置中的行height，是因为erp中有多个控件放在一个grid cell中的用法，会将row高度撑大
    }

    addCell(cell: Cell) {
        this.addChild(cell);
    }

    addTh(th: Block) {
        this.addChild(th);
    }

    appendChild(th: Block) {
        this.th = th;
        this.mountChild(th);
    }

    getCell(col: number): Cell {
        return <Cell>this.getChildAt(col);
    }

    setCellValue(col: number, value: any): void {
        var cell = this.getCell(col);
        cell.setText(value);
    }

    setCellElement(col: number, element: AbstractDomElement): Cell {
        var cell = this.getCell(col);
        cell.setAttr("rowindex", String(element.getAttr("rowindex")));
        cell.setAttr("colindex", String(element.getAttr("colindex")));
        cell.setContent(element);
        return cell;
    }
}

class Cell extends AbstractDomElement {

    private marked: boolean = false;

    constructor(el: HTMLElement | null) {
        super(el);
    }

    setValue(value: string): void {
        this.getContent()?.setText(value);

    }

    appendChild(textEditor: TextEditor) {
        this.getEl().appendChild(textEditor.getEl());
    }

    protected createEl(tag: string): HTMLElement {
        return document.createElement("td");
    }

    setContent(element: AbstractDomElement): void {
        this.addChild(element);
    }

    getContent(): AbstractDomElement | null {
        if (this.getChildCount() == 0) return null;
        return this.getChildAt(0);
    }

    setMarked(b: boolean) {
        this.marked = b;
        //b ? this.addClass('cell_marked') : this.removeClass('cell_marked');
    }

    isMarked(): boolean {
        return this.marked;
    }
}

class Block extends AbstractDomElement {

    private marked: boolean = false;

    protected createEl(tag: string): HTMLElement {
        return document.createElement("th");
    }

    setContent(element: AbstractDomElement): void {
        this.addChild(element);
    }

    getContent(): AbstractDomElement | null {
        if (this.getChildCount() == 0) return null;
        return this.getChildAt(0);
    }

    setMarked(b: boolean) {
        this.marked = b;
        //b ? this.addClass('cell_marked') : this.removeClass('cell_marked');
    }

    isMarked(): boolean {
        return this.marked;
    }
}

class Col extends AbstractDomElement {

    private marked: boolean = false;

    protected createEl(tag: string): HTMLElement {
        return document.createElement("col");
    }

    setContent(element: AbstractDomElement): void {
        this.addChild(element);
    }

    getContent(): AbstractDomElement | null {
        if (this.getChildCount() == 0) return null;
        return this.getChildAt(0);
    }

    setMarked(b: boolean) {
        this.marked = b;
        //b ? this.addClass('cell_marked') : this.removeClass('cell_marked');
    }

    isMarked(): boolean {
        return this.marked;
    }
}


/****************************
 *
 *

 .designingForm table.layout > tbody > tr > td {
    border-bottom: 1px dashed rgb(224, 224, 224);
    border-right: 1px dashed rgb(224, 224, 224);
}

 tr {
    display: table-row;
    vertical-align: inherit;
    border-color: inherit;
}
 .ui-glp .layout {
    border-collapse: separate;
    border-spacing: 0px;
    table-layout: fixed;
}

 .designingForm table.layout {
    border-top: 1px dashed rgb(224, 224, 224);
    border-left: 1px dashed rgb(224, 224, 224);
}
 .ui-glp .layout {
    border-collapse: separate;
    border-spacing: 0px;
    table-layout: fixed;
}

 */