import ArrayList from "../../../../../common/struct/ArrayList";
import { EToggleStatus, ETreeNodeType } from "../../../../enum/Enums";
import Events from "../../../../event/Events";
import UserEvent from "../../../../event/UserEvent";
import AbstractModel from "../../../../model/AbstractModel";
import { IItemData } from "../../../../struct/DataStruct";
import TreeModel from "./TreeModel";

/**
 * 树节点模型
 * 
 * @author chenbinbin
 */
export default class TreeItemModel extends AbstractModel {

    private clsIcon: string;
    
    private treeModel: TreeModel;

    private nodeType: ETreeNodeType = ETreeNodeType.Leaf;

    private toggleStatus: EToggleStatus = EToggleStatus.Collapase;

    private children: ArrayList<TreeItemModel> | null = null;

    private parent: TreeItemModel | null = null;

    private bSelected: boolean = false;

    private itemData: IItemData;

    constructor(itemData: IItemData, clsIcon: string, treeModel: TreeModel, nodeType: ETreeNodeType = ETreeNodeType.Leaf) {
        super();
        this.itemData = itemData;
        this.clsIcon = clsIcon;
        this.treeModel = treeModel;
        this.nodeType = nodeType;
    }

    public getTreeModel(): TreeModel {
        return this.treeModel;
    }

    getItemData(): IItemData {
        return this.itemData;
    }

    getData(): string {
        return this.itemData.data;
    }

    getIconClass() {
        return this.clsIcon;
    }

    getCaption(): string {
        return this.itemData.caption;
    }

    setParent(parent: TreeItemModel): void {
        this.parent = parent;
    }

    getParent(): TreeItemModel | null {
        return this.parent;
    }

    addChild(item: TreeItemModel): void {
        if (this.isLeaf()) {
            this.nodeType = ETreeNodeType.Branch;
        } 
        this.ensureChildren().add(item);
        item.setParent(this);
    }

    getChildCount(): number {
        if (this.children == null) return 0;
        return this.children.getSize();
    }

    getChildAt(index: number): TreeItemModel | null {
        if (this.children == null) return null;
        return this.children.getList()[index];
    }

    isLeaf(): boolean {
        return this.nodeType == ETreeNodeType.Leaf;
    }

    getChildren(): TreeItemModel[] {
        if (this.children == null) return [];
        return this.children.getList();
    }

    getNodeType(): ETreeNodeType {
        return this.nodeType;
    }

    setNodeType(nodeType: ETreeNodeType): void {
        this.nodeType = nodeType;
    }

    getToggleStatus(): EToggleStatus {
        return this.toggleStatus;
    }

    setToggleStatus(toggleStatus: EToggleStatus) {
        if (this.toggleStatus != toggleStatus) {
            this.toggleStatus = toggleStatus;
           if (this.isExpand()) {
               this.fireEvent(Events.Tree.EXPAND);
           } else {
               this.fireEvent(Events.Tree.COLLAPASE);
           }
        }
    }

    isCollapase(): boolean {
        return this.toggleStatus == EToggleStatus.Collapase;
    }

    isExpand(): boolean {
        return this.toggleStatus == EToggleStatus.Expand;
    }

    setSelect(b: boolean) {
        this.bSelected = b;
        this.fireEvent(Events.COMMON_SELECT_CHANGE);
    }

    isSelected(): boolean {
        return this.bSelected;
    }

    getLevel(): number {
        let level = 0;
        if (this.parent) {
            level = this.parent.getLevel() + 1;
        }
        return level;
    }

    clearChildren(): void {
        this.children && this.children.clearAll();
    }

    ensureChildren(): ArrayList<TreeItemModel> {
        if (this.children == null) {
            this.children = new ArrayList<TreeItemModel>();
        }
        return this.children;
    }

    private fireEvent(eventKey: string): void {
        this.fireModelEvent(new UserEvent(eventKey, this, this.treeModel));
    }

}