package com.bokesoft.yes.tool;

import com.bokesoft.yes.parser.Item;
import com.bokesoft.yes.parser.SyntaxTree;
import org.apache.commons.lang3.StringUtils;

import java.util.ArrayList;
import java.util.List;

/**
 * 表达式的格式
 */
public class FormulaFormat {
	private final String content;
	private final List<Item> items;
	private final List<Integer> itemIndexes;
	/** 将表达式按Item及其前后缀都分开来的字符串 */
	private final List<String> formats = new ArrayList<String>();
	
	public FormulaFormat(String formula, List<Item> leafItems) {
		this.content = formula;
		this.items = leafItems;
		itemIndexes = new ArrayList<Integer>(leafItems.size());
		length = formula.length();
	}

	public static FormulaFormat parse(String formula, SyntaxTree syntaxTree) {
		List<Item> leafItems = FormulaUtil.getLeafItems(syntaxTree.getRoot());
		FormulaFormat result = new FormulaFormat(formula, leafItems);
		result.parse();
		return result;
	}

	private final int length;
	private int pos = 0;
	private int start = 0;
	private char c;

	/** 是否有忽略字符的标志位 */
	private boolean b_skip = false;
	private boolean c_skip = false;
	private boolean semicolon_skip = false;
	
	private void parse() {
		int index = 0;
		int lastPos = 0;
		int indexIndex = 0;
		while (true) {
			// Skip all backspace and tab
			c = content.charAt(pos);
			++pos;
			do {
				c = skipBlank(c);
				c = skipComments(c);
			} while (this.c_skip || this.b_skip);
			String tokenString = getItemLexValue(items.get(indexIndex));
			if (tokenString.equals("}") && c == ';') {
				do {
					c = skipBlank(c);
					c = skipComments(c);
					c = skipSemicolon(c);
				} while (this.c_skip || this.b_skip || this.semicolon_skip);
			}
			if (tokenString.length() > 0
					&& content.substring(pos - 1, pos + tokenString.length() - 1).equals(tokenString)) {
				formats.add(pos > lastPos + 1 ? content.substring(lastPos, pos - 1) : "");
				formats.add(content.substring(pos - 1, pos + tokenString.length() - 1));
				pos += tokenString.length() - 1;
			} else if (tokenString.equals("<>")
					&& content.substring(pos - 1, pos + tokenString.length() - 1).equals("!=")) { // 考虑<>和!=的通用性
				formats.add(pos > lastPos + 1 ? content.substring(lastPos, pos - 1) : "");
				formats.add(content.substring(pos - 1, pos + tokenString.length() - 1));
				pos += tokenString.length() - 1;
			} else if ((tokenString.equalsIgnoreCase("true") || tokenString.equalsIgnoreCase("false"))
					&& content.substring(pos - 1, pos + tokenString.length() - 1).equalsIgnoreCase(tokenString)) { // 考虑true、false不具体大小写
				formats.add(pos > lastPos + 1 ? content.substring(lastPos, pos - 1) : "");
				formats.add(content.substring(pos - 1, pos + tokenString.length() - 1));
				pos += tokenString.length() - 1;
			} else if (content.substring(pos, pos + tokenString.length()).equals(tokenString)) { // 考虑字符串常量
				formats.add(pos > lastPos + 1 ? content.substring(lastPos, pos - 1) : "");
				formats.add(content.substring(pos - 1, pos + tokenString.length() + 1));
				pos += tokenString.length() + 1;
			} else {
				throw new RuntimeException("ERROR!");
			}
			lastPos = pos;
			indexIndex++;
			itemIndexes.add(index + 2);
			index += 2;
			if (indexIndex == items.size()) {
				formats.add(pos < length ? content.substring(pos, length) : "");
				break;
			}
		}
	}
	
	private static String getItemLexValue(Item item) {
		String fullLexValue = item.getFullLexValue();
		return StringUtils.isBlank(fullLexValue) ? item.getLexValue() : fullLexValue;
	}

    /**
     * 略过空白
     * @param c 起始字符串
     * @return 空白之后的第一个字符
     */
    private char skipBlank(char c) {
        char result = c;
    	this.b_skip = false;
        
        while (c == ' ' || c == '\t' || c == '\n' || c == '\r')
        {
        	this.b_skip = true;
            if (pos < length)
            {
                c = content.charAt(pos);
            }
            else
            {
                return result;
            }
            result = c;
            ++pos;
            ++this.start;
        }
        return result;
    }

	/**
	 * 略过分号
	 * 
	 * @param c
	 * @return
	 */
	private char skipSemicolon(char c) {
		char result = c;
		this.semicolon_skip = false;

		while (c == ';') {
			this.semicolon_skip = true;
			if (pos < length) {
				c = content.charAt(pos);
			} else {
				return result;
			}
			result = c;
			++pos;
			++this.start;
		}
		return result;
	}

	/**
	 * 略过注释
	 * @param c 起始字符
	 * @return 返回注释之后的第一个字符
	 */
    private char skipComments(char c) {
        char result = c;
        this.c_skip = false;
        if (pos < length)
        {
            int oldPos = pos;
            int oldStart = start;
            boolean found = false;
            if (c == '/')
            {
                if (pos < length)
                {
                    c = content.charAt(pos);
                    ++pos;
                    ++this.start;
                    if (c == '/')
                    {
                        // Until next \n
                        if (pos < length)
                        {
                            c = content.charAt(pos);
                            ++pos;
                            ++this.start;
                            while (c != '\n' && pos < length)
                            {
                                if (pos < length)
                                {
                                    c = content.charAt(pos);
                                    ++pos;
                                    ++this.start;
                                }
                            }
                            // Forward to next
                            if (pos < length)
                            {
                                c = content.charAt(pos);
                                ++pos;
                                ++this.start;
                                result = c;
                            } else {
                            	result = '$';
                            }
                            
                            found = true;
                        }
                    }
                    else if (c == '*')
                    {
                        // Until next * and /
                        if (pos < length)
                        {
                            c = content.charAt(pos);
                            ++pos;
                            ++this.start;
                            
                            while ( true ) {
                            	// 找到*，那么匹配/，否则继续
                            	if ( c == '*' ) {
                            		// 如果匹配/，则结束
                            		if ( pos < length && content.charAt(pos) == '/' ) {
                            			++pos;
                                        ++this.start;
                                        found = true;
                                        break;
                            		}
                            	}
                            	if ( pos < length ) {
                            		c = content.charAt(pos);
                                    ++pos;
                                    ++this.start;
                            	} else {
                            		break;
                            	}
                            }
                            
                            if ( found ) {
	                            // Forward to next
	                            if (pos < length)
	                            {
	                                c = content.charAt(pos);
	                                ++pos;
	                                ++this.start;
	                                result = c;
	                            } else {
	                            	result = '$';
	                            }
                            }
                        }
                    }
                }
            }
            if (!found)
            {
                pos = oldPos;
                this.start = oldStart;
            } else {
            	this.c_skip = true;
            }
        }
        return result;
    }

	/**
	 * 取语法树项目前的格式信息（含空格和注释），只限于明细项目
	 * @return
	 */
	public String getPreFormat(Item item) {
		int index = items.indexOf(item);
		if (index < 0) {
			throw new RuntimeException("ERROR!");
		}
		return formats.get(index * 2);
	}
	
	/**
	 * 取最后的格式信息（含空格和注释）
	 * @return
	 */
	public String getLastFormat() {
		return formats.get(formats.size() - 1);
	}
	
	/**
	 * 取语法树项目的原始字符串，只限于明细项目
	 * @param item
	 * @return
	 */
	public String getOrgText(Item item) {
		int index = items.indexOf(item);
		if (index < 0) {
			throw new RuntimeException("ERROR!");
		}
		return formats.get(index * 2 + 1);
	}
	
	@Override
	public String toString() {
		return formats.toString();
	}
}
