package com.bokesoft.yes.design.xml.node;

import com.bokesoft.yes.design.xml.parse.Element;
import com.bokesoft.yes.meta.persist.dom.xml.defaultnode.DefaultNodeDefine;
import com.bokesoft.yigo.meta.bpm.process.attribute.timer.MetaTimerAutoDeny;
import com.bokesoft.yigo.meta.bpm.process.attribute.timer.MetaTimerAutoPass;
import com.bokesoft.yigo.meta.bpm.process.attribute.timer.MetaTimerItem;
import com.bokesoft.yigo.meta.bpm.process.node.MetaAudit;
import com.bokesoft.yigo.meta.bpm.process.perm.MetaEnablePermItem;
import com.bokesoft.yigo.meta.bpm.process.perm.MetaVisiblePermItem;
import com.bokesoft.yigo.meta.common.MetaCustomConditionPara;
import com.bokesoft.yigo.meta.common.MetaExtendCollection;
import com.bokesoft.yigo.meta.common.MetaMacroCollection;
import com.bokesoft.yigo.meta.commondef.MetaStatusCollection;
import com.bokesoft.yigo.meta.dataobject.*;
import com.bokesoft.yigo.meta.setting.MetaSysColumn;
import org.apache.commons.lang3.StringUtils;

import java.util.*;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.Stream;

public class XmlTree {
    /**
     * XML节点的根
     */
    private final TagNode root;

    public XmlTree(TagNode root) {
        this.root = root;
    }

    public TagNode getRoot() {
        return this.root;
    }

    public void addPreComment(List<AbstractNode> comments) {
        root.addPreComment(comments);
    }

    public void addLastComment(List<AbstractNode> comments) {
        root.addLastComment(comments);
    }

    private final List<AbstractNode> allNodeWithCommnet = new Vector<AbstractNode>();
    private HashMap<String, AbstractNode> mapNodes = new HashMap<String, AbstractNode>();
    private final HashMap<String, AbstractNode> relativelyMapNodes = new HashMap<String, AbstractNode>();

    public HashMap<String, AbstractNode> getMapNodes() {
        return mapNodes;
    }

    public void updateMap(boolean isExamination) throws Throwable {
        List<AbstractNode> commentList = new ArrayList<AbstractNode>();
        updateMap(root, commentList, isExamination);
    }

    private void updateMap(TagNode tagNode, List<AbstractNode> commentList) throws Throwable {
        if (Objects.isNull(tagNode)) {
            return;
        }
        tagNode.addPreComment(commentList);
        recordNodeWithComment(commentList, tagNode);
        commentList.clear();
        assertUniqueKey(tagNode);
        if (tagNode.getPrimaryKey() != null) {
            String parentPrimaryKey = "";
            TagNode parent = tagNode.getParent();
            parentPrimaryKey = getParentPrimaryKey(parentPrimaryKey, parent);
            relativelyMapNodes.put(parentPrimaryKey + "->" + tagNode.getPrimaryKey(), tagNode);
            mapNodes.put(tagNode.getPrimaryKey(), tagNode);

        }
        List<AbstractNode> children = tagNode.getChildren();
        AbstractNode node = null;
        AbstractNode tmpTagNode = null;
        int size = children.size();
        for (int n = 0; n < size; n++) {
            node = children.get(n);
            if (node instanceof TagNode) {
                updateMap((TagNode) node, commentList);
                tmpTagNode = node;
            } else if (node instanceof CommentNode) {
                commentList.add(node);
            } else if (node instanceof CDataNode) {
                node.addPreComment(commentList);
                recordNodeWithComment(commentList, node);
                commentList.clear();
                tmpTagNode = node;
            }
        }
        if (tmpTagNode != null) {
            tmpTagNode.addLastComment(commentList);
            recordNodeWithComment(commentList, tmpTagNode);
            commentList.clear();
        }
    }

    private void updateMap(TagNode tagNode, List<AbstractNode> commentList, boolean isExamination) throws Throwable {
        if (isExamination) {
            mapNodes = new HashMap<>();
            updateMap(tagNode, commentList);
        } else {
            if (Objects.isNull(tagNode)) {
                return;
            }
            tagNode.addPreComment(commentList);
            recordNodeWithComment(commentList, tagNode);
            commentList.clear();
            String parentPrimaryKey = "";
            TagNode parent = tagNode.getParent();
            parentPrimaryKey = getParentPrimaryKey(parentPrimaryKey, parent);
            relativelyMapNodes.put(parentPrimaryKey + "->" + tagNode.getPrimaryKey(), tagNode);
            mapNodes.put(tagNode.getPrimaryKey(), tagNode);
            List<AbstractNode> children = tagNode.getChildren();
            AbstractNode node = null;
            AbstractNode tmpTagNode = null;
            int size = children.size();
            for (int n = 0; n < size; n++) {
                node = children.get(n);
                if (node instanceof TagNode) {
                    updateMap((TagNode) node, commentList, isExamination);
                    tmpTagNode = node;
                } else if (node instanceof CommentNode) {
                    commentList.add(node);
                } else if (node instanceof CDataNode) {
                    node.addPreComment(commentList);
                    recordNodeWithComment(commentList, node);
                    commentList.clear();
                    tmpTagNode = node;
                }
            }
            if (tmpTagNode != null) {
                tmpTagNode.addLastComment(commentList);
                recordNodeWithComment(commentList, tmpTagNode);
                commentList.clear();
            }
        }

    }

    private void assertUniqueKey(TagNode tagNode) throws Throwable {
        if (Objects.isNull(tagNode.getPrimaryKey())) {
            return;
        }
        if (StringUtils.equalsAnyIgnoreCase(tagNode.getTagName(), "Item", "Var", "GridRow", "Column", "Field","SourceField", "Entry",
                "EntryItem", "Operation", "OptPermItem", MetaCustomConditionPara.TAG_NAME, "DropdownItem", "ExtraLayout", "Collapsible", "Item",
                MetaEmbedTableCollection.TAG_NAME, MetaPreLoadProcess.TAG_NAME, MetaPostLoadProcess.TAG_NAME, MetaPreSaveProcess.TAG_NAME, MetaPostSaveProcess.TAG_NAME,
                MetaPreDeleteProcess.TAG_NAME, MetaPostDeleteProcess.TAG_NAME, MetaStatusCollection.TAG_NAME, MetaStatusTriggerCollection.TAG_NAME,
                MetaExtendCollection.TAG_NAME, MetaCheckRuleCollection.TAG_NAME, MetaMacroCollection.TAG_NAME, MetaSysColumn.TAG_NAME,"ItemFilter","Filter","FilterValue",
                MetaEnablePermItem.TAG_NAME, MetaVisiblePermItem.TAG_NAME, MetaTimerItem.TAG_NAME, MetaTimerAutoDeny.TAG_NAME, MetaTimerAutoPass.TAG_NAME,
                MetaAudit.TAG_NAME)) {
            return;
        }

        // 不能重复key的组件
        Set<String> panelType = Stream.of("Grid", "SubDetail", "GridLayoutPanel", "FlexFlowLayoutPanel",
                        "SplitPanel", "TabPanel", "FlowLayoutPanel", "ColumnLayoutPanel", "ListView", "Embed")
                .collect(Collectors.toCollection(HashSet::new));
        Set<String> intersection = new HashSet<>();
        if (panelType.contains(tagNode.getTagName())) {
            String key = tagNode.getAttributes().get("Key");
            Set<String> collect = panelType.stream()
                    .filter(m -> !StringUtils.equals(tagNode.getTagName(), m)) // 排除自己的类型
                    .map(m -> m + "@" + key) // 拼接成primaryKey
                    .collect(Collectors.toSet()); // key与不同组件的组合
            intersection.clear();
            intersection.addAll(collect);
            intersection.retainAll(mapNodes.keySet());// 与mapNodes的交集
        }
        // 当mapNodes存在相同key 或者 存在某些组件key相等时 报错
        boolean flag = mapNodes.containsKey(tagNode.getPrimaryKey()) || intersection.size() != 0;
        if (flag) {
            throw new Throwable("存在重复Key:" + tagNode.getPrimaryKey());
        }
    }

    /**
     * 记录含有注释的节点
     *
     * @param commnetList
     * @param node
     */
    private void recordNodeWithComment(List<AbstractNode> commnetList, AbstractNode node) {
        if (commnetList.size() > 0 && !allNodeWithCommnet.contains(node)) {
            allNodeWithCommnet.add(node);
        }
    }

    /**
     * 获取整个表单的CData
     *
     * @return
     */
    public List<CDataNode> findCDataNodes() {
        List<CDataNode> list = new ArrayList<CDataNode>();
        relativelyMapNodes.forEach((a, Node) -> {
            if (Node instanceof CDataNode) {
                list.add((CDataNode) Node);
            } else if (Node instanceof TagNode) {
                ((TagNode) Node).findCDataNodesByTagName(list);
            }
        });
        return list;
    }

    /**
     * 获取父节点Key
     *
     * @param parent
     * @param parentPrimaryKey
     * @return
     */
    private String getParentPrimaryKey(String parentPrimaryKey, TagNode parent) {
        if (parent != null) {
            String attrKey = DefaultNodeDefine.getInstance().getPrimaryKey(parent.getTagName());
            String primaryValue = parent.getAttributes().get(attrKey);
            if (StringUtils.isEmpty(primaryValue)) {
                parentPrimaryKey = parentPrimaryKey.concat(getParentPrimaryKey(parentPrimaryKey, parent.getParent()));
            } else {
                parentPrimaryKey = parent.getPrimaryKey();
            }
        }
        return parentPrimaryKey;
    }

    /**
     * 获取文档中的所有注释
     *
     * @return
     */
    public List<AbstractNode> getNodesWithComment() {
        return allNodeWithCommnet;
    }

    public TagNode getTagNode(String key) {
        return (TagNode) mapNodes.get(key);
    }

    public TagNode getTagNodeByRelatively(String key) {
        return (TagNode) relativelyMapNodes.get(key);
    }

    public boolean containKey(String key) {
        return mapNodes.containsKey(key);
    }

    /**
     * 老的xml字符串
     */
    private String orgXml;

    public void setOrgXml(String xml) {
        this.orgXml = xml;
    }

    public String getOrgXml() {
        return orgXml;
    }

    private List<Integer> lines = null;

    private void initLines() {
        if (Objects.isNull(lines)) {
            lines = new ArrayList<Integer>();
            lines.add(0);
            int pos = -1;
            while (true) {
                pos = orgXml.indexOf("\n", pos + 1);
                if (pos >= 0) {
                    lines.add(pos);
                } else {
                    break;
                }
            }
        }
    }

    public int getLineCount() {
        initLines();
        return lines.size();
    }

    public int getLineByIndex(int index) {
        initLines();
        int left = 0;
        int size = lines.size();
        int right = size - 1;
        int mid;
        while (left <= right) {
            mid = (left + right) >> 1;
            if (lines.get(mid) < index) {
                if (mid < size && lines.get(mid + 1) > index) {
                    return mid;
                }
                left = mid + 1;
            } else if (lines.get(mid) > index) {
                if (mid > 0 && lines.get(mid - 1) <= index) {
                    return mid - 1;
                }
                right = mid - 1;
            } else {
                return mid;
            }
        }
        return size;
    }

    public String getLineString(int line) {
        initLines();
        if ((line + 1) <= lines.size() - 1) {
            return orgXml.substring(lines.get(line) + 1, lines.get(line + 1) + 1);
        }
        return null;

    }

    public int getLineStartPos(int line) {
        initLines();
        //防止数组越界
        if (line > lines.size() - 1) {
            line = lines.size() - 1;
        }
        return lines.get(line) + 1;
    }

    public String getNodeString(int nodeStratLine) {
        int startPos = getLineStartPos(nodeStratLine);
        int nodeEndLine = getNodeEndLine(nodeStratLine);
        int endPos = getLineStartPos(nodeEndLine + 1);
        return orgXml.substring(startPos, endPos).trim();
    }

    public int getNodeEndLine(int nodeStartLine) {
        String lineString = getLineString(nodeStartLine).trim();
        if (lineString.endsWith("/>")) {
            return nodeStartLine;
        } else {
            String tagName = getSubString(lineString, "<([A-Za-z_]\\w*)[\\s>]");
            Stack<String> stack = new Stack<String>();
            stack.push(tagName);
            int endLine = nodeStartLine + 1;
            boolean isCDATA = false;
            boolean isComment = false;
            boolean isDataBingContent = false;
            while (endLine < getLineCount()) {
                if (getLineString(endLine) != null) {
                    String nextLineString = getLineString(endLine).trim();
                    if (StringUtils.isEmpty(nextLineString)) {
                        //如果包含空行直接不处理
                    } else if (isCDATA) {
                        if (nextLineString.endsWith("]]>")) {
                            isCDATA = false;
                        }
                    } else if (nextLineString.startsWith("<![CDATA[")) {
                        if (!nextLineString.endsWith("]]>")) {
                            isCDATA = true;
                        }
                    } else if (isComment) {
                        if (nextLineString.endsWith("-->")) {
                            isComment = false;
                        }
                    } else if (nextLineString.startsWith("<!--")) {
                        if (!nextLineString.endsWith("-->")) {
                            isComment = true;
                        }
                    } else if (isDataBingContent) {
                        if (nextLineString.endsWith("/>") || nextLineString.endsWith("</DataBinding>")|| nextLineString.endsWith("</DataBinding>\n")) {
                            isDataBingContent = false;
                        }
                    } else if (nextLineString.startsWith("<DataBinding")) {
                        if (!nextLineString.endsWith("/>") || !nextLineString.endsWith(">")) {
                            isDataBingContent = true;
                        }
                        if (nextLineString.endsWith("</DataBinding>")){
                            isDataBingContent = false;
                        }
                    } else if (nextLineString.startsWith("</")) {
                        String closeTagName = getSubString(nextLineString, "</([A-Za-z_]\\w*)>");
                        if (!closeTagName.equals(stack.peek())) {
                            throw new RuntimeException("XML节点关系不对。");
                        }
                        stack.pop();
                        if (stack.isEmpty()) {
                            break;
                        }
                    } else if (!nextLineString.endsWith("/>")) {
                        String nextTagName = getSubString(nextLineString, "<([A-Za-z_]\\w*)[\\s>]");
                        stack.push(nextTagName);
                    }
                    endLine++;
                } else {
                    break;
                }
            }
            return endLine;
        }

    }

    /**
     * 根据范围取所有的明细节点
     *
     * @param startPos
     * @param endPos
     * @return
     */
    public List<AbstractNode> getDetailNodeByLine(int startPos, int endPos) {
        List<AbstractNode> result = new ArrayList<AbstractNode>();
        getDetailNodeByLine(startPos, endPos, root, result);
        return result;
    }

    private void getDetailNodeByLine(int startPos, int endPos, AbstractNode node, List<AbstractNode> detailNodes) {
        Element element = node.getElement();
        int nodeStartPos = element.getBgIndex();
        List<AbstractNode> children = node instanceof TagNode ? ((TagNode) node).getChildren() : null;
        int size = Objects.isNull(children) ? 0 : children.size();
        int nodeEndPos = size == 0 ? element.getBgIndex() : children.get(size - 1).getElement().getEndIndex();
        if (!((nodeStartPos < startPos && nodeEndPos > endPos) || (nodeStartPos >= startPos && nodeStartPos <= endPos)
                || (nodeEndPos >= startPos && nodeEndPos <= endPos))) {
            return;
        }
        if ((nodeStartPos < startPos && nodeEndPos > endPos && size == 0) || (nodeStartPos >= startPos && nodeStartPos <= endPos)
                || (nodeEndPos >= startPos && nodeEndPos <= endPos)) {
            detailNodes.add(node);
        }
        for (int i = 0; i < size; i++) {
            AbstractNode child = children.get(i);
            getDetailNodeByLine(startPos, endPos, child, detailNodes);
        }
    }

    /**
     * 根据正则表达式取字符串中的值
     *
     * @param s       <OperationCollection>
     * @param pattern <([A-Za-z_]\w*)\s
     * @return
     */
    private static String getSubString(String s, String pattern) {
        Pattern r = Pattern.compile(pattern);
        Matcher m = r.matcher(s);
        if (m.find()) {
            return m.group(1);
        }
        throw new RuntimeException("字符串" + s + "中没有找到" + pattern);
    }

}