package com.bokesoft.yes.design.vest;

import com.bokesoft.yes.design.constant.ConstantUtil;
import com.bokesoft.yes.helper.XmlHelper;
import com.bokesoft.yes.meta.persist.dom.xml.XmlParser;
import com.bokesoft.yes.meta.persist.dom.xml.node.AbstractNode;
import com.bokesoft.yes.meta.persist.dom.xml.node.TagNode;
import com.bokesoft.yes.meta.persist.dom.xml.node.XmlTree;
import com.bokesoft.yigo.common.util.FileUtil;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;

import java.io.IOException;
import java.util.*;

public class DefaultVestCreator {
    private final String vestKey;
    private final String vestCaption;
    private final String sourceResource;

    public DefaultVestCreator(String vestKey, String vestCaption, String sourceResource) {
        this.vestKey = vestKey;
        this.vestCaption = vestCaption;
        this.sourceResource = sourceResource;
    }

    public String createVestXml(String sourceKey,boolean mergeToSource) throws IOException {
        XmlTree xmlTree = XmlParser.parse(FileUtil.File2String(this.sourceResource), true);
        TagNode rootNode = xmlTree.getRoot();

        TagNode vestRootNode = this.createVestNode(rootNode);
        Objects.requireNonNull(vestRootNode);

        vestRootNode.setAttribute(ConstantUtil.KEY, this.vestKey);
        vestRootNode.setAttribute(ConstantUtil.CAPTION, this.vestCaption);
        vestRootNode.setAttribute(ConstantUtil.EXTEND, sourceKey);
        if (mergeToSource){
            vestRootNode.setAttribute("MergeToSource", String.valueOf(mergeToSource));

        }
        return XmlHelper.format(vestRootNode, 0);
    }

    private TagNode createVestNode(TagNode node) {
        // 1.节点及其所有子节点都不会出现在马甲中，直接返回null
        if (node.getTagName().endsWith("Collection")
                ||StringUtils.equalsAny(node.getTagName(), "SplitSize"
                , "Var"
                , "Item"
                , "ItemFilter"
                ,"CustomConditionPara"
        )) {
            return null;
        }

        // 2.布局节点
        TagNode layoutNode = this.createVestLayoutNode(node);
        if (Objects.nonNull(layoutNode)) {
            return layoutNode;
        }

        // 3.面板节点
        TagNode panelNode = this.createVestPanelNode(node);
        if (Objects.nonNull(panelNode)) {
            return panelNode;
        }

        // 4.表单组件节点
        return this.createVestFormNode(node);
    }

    private TagNode createVestLayoutNode(TagNode node) {
        Pair<String, String> pair = this.getPrimaryAttr(node);
        if (Objects.nonNull(pair)) {
            return null;
        }

        TagNode parentNode = node.getParent();
        if (Objects.isNull(parentNode)) {
            return null;
        }

        if (!StringUtils.endsWith(parentNode.getTagName(), "Panel")) {
            return null;
        }

        return (TagNode) node.clone();
    }

    private TagNode createVestPanelNode(TagNode node) {
        if (!StringUtils.endsWith(node.getTagName(), "Panel")) {
            return null;
        }
        Pair<String, String> pair = this.getPrimaryAttr(node);
        TagNode vestNode = new TagNode(node.getTagName());
        if (Objects.nonNull(pair)) {
            vestNode = new TagNode(node.getTagName());
            final String[] needCopy2VestAttrs = {"X", "Y", "XSpan", "YSpan"};
            vestNode.setAttribute(pair.getLeft(), pair.getValue());
            VestUtil.dealSpecialAttrs(vestNode, node, needCopy2VestAttrs);
        }
        List<TagNode> childTagNodes = this.createVestChildNode(node);
        for (TagNode next : childTagNodes) {
            vestNode.addNode(next);
        }
        return vestNode;
    }

    private TagNode createVestFormNode(TagNode node) {
        TagNode vestNode = null;

        Pair<String, String> pair = this.getPrimaryAttr(node);
        if (Objects.nonNull(pair)) {
            vestNode = new TagNode(node.getTagName());
            final String[] needCopy2VestAttrs = {"X", "Y", "XSpan", "YSpan"};
            vestNode.setAttribute(pair.getLeft(), pair.getValue());
            VestUtil.dealSpecialAttrs(vestNode, node, needCopy2VestAttrs);
            VestUtil.dealOptRefKey(vestNode, node);
        }

        List<TagNode> childTagNodes = this.createVestChildNode(node);
        for (TagNode next : childTagNodes) {
            if (Objects.isNull(vestNode)) {
                vestNode = new TagNode(node.getTagName());
            }
            vestNode.addNode(next);
        }
        return vestNode;
    }

    private List<TagNode> createVestChildNode(TagNode node) {
        List<AbstractNode> childNodeList = node.getChildren();
        if (CollectionUtils.isEmpty(childNodeList)) {
            return Collections.emptyList();
        }

        List<TagNode> vestChildNodeList = new ArrayList<>();
        for (AbstractNode childNode : childNodeList) {
            boolean b = childNode instanceof TagNode;
            if (!b) {
                continue;
            }

            TagNode vestChildNode = this.createVestNode((TagNode) childNode);
            if (Objects.nonNull(vestChildNode)) {
                vestChildNodeList.add(vestChildNode);
            }
        }
        return vestChildNodeList;
    }

    private Pair<String, String> getPrimaryAttr(TagNode node) {
        String attrName = node.getPrimaryAttrName();
        if (StringUtils.isBlank(attrName)) {
            return null;
        }
        Map<String, String> attrMap = node.getAttributes();
        String attrVal = attrMap.get(attrName);
        if (StringUtils.isBlank(attrVal)) {
            return null;
        }
        return Pair.of(attrName, attrVal);
    }
}
