package com.bokesoft.yes.design.search;

import java.io.File;
import java.util.List;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.commons.io.FileUtils;

import com.bokesoft.yes.design.io.LoadFileTree;
import com.bokesoft.yes.design.search.text.FileMatchResource;
import com.bokesoft.yes.design.search.text.TextSearchMatchAccess;
import com.bokesoft.yes.design.xml.parse.Element;
import com.bokesoft.yes.design.xml.parse.Util;

/**
 * 文本搜索
 */
public class TextSearchVisitor {

	private final TextSearchRequestor fCollector;

	/** 匹配 */
	private final Pattern fSearchPattern;

	public TextSearchVisitor(TextSearchRequestor collector, Pattern searchPattern) {
		fCollector = collector;
		fSearchPattern = searchPattern;
	}

	/**
	 * 搜索
	 * 
	 * @return
	 */
	public boolean search() {

		// 只是处理xml文件，读取xml文件，匹配
		long start = System.currentTimeMillis();

		ExecutorService executorService = new ThreadPoolExecutor(10, 64, 10L, TimeUnit.MILLISECONDS,
				new LinkedBlockingQueue<Runnable>());

		try {
			fCollector.beginReporting();

			List<String> solutionsPath = LoadFileTree.getAllSolutionsPath();
			for (String solutionPath : solutionsPath) {
				File solutionDir = new File(solutionPath);
				this.processDir(solutionDir, solutionDir, executorService);
			}

			executorService.shutdown();
			while (!executorService.awaitTermination(100, TimeUnit.MILLISECONDS))
				;
		} catch (Exception e) {
			e.printStackTrace();
		} finally {
			fCollector.endReporting();
		}

		long end = System.currentTimeMillis();

//		System.out.println("-----查询耗时--------" + (end - start));
		return true;
	}

	public boolean processDir(File dir, File solutionDir, ExecutorService executorService) throws Exception {
		if (dir != null && dir.isDirectory()) {
			for (File child : dir.listFiles()) {
				if (child.isDirectory()) {
					processDir(child, solutionDir, executorService);
					// executorService.submit(() -> processDir(child, solutionDir,
					// executorService));
				} else {
					executorService.submit(() -> processFile(child, solutionDir));
				}
			}
		}

		return false;
	}

	public boolean processFile(File file, File solutionDir) throws Exception {
		Matcher matcher = fSearchPattern.pattern().length() == 0 ? null : fSearchPattern.matcher("");

		// 后面做文件名称的过滤，无匹配
		if (!fCollector.acceptFile(file, solutionDir) || matcher == null) {
			return false;
		}

		String fileContent = FileUtils.readFileToString(file, "UTF-8");
		String fileType = getFileType(fileContent);
		
		if (fCollector.acceptFileType(fileType)) {
			locateMatches(file, solutionDir, fileContent, matcher, fileType);
		}

		return true;
	}

	/**
	 * 匹配文件内容
	 */
	private void locateMatches(File file, File solutionDir, String searchInput, Matcher matcher, String fileType) {
		matcher.reset(searchInput);

		FileMatchResource fileMatchRes = new FileMatchResource(solutionDir, file, fileType);
		while (matcher.find()) {
			int start = matcher.start();
			int end = matcher.end();

			if (end != start) {
				ReusableMatchAccess access = new ReusableMatchAccess();
				access.initialize(fileMatchRes, start, end - start, searchInput);

				// 添加
				fCollector.acceptPatternMatch(access);

			}
		}

	}

	private String getFileType(String searchInput) {
		int ix = Util.getBeginIndex(searchInput);
		int maxIndex = searchInput.length();

		while (ix < maxIndex) {
			Element ele = Util.getNextElement(searchInput, ix);
			if (ele.getType() == Element.TAG) {
				return ele.getTagName();
			}
			ix = ele.getEndIndex() + 1;
		}

		return "";
	}

	public static class ReusableMatchAccess extends TextSearchMatchAccess {
		private FileMatchResource fFile;
		private int fOffset;
		private int fLength;
		private CharSequence fContent;

		public void initialize(FileMatchResource file, int offset, int length, CharSequence content) {
			fFile = file;
			fOffset = offset;
			fLength = length;
			fContent = content;
		}

		@Override
		public int getMatchOffset() {
			return fOffset;
		}

		@Override
		public int getMatchLength() {
			return fLength;
		}

		@Override
		public int getFileContentLength() {
			return fContent.length();
		}

		@Override
		public char getFileContentChar(int offset) {
			return fContent.charAt(offset);
		}

		@Override
		public String getFileContent(int offset, int length) {
			return fContent.subSequence(offset, offset + length).toString();
		}

		@Override
		public FileMatchResource getFile() {
			return fFile;
		}
	}
}
