package com.bokesoft.yes.design.search;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.bokesoft.yes.design.search.text.Match;

/**
 * 搜索结果
 */
public class FileSearchResult {

	private static final Match[] EMPTY_ARRAY = new Match[0];

	private final Map<Object, List<Match>> fElementsToMatches;

	public FileSearchResult() {
		fElementsToMatches = new HashMap<>();
	}

	public void addMatch(Match match) {
		synchronized (fElementsToMatches) {
			doAddMatch(match);
		}
	}

	public void addMatches(Match[] matches) {
		synchronized (fElementsToMatches) {
			for (Match match : matches) {
				doAddMatch(match);
			}
		}
	}

	private boolean doAddMatch(Match match) {

		List<Match> matches = fElementsToMatches.get(match.getElement());

		if (matches == null) {
			matches = new ArrayList<>();
			fElementsToMatches.put(match.getElement(), matches);
			matches.add(match);
			return true;
		}

		if (!matches.contains(match)) {
			insertSorted(matches, match);
			return true;
		}

		return false;
	}

	private static void insertSorted(List<Match> matches, Match match) {
		int insertIndex = getInsertIndex(matches, match);
		matches.add(insertIndex, match);
	}

	private static int getInsertIndex(List<Match> matches, Match match) {
		int count = matches.size();
		int min = 0;
		int max = count - 1;
		while (min <= max) {
			int mid = (min + max) / 2;
			Match data = matches.get(mid);
			int compare = compare(match, data);
			if (compare > 0)
				max = mid - 1;
			else
				min = mid + 1;
		}

		return min;
	}

	private static int compare(Match match1, Match match2) {
		int diff = match2.getOffset() - match1.getOffset();
		if (diff != 0)
			return diff;
		return match2.getLength() - match1.getLength();
	}

	public void removeAll() {
		synchronized (fElementsToMatches) {
			doRemoveAll();
		}
	}

	private void doRemoveAll() {
		fElementsToMatches.clear();
	}

	public void removeMatch(Match match) {
		synchronized (fElementsToMatches) {
			doRemoveMatch(match);
		}
	}

	public void removeMatches(Match[] matches) {
		synchronized (fElementsToMatches) {
			for (Match match : matches) {
				doRemoveMatch(match);
			}
		}
	}

	private boolean doRemoveMatch(Match match) {
		boolean existed = false;
		List<Match> matches = fElementsToMatches.get(match.getElement());
		if (matches != null) {
			existed = matches.remove(match);
			if (matches.isEmpty())
				fElementsToMatches.remove(match.getElement());
		}
		return existed;
	}

	public int getMatchCount() {
		int count = 0;
		synchronized (fElementsToMatches) {
			for (List<Match> element : fElementsToMatches.values()) {
				if (element != null)
					count += element.size();
			}
		}
		return count;
	}

	public int getMatchCount(Object element) {
		List<Match> matches = fElementsToMatches.get(element);
		if (matches != null)
			return matches.size();
		return 0;
	}

	public Object[] getElements() {
		synchronized (fElementsToMatches) {
			return fElementsToMatches.keySet().toArray();
		}
	}

	public Match[] getMatches(Object element) {
		synchronized (fElementsToMatches) {
			List<Match> matches = fElementsToMatches.get(element);
			if (matches != null)
				return matches.toArray(new Match[matches.size()]);
			return EMPTY_ARRAY;
		}
	}

}
