package com.bokesoft.yes.design.codemirror;

import com.bokesoft.yes.common.log.LogSvr;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.lang.reflect.Method;
import java.util.Objects;

/***
 * 自定义类加载器，加载所有target目录下的class文件
 * @author zhsy
 *
 */
public class MyClassLoader extends ClassLoader {

	private final String classPath;

	public MyClassLoader(String classPath) {
		this.classPath = classPath;
	}

	/**
	 * 编写findClass方法的逻辑
	 *
	 * @param name
	 * @return
	 * @throws ClassNotFoundException
	 */
	@Override
	protected Class<?> findClass(String name) throws ClassNotFoundException {
		// 获取类的class文件字节数组
		byte[] classData = getClassData(name);
		if (Objects.isNull(classData)) {
			throw new ClassNotFoundException();
		} else {
			// 生成class对象
			return defineClass(name, classData, 0, classData.length);
		}
	}

	/**
	 * 编写获取class文件并转换为字节码流的逻辑
	 *
	 * @param className
	 * @return
	 */
	private byte[] getClassData(String className) {
		// 读取类文件的字节
		String path = classNameToPath(className);
		InputStream inStream = null;
		ByteArrayOutputStream stream = null;
		try {
			inStream = new FileInputStream(path);
			stream = new ByteArrayOutputStream();
			byte[] buffer = new byte[2048];
			int num = 0;
			// 读取类文件的字节码
			while ((num = inStream.read(buffer)) != -1) {
				stream.write(buffer, 0, num);
			}
			return stream.toByteArray();
		} catch (IOException e) {
			LogSvr.getInstance().error(e.getMessage(), e);
		} finally {
			try {
				if (inStream != null) {
					inStream.close();
				}
				if (stream != null) {
					stream.close();
				}
			} catch (IOException e) {
				LogSvr.getInstance().error(e.getMessage(), e);
			}
		}
		return null;
	}

	/**
	 * 类文件的完全路径
	 *
	 * @param className
	 * @return
	 */
	private String classNameToPath(String className) {
		return classPath + File.separatorChar
				+ className.replace('.', File.separatorChar) + ".class";
	}

	public static void main(String[] args) {
		String classPath = "D:\\source2\\bokeerp\\autorecord\\target\\classes\\";
		MyClassLoader loader = new MyClassLoader(classPath);

		try {
			//加载指定的class文件
			Class<?> clazz = loader.loadClass("com.bokesoft.erp.record.AutoRecord");
			Method[] methods = clazz.getDeclaredMethods();
		} catch (Exception e) {
			LogSvr.getInstance().error(e.getMessage(), e);
		}
	}
}