package com.bokesoft.yes.design.bpm.util;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.bokesoft.yes.design.cmd.XmlFileProcessor;
import com.bokesoft.yes.design.enums.NodeKeyAndCaptionEnum;
import com.bokesoft.yes.design.io.LoadFileTree;
import com.bokesoft.yes.design.constant.ConstantUtil;
import com.bokesoft.yes.design.utils.XMLWriter;
import com.bokesoft.yes.meta.persist.dom.bpm.MetaProcessSave;
import com.bokesoft.yigo.meta.base.IMetaResolver;
import com.bokesoft.yigo.meta.bpm.process.MetaProcess;
import com.bokesoft.yigo.meta.factory.IMetaFactory;
import com.bokesoft.yigo.meta.factory.MetaFactory;
import org.apache.commons.lang3.exception.ExceptionUtils;
import org.apache.commons.lang3.StringUtils;
import org.dom4j.Attribute;
import org.dom4j.Document;
import org.dom4j.Element;
import org.dom4j.io.OutputFormat;
import org.dom4j.io.SAXReader;
import org.apache.commons.collections.CollectionUtils;

import java.io.File;
import java.io.FileOutputStream;
import java.util.*;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.logging.Logger;
import java.util.regex.Pattern;

/***
 * bpm转换为json操作工具类
 * @date 2020-09-25
 * @author zhsy
 */
public class BpmOperToJsonUtil {

	private static final Logger logger = Logger.getLogger(BpmOperToJsonUtil.class.getName());
	public static final BpmOperToJsonUtil instance = new BpmOperToJsonUtil();
	/***BPM目录*/
	private static final String BPM_DIRECTORY = "BPM";
	/***xml扩展名*/
	private static final String XML_EXTENSION = ".xml";
	/**
	 * 用于保存临时文件的栈(先进后出比较适合)，主要的作用是用于撤销和保存操作
	 */
	private static final Stack<String> bpmStack = new Stack<>();

	/***
	 * 新增xml文件
	 * @param projectKey 项目key,例如:hrconfig,mmconfig等
	 * @param key 工作流key
	 * @param caption 工作流描述
	 * @param version 版本号
	 * @Return {"result":true或false,"filePath":""}
	 */
	public Map<String, String> createXml(String projectKey, String key, String caption, String version) {
		Map<String, String> result = new HashMap<>(10);
		result.put("result", "true");
		String filePath = "";
		try {
			// 校验
			if (validateForCreateXml(projectKey, key, caption, version, result)) {
				return result;
			}

			int verionNo = Integer.parseInt(version) + 1;
			if (key.contains("_V")) {
				key = key.substring(0, key.indexOf("_V")) + "_V" + verionNo;
			} else {
				if (verionNo > 1) {
					key = key + "_V" + verionNo;
				}
			}
				boolean a = true;
				while (a){
					if (!LoadFileTree.isExistKey(key)){
						a = false;
						break;
					}
					verionNo ++;
					key = key.split("_V")[0] + "_V" + verionNo;
				}



			MetaProcess process = new MetaProcess();
			process.setKey(key);
			process.setCaption(caption);
			result.put("caption", caption);
			process.setVersion(verionNo);
			result.put("version", String.valueOf(verionNo));
			IMetaFactory metaFactory = MetaFactory.getGlobalInstance();
			IMetaResolver resolver = metaFactory.getProjectResolver(projectKey);
			String newFileName = key + XML_EXTENSION;
			// 保存xml文件到磁盘
			MetaProcessSave metaProcessSave = new MetaProcessSave(process);
			metaProcessSave.save(resolver, File.separator + BPM_DIRECTORY + File.separator + newFileName);
			String versionStr = "0".equals(version) ? "" : " Version=\"" + verionNo + "\"";


			// 调用新增xml文件方法
			filePath = LoadFileTree.newBpmFile(projectKey, newFileName, key, caption);
			String fileContent = "<?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"no\"?>\n" +
					"<Process Caption=\"" + caption + "\" Key=\"" + key + "\"" + versionStr + ">\n" +
					"<Begin ID='1' Key='Begin' Caption='起始'>" +
					"<NodeGraphic X='303' Y='131'/>" +
					"<TransitionCollection>" +
					"<SequenceFlow ID='3' Key='SequenceFlow' Caption='顺序流' TargetNodeKey='End'/>" +
					"</TransitionCollection>" +
					"</Begin>" +
					"<End ID='2' Key='End' Caption='结束'>" +
					"<NodeGraphic X='303' Y='597'/>" +
					"</End>" + "\n" +
					"</Process>";
			XmlFileProcessor.instance.saveTempFile(filePath, fileContent,projectKey);

			result.put("result", "true");
			result.put("filePath", filePath);
			result.put("version", String.valueOf(verionNo));
			result.put("fileName", newFileName.replaceAll(".xml", ""));
		} catch (Throwable e) {
			logger.warning("新增xml文件异常,异常为:" + ExceptionUtils.getStackTrace(e));
			if (StringUtils.isNotEmpty(filePath)) {
				try {
					LoadFileTree.removeFilePath(filePath);
				} catch (Throwable e1) {
					logger.warning("删除文件异常,异常为:" + ExceptionUtils.getStackTrace(e1));
				}
			}
			result.put("result", "false");
			result.put("filePath", "");
			result.put("msg", e.getMessage());
		}
		return result;
	}

	/***
	 * 创建xml文件校验
	 * @param projectKey 项目标识，如：hrconfig
	 * @param key 标识
	 * @param caption 名称
	 * @param version 版本
	 * @param result 校验结果
	 * @return true:校验失败；false:校验成功
	 */
	private boolean validateForCreateXml(String projectKey, String key, String caption, String version,
										 Map<String, String> result) {
		// 非空校验
		if (StringUtils.isEmpty(projectKey)) {
			result.put("result", "false");
			result.put("filePath", "");
			result.put("msg", "项目不能为空");
			return true;
		}
		if (StringUtils.isEmpty(key)) {
			result.put("result", "false");
			result.put("filePath", "");
			result.put("msg", "标识不能为空");
			return true;
		}
		if (StringUtils.isEmpty(caption)) {
			result.put("result", "false");
			result.put("filePath", "名称不能为空");
			result.put("msg", "");
			return true;
		}

		String pattern = "([A-Za-z_]\\w*)";
		boolean isMatch = Pattern.matches(pattern, key);
		if (!isMatch) {
			result.put("result", "false");
			result.put("filePath", "");
			result.put("msg", "标识不正确，不是字母或下划线开始的");
			return true;
		}

		// 如果是新增文件，并且文件已经存在，直接返回错误信息
		if (LoadFileTree.isExistKey(key) && "0".equals(version)) {
			result.put("result", "false");
			result.put("filePath", "");
			result.put("msg", "文件'" + key + "'已存在");
			return true;
		}
		return false;
	}

	/***
	 * 转换xml文件为json文件
	 * @param filePath xml文件路径
	 *
	 */
	public JSONObject convertXmlToJson(String filePath) throws Throwable {
		JSONObject workFlowJson = new JSONObject();
		String tmpFile = XmlFileProcessor.instance.getTmpFile(filePath);
		if (StringUtils.isBlank(tmpFile)) {
			tmpFile = filePath;
		}
		// 创建SAX读取器
		SAXReader reader = new SAXReader();
		// 加载文档
		Document document = reader.read(new File(tmpFile));
		// 获取根节点
		Element root = document.getRootElement();
		// 获取文件的 caption
		String caption = root.attributeValue(ConstantUtil.CAPTION);
		try {
			// 下面是节点列表json设置
			JSONObject states = new JSONObject();
			// 连线列表json设置
			JSONObject paths = new JSONObject();

			// 设置xml的各个节点属性值
			for (NodeKeyAndCaptionEnum value : NodeKeyAndCaptionEnum.values()) {
				if ("Swimline".equals(value.getKey())) {
					Element swimlineCollection = root.element("SwimlineCollection");
					if (null != swimlineCollection) {
						setNodeXmlToJson(swimlineCollection, states, paths, value.getKey());
					}
				} else {
					setNodeXmlToJson(root, states, paths, value.getKey());
				}
			}

			// 设置老的xml的x坐标及矩形框的宽度
			setOldXmlXAndWidth(states, paths, filePath, document);

			// 设置工作流属性值
			JSONObject workFlowDesc = setWorkFlowAttributes(root);

			// 节点列表
			workFlowJson.put("states", states);
			// 连线列表
			workFlowJson.put("paths", paths);
			// 工作流描述
			workFlowJson.put("props", workFlowDesc);
			//工作流名称
			workFlowJson.put(ConstantUtil.CAPTION, caption);
		} catch (Throwable e) {
			logger.warning("转换xml文件为json文件异常，异常为:" + ExceptionUtils.getStackTrace(e));
		}
		return workFlowJson;
	}

	/***
	 * 设置老的xml的x坐标及矩形框宽度
	 * @param states 节点集合
	 * @param paths 路径集合
	 * @param filePath 文件路径
	 * @param document 操作文档
	 */
	private void setOldXmlXAndWidth(JSONObject states, JSONObject paths, String filePath, Document document) {
		// 对老的xml中的节点x坐标进行处理，增加长度增加150
		// x坐标值
		int x;
		// 是否是老的xml文件，默认false
		boolean isOldXml = false;
		for (String key : states.keySet()) {
			x = states.getJSONObject(key).getJSONObject("attr").getInteger(ConstantUtil.x);
			if (x < 150) {
				isOldXml = true;
				break;
			}
		}

		if (isOldXml) {
			// 节点类型
			String nodeType;
			// 矩形框宽度
			int width = 40;
			// 不需要调整宽度的节点Key列表
			List<String> nodeKeyList = Arrays.asList("Begin", "End", "BranchEnd");
			boolean isAjustWidth;
			// for循环:调整节点的x坐标
			for (String key : states.keySet()) {
				nodeType = states.getJSONObject(key)
						.getJSONObject("props").getJSONObject("NodeType").getString("value");
				x = states.getJSONObject(key).getJSONObject("attr").getInteger(ConstantUtil.x);
				states.getJSONObject(key).getJSONObject("attr").put(ConstantUtil.x, (x + 150));

				// 是否需要调整宽度，默认true
				isAjustWidth = true;
				// 判断是否需要调整宽度
				for (String k : nodeKeyList) {
					if (k.equals(nodeType)) {
						isAjustWidth = false;
						break;
					}
				}

				// 不需要调整宽度，就跳过
				if (!isAjustWidth) {
					continue;
				}

				boolean isTaskNode = false;
				// 任务等使用矩形框的节点
				List<String> taskKeys = Arrays.asList("UserTask", "ServiceTask", "ManualTask", "Audit", "DataMap",
						"SubProcess", "Inline", "StateAction", "Countersign", "Swimline",
						"MultiUserTask", "MultiAudit");
				for (String k : taskKeys) {
					isTaskNode = k.equals(nodeType);
					if (isTaskNode) {
						break;
					}
				}
				if (isTaskNode) {
					if (NodeKeyAndCaptionEnum.MultiUserTask.getKey().equals(nodeType) &&
							NodeKeyAndCaptionEnum.MultiAudit.getKey().equals(nodeType)) {
						width = Math.max(width, 130);
					} else if (NodeKeyAndCaptionEnum.Audit.getKey().equals(nodeType)) {
						width = Math.max(width, 100);
					} else {
						width = Math.max(width, 115);
					}
				} else {
					width = Math.max(width, 48);
				}
				states.getJSONObject(key).getJSONObject("attr").put("width", width);
			}

			// for循环:调整路径的x坐标
			for (String key : paths.keySet()) {
				x = paths.getJSONObject(key).getJSONObject("textPos").getInteger(ConstantUtil.x);
				paths.getJSONObject(key).getJSONObject("textPos").put(ConstantUtil.x, (x + 150));
			}

			// 更新xml文件中的节点x坐标及宽度(开始节点，结束节点，分支结束等节点宽度不修改)、路径的x坐标
			updateOldXmlXAndWidth(nodeKeyList, filePath, document);
		}
	}

	/***
	 * 更新xml文件中的节点x坐标及宽度(开始节点，结束节点，分支结束等节点宽度不修改)、路径的x坐标
	 * @param nodeKeyList 不需要设置宽度的节点列表
	 * @param filePath 文件路径
	 * @param document 操作文档
	 */
	private void updateOldXmlXAndWidth(List<String> nodeKeyList, String filePath, Document document) {
		FileOutputStream fileOutputStream = null;
		XMLWriter writer = null;
		try {
			// 获取根节点
			Element root = document.getRootElement();
			// 获取根节点下所有节点
			List<Element> elements = root.elements();

			boolean isAjustWidth;
			for (Element ele : elements) {
				// 是否需要调整宽度，默认true
				isAjustWidth = true;
				if (null != ele.element("NodeGraphic")) {
					String pX = ele.element("NodeGraphic").attributeValue(ConstantUtil.X);
					ele.element("NodeGraphic").remove(ele.element("NodeGraphic").attribute(ConstantUtil.X));
					ele.element("NodeGraphic").addAttribute(ConstantUtil.X, String.valueOf(Integer.parseInt(pX) + 150));
					// 判断是否需要调整宽度
					for (String k : nodeKeyList) {
						if (k.equals(ele.getName())) {
							isAjustWidth = false;
							break;
						}
					}

					// 不调整宽度，就跳过
					if (!isAjustWidth) {
						continue;
					}

					// 计算矩形框的宽度
					int width_ = ele.attributeValue(ConstantUtil.CAPTION).length() * 20;
					if (null != ele.element("NodeGraphic").attribute("Width")) {
						ele.element("NodeGraphic").remove(ele.element("NodeGraphic").attribute("Width"));
					}
					ele.element("NodeGraphic").addAttribute("Width", String.valueOf(width_));
				}

				// 泳道的处理
				if ("SwimlineCollection".equals(ele.getName())) {
					List<Element> swimlines = ele.elements("Swimline");
					if (!CollectionUtils.isEmpty(swimlines)) {
						int x_;
						for (Element element : swimlines) {
							StringBuilder postion = new StringBuilder();
							String[] array1 = element.attributeValue("Position").split(",");
							String[] array2 = array1[0].split(":");
							x_ = Integer.parseInt(array2[1]) + 150;
							postion.append("X:").append(x_).append(",").append(array1[1]);
							element.remove(element.attribute("Position"));
							element.addAttribute("Position", postion.toString());
						}
					}
				}
			}

			// 格式化输出流，同时指定编码格式。也可以在FileOutputStream中指定。
			OutputFormat format = getOutputFormat();

			fileOutputStream = new FileOutputStream(filePath);
			writer = new XMLWriter(fileOutputStream, format);
			writer.write(document);
		} catch (Exception e) {
			ExceptionUtils.getStackTrace(e);
			logger.warning("更新xml文件异常，异常为:" + ExceptionUtils.getStackTrace(e));
		} finally {
			closeFileStream(fileOutputStream, writer);
		}
	}

	/***
	 * 关闭文件流
	 * @param fileOutputStream 文件输出流
	 * @param writer 写对象
	 */
	private void closeFileStream(FileOutputStream fileOutputStream, XMLWriter writer) {
		try {
			if (null != fileOutputStream) {
				fileOutputStream.close();
			}
			if (null != writer) {
				writer.close();
			}
		} catch (Exception e) {
			ExceptionUtils.getStackTrace(e);
			logger.warning("关闭writer异常，异常为:" + ExceptionUtils.getStackTrace(e));
		}
	}

	/***
	 * 写xml格式化
	 * @return 格式化类
	 */
	private OutputFormat getOutputFormat() {
		// 格式化输出流，同时指定编码格式。也可以在FileOutputStream中指定。
		OutputFormat format = OutputFormat.createPrettyPrint();
		// 设置xml标签和Process标签之间没有空行
		format.setNewLineAfterDeclaration(false);
		format.setIndentSize(4);
		format.setExpandEmptyElements(false);
		format.setPadText(false);
		return format;
	}

	/***
	 * 设置节点xml转json
	 * @param root 根节点
	 * @param states 节点列表key
	 * @param paths 路径列表key
	 * @param nodeName 节点名
	 */
	private void setNodeXmlToJson(Element root, JSONObject states, JSONObject paths, String nodeName) {
		// 获取到"Join"节点
		List<Element> joinList = root.elements(nodeName);

		for (Element element : joinList) {
			states.put(element.attributeValue("Key"), getNodeJson(element, nodeName));
			// "End"和”BranchEnd“节点没有连接线
			if (!"End".equals(nodeName)) {
				// 获取并设置节点下的连线
				this.hendleLines(element, paths);
			}
		}
	}

	/***
	 * 设置工作流属性的值
	 * @param root 根元素
	 * @return 设置后的JSON对象
	 */
	private JSONObject setWorkFlowAttributes(Element root) {
		List<String> keyList = Arrays.asList(ConstantUtil.KEY, ConstantUtil.CAPTION, ConstantUtil.VERSION, "DMTable", ConstantUtil.FORM_KEY, "PermCollection",
				"IgnoreFormState", "QueryRetreatWorkitem", "TemplateKey", "KillInstanceTrigger",
				"LockWorkitem", "BPMGraphInfoPath", "WorkFlowHookPath");
		JSONObject workFlowDescSub = new JSONObject();
		JSONObject temp;
		String value;
		boolean isEmpty;
		for (String key : keyList) {
			value = root.attributeValue(key);
			isEmpty = null == value || StringUtils.isEmpty(value);
			switch (key) {
				case "DMTable":
					value = setDMTableJsonFromXml(root, value, key);
					break;
				case ConstantUtil.VERSION:
					if (isEmpty) {
						value = "1";
					}
					break;
				case "QueryRetreatWorkitem":
					if (isEmpty) {
						value = "true";
					}
					break;
				case "LockWorkitem":
					if (isEmpty) {
						value = "false";
					}
					break;
				case "BPMGraphInfoPath":
					value = Objects.isNull(root.element("Expand"))? "":root.element("Expand").attributeValue("BPMGraphInfoPath");
					break;
				case "WorkFlowHookPath":
					value = Objects.isNull(root.element("Expand"))? "" :root.element("Expand").attributeValue("WorkFlowHookPath");
					break;
				case "PermCollection":
					if (null != root.element("PermCollection")) {
						value = setPerm(root.element("PermCollection")).toJSONString();
					}
					value = "{}".equals(value) ? "" : value;
					break;
				default:
					break;
			}
			temp = new JSONObject();
			temp.put("value", value);
			workFlowDescSub.put(key, temp);
		}
		JSONObject workFlowDesc = new JSONObject();
		workFlowDesc.put("props", workFlowDescSub);
		return workFlowDesc;
	}

	/***
	 * 根据xml文件设置DMTable json
	 * @param root 根元素
	 * @param value 值对象
	 * @param key 键值
	 * @return DMTable json
	 */
	private String setDMTableJsonFromXml(Element root, String value, String key) {
		if (null != root.element(key) && null != root.element(key).elements("Field")) {
			List<Element> fields = root.element(key).elements("Field");
			JSONArray array = new JSONArray();
			JSONObject jsonObj;
			JSONObject jsonDetailObj;
			String type;
			String detail;
			for (int i = 0; i < fields.size(); i++) {
				jsonObj = new JSONObject();
				jsonObj.put("id", (i + 1));
				jsonObj.put(ConstantUtil.KEY, fields.get(i).attributeValue(ConstantUtil.KEY));
				if (StringUtils.isNotEmpty(fields.get(i).attributeValue(ConstantUtil.CAPTION))) {
					jsonObj.put(ConstantUtil.CAPTION, fields.get(i).attributeValue(ConstantUtil.CAPTION));
				} else {
					jsonObj.put(ConstantUtil.CAPTION, "");
				}
				type = fields.get(i).attributeValue(ConstantUtil.TYPE);
				if (StringUtils.isNotEmpty(type)) {
					jsonObj.put(ConstantUtil.TYPE, type);
					jsonObj.put("Value", fields.get(i).attributeValue("ConstValue"));
				} else {
					jsonObj.put(ConstantUtil.TYPE, "Field");
					jsonObj.put("Value", fields.get(i).attributeValue("SourceFieldKey"));
				}
				detail = fields.get(i).attributeValue("Detail");
				if (StringUtils.isNotEmpty(detail)) {
					jsonDetailObj = JSON.parseObject(detail);
				} else {
					jsonDetailObj = new JSONObject();
					List<String> keys = Arrays.asList(ConstantUtil.KEY, ConstantUtil.CAPTION, ConstantUtil.TYPE, "ColumnName", "DefaultVaue", "Length",
							"Precision", "Scale");
					for (String ele : keys) {
						if (ConstantUtil.KEY.equals(ele)) {
							jsonDetailObj.put(ele, fields.get(i).attributeValue(ele));
						} else {
							jsonDetailObj.put(ele, "");
						}
					}
				}
				jsonObj.put("Detail", jsonDetailObj);
				array.add(jsonObj);
			}
			value = array.toJSONString();
		}
		return value;
	}

	/***
	 * 获取节点的json
	 * @param element 元素节点
	 * @param nodeType 节点类型
	 * @return 节点json
	 */
	private JSONObject getNodeJson(Element element, String nodeType) {
		JSONObject json = new JSONObject();
		try {
			// 设置节点文本
			setNodeText(element, json);

			// 设置节点坐标
			setNodeCoordinate(element, nodeType, json);

			// 设置节点类型，这个很重要，不设置，前端图会显示错误
			json.put("type", nodeType);

			// 设置公共属性"NodeType,ID,Key,Caption"
			JSONObject props = setCommonAttributes(element);

			// 不是泳道类型，设置Image等8个属性
			if (!NodeKeyAndCaptionEnum.Swimline.getKey().equals(nodeType)) {
				setImageAndSoOnAttrbutesFromXml(props, element);
			}

			// 根据类型设置各自的属性
			setSelfAttributesFromXml(element, nodeType, props);

			json.put("props", props);
		} catch (Throwable e) {
			logger.warning("获取节点的json方法getNodeJson()操作失败，异常信息为:" + ExceptionUtils.getStackTrace(e));
		}
		return json;
	}

	/***
	 * 设置节点坐标
	 * @param element 当前节点
	 * @param nodeType 节点类型
	 * @param json 设置后的json
	 */
	private void setNodeCoordinate(Element element, String nodeType, JSONObject json) {
		// 坐标
		Element nodeGraphic = element.element("NodeGraphic");
		int x = 150;
		int y = 150;
		int height = 48;
		int width = 40;
		if (null != nodeGraphic) {
			x = Integer.parseInt(nodeGraphic.attributeValue(ConstantUtil.X));
			y = Integer.parseInt(nodeGraphic.attributeValue(ConstantUtil.Y));
			height = StringUtils.isNotEmpty(nodeGraphic.attributeValue(ConstantUtil.HEIGHT))
					? Integer.parseInt(nodeGraphic.attributeValue(ConstantUtil.HEIGHT)) : 48;
			height = Math.max(height, 48);
			boolean isTaskNode = false;
			// 任务等使用矩形框的节点
			List<String> taskKeys = Arrays.asList("UserTask", "ServiceTask", "ManualTask", "Audit", "DataMap",
					"SubProcess", "Inline", "StateAction", "Countersign", "Swimline",
					"MultiUserTask", "MultiAudit");
			for (String key : taskKeys) {
				isTaskNode = key.equals(nodeType);
				if (isTaskNode) {
					break;
				}
			}

			// 设置宽度
			if (isTaskNode) {
				if (StringUtils.isEmpty(nodeGraphic.attributeValue("Width"))) {
					if (NodeKeyAndCaptionEnum.MultiUserTask.getKey().equals(nodeType) ||
							NodeKeyAndCaptionEnum.MultiAudit.getKey().equals(nodeType)) {
						width = 130;
					} else if (NodeKeyAndCaptionEnum.Audit.getKey().equals(nodeType)) {
						width = 100;
					} else {
						width = 115;
					}
				} else {
					width = Integer.parseInt(nodeGraphic.attributeValue("Width"));
					if (NodeKeyAndCaptionEnum.MultiUserTask.getKey().equals(nodeType) ||
							NodeKeyAndCaptionEnum.MultiAudit.getKey().equals(nodeType)) {
						width = Math.max(width, 130);
					} else if (NodeKeyAndCaptionEnum.Audit.getKey().equals(nodeType)) {
						width = Math.max(width, 100);
					} else {
						width = Math.max(width, 115);
					}
				}
			} else {
				if (StringUtils.isEmpty(nodeGraphic.attributeValue("Width"))) {
					width = 48;
				} else {
					width = Integer.parseInt(nodeGraphic.attributeValue("Width"));
				}
			}
		}

		// 泳道重新赋值
		if (NodeKeyAndCaptionEnum.Swimline.getKey().equals(nodeType)) {
			Attribute position = element.attribute("Position");
			x = Integer.parseInt(position.getValue().split(",")[0].split(":")[1]);
			y = Integer.parseInt(position.getValue().split(",")[1].split(":")[1]);
			String size = element.attributeValue("Size");
			width = Integer.parseInt(size.split(",")[0].split(":")[1]);
			height = Integer.parseInt(size.split(",")[1].split(":")[1]);
		}

		// attr
		JSONObject attr = new JSONObject();
		attr.put(ConstantUtil.x, x);
		attr.put(ConstantUtil.y, y);
		attr.put("width", width);
		attr.put("height", height);
		json.put("attr", attr);
	}

	/***
	 * 设置节点文本
	 * @param element 当前节点
	 * @param json 设置后的json
	 */
	private void setNodeText(Element element, JSONObject json) {
		String text;
		text = element.attributeValue(ConstantUtil.CAPTION);
		JSONObject textJson = new JSONObject();
		textJson.put("text", text);
		json.put("text", textJson);
	}

	/***
	 * 根据节点类型，设置各自属性
	 * @param element 当前节点
	 * @param nodeType 节点类型
	 * @param props 设置后的属性json
	 */
	private void setSelfAttributesFromXml(Element element, String nodeType, JSONObject props) {
		// 设置”开始“和”结束“自己的属性
		if (NodeKeyAndCaptionEnum.Begin.getKey().equals(nodeType)
				|| NodeKeyAndCaptionEnum.End.getKey().equals(nodeType)) {
			setAttributeToJson(element, props, "Status");
		}

		// 设置“内联流程“自己的属性
		if (NodeKeyAndCaptionEnum.Inline.getKey().equals(nodeType)) {
			setAttributeToJson(element, props, "InlineProcessKey");
		}

		// 设置“泳道”自己的属性
		if (NodeKeyAndCaptionEnum.Swimline.getKey().equals(nodeType)) {
			setSwimline(props, element);
		}

		// 设置“路由节点”、“选择节点”自己的属性
		if (NodeKeyAndCaptionEnum.GateWay.getKey().equals(nodeType)
				|| NodeKeyAndCaptionEnum.Decision.getKey().equals(nodeType)) {
			setAttributeToJson(element, props, "Condition");
		}

		// 设置"系统任务"自己的属性
		if (NodeKeyAndCaptionEnum.ServiceTask.getKey().equals(nodeType)) {
			JSONObject value = new JSONObject();
			if (null != element.element(ConstantUtil.ACTION)) {
				value.put("value", element.element(ConstantUtil.ACTION).getData().toString().trim());
			} else {
				value.put("value", "");
			}
			props.put(ConstantUtil.ACTION, value);
		}

		// 设置"复杂合并"自己的属性
		if (NodeKeyAndCaptionEnum.ComplexJoin.getKey().equals(nodeType)) {
			// 设置"合并顺序流数量"属性
			setAttributeToJson(element, props, "JoinCount");
			// 设置"合并条件"属性
			JSONObject value = new JSONObject();
			if (null != element.element("JoinCondition")) {
				value.put("value", element.element("JoinCondition").getData().toString().trim());
			} else {
				value.put("value", "");
			}
			props.put("JoinCondition", value);
		}

		// 设置”用户任务“、”多人任务节点“、"会签节点"、”数据映射“自己的属性
		if (NodeKeyAndCaptionEnum.UserTask.getKey().equals(nodeType)
				|| NodeKeyAndCaptionEnum.MultiUserTask.getKey().equals(nodeType)
				|| NodeKeyAndCaptionEnum.Countersign.getKey().equals(nodeType)
				|| NodeKeyAndCaptionEnum.DataMap.getKey().equals(nodeType)) {
			setUserTaskAttributesFromXml(props, element, nodeType);
		}

		// 设置”审批任务“和"多人审批节点"和"子流程"自己的属性
		if (NodeKeyAndCaptionEnum.Audit.getKey().equals(nodeType)
				|| NodeKeyAndCaptionEnum.MultiAudit.getKey().equals(nodeType)
				|| NodeKeyAndCaptionEnum.SubProcess.getKey().equals(nodeType)) {
			setAuditAttributesFromXml(props, element, nodeType);
		}
       if (NodeKeyAndCaptionEnum.StateAction.getKey().equals(nodeType)){
	       setStateActionAttributesFromXml(props, element);
}
		// 设置”状态节点“自己的属性
		if (NodeKeyAndCaptionEnum.State.getKey().equals(nodeType)) {
			setStateAttributesFromXml(props, element);
		}
	}

	private void setStateActionAttributesFromXml(JSONObject props, Element node) {
		List<String> keyList = Arrays.asList("Enable", "Visible", "Icon", "TemplateKey", "Action",
				"OperationKey", "OperationCaption","RevokeCondition","ParticipatorCollection");
		JSONObject value;
		for (String k : keyList) {
			value = new JSONObject();
			switch (k) {
				case "Enable":
				case "Visible":
				case "Icon":
				case "TemplateKey":
				case "RevokeCondition":
				case "Action":
					if (null == node.attributeValue(k) || StringUtils.isEmpty(node.attributeValue(k))) {
						value.put("value", "");
					} else {
						value.put("value", node.attributeValue(k));
					}
					break;
				case "Perm":
					value.put("value", JSON.toJSONString(setPerm(node)));
					break;
				case "OperationKey":
				if (null != node.element("Operation")){
					value.put("value", node.element("Operation").attributeValue("Key"));
				}else {
					value.put("value", "");
				}
					break;
				case "OperationCaption":
					if (null != node.element("Operation")){
						value.put("value", node.element("Operation").attributeValue("Caption"));
					}else {
						value.put("value", "");
					}
					break;
				case "ParticipatorCollection":
					value.put("value", JSON.toJSONString(setParticipatorCollection(node)));
					break;
				default:
					break;
			}
			props.put(k, value);
		}
	}

	/***
	 * 添加属性到json中
	 * @param element 节点
	 * @param props 属性json
	 * @param attributeName 属性名
	 */
	private void setAttributeToJson(Element element, JSONObject props, String attributeName) {
		JSONObject value = new JSONObject();
		value.put("value", element.attributeValue(attributeName));
		props.put(attributeName, value);
	}

	/***
	 * 设置公共属性
	 * @param element 当前元素
	 * @return 设置后的属性对象
	 */
	private JSONObject setCommonAttributes(Element element) {
		JSONObject props = new JSONObject();
		List<String> keyList = Arrays.asList("NodeType", "ID", ConstantUtil.KEY, ConstantUtil.CAPTION);
		JSONObject value;
		for (String k : keyList) {
			if ("NodeType".equals(k)) {
				value = new JSONObject();
				value.put("value", element.getName());
				props.put("NodeType", value);
			} else {
				value = new JSONObject();
				value.put("value", element.attributeValue(k));
				props.put(k, value);
			}
		}
		return props;
	}

	/***
	 * 设置属性"Image、PastImage、LastImage、BPMState、WorkitemRevoke、RetreatCaption、RevokeTrigger、IsSync"值,从xml获取数据
	 * @param props 目标属性json对象
	 * @param node 源xml元素
	 */
	private void setImageAndSoOnAttrbutesFromXml(JSONObject props, Element node) {
		List<String> keyList = Arrays.asList("Image", "PastImage", "LastImage", "BPMState", "WorkitemRevoke",
				"RetreatCaption", "RevokeTrigger", "IsSync");
		JSONObject value;
		for (String key : keyList) {
			switch (key) {
				case "Image":
				case "PastImage":
				case "LastImage":
					value = new JSONObject();
					value.put("value", node.element("NodeGraphic").attributeValue(key));
					props.put(key, value);
					break;
				case "BPMState":
				case "RetreatCaption":
				case "RevokeTrigger":
					value = new JSONObject();
					value.put("value", node.attributeValue(key));
					props.put(key, value);
					break;
				case "WorkitemRevoke":
					value = new JSONObject();
					if (null == node.attributeValue(key) || StringUtils.isEmpty(node.attributeValue(key))) {
						value.put("value", "true");
					} else {
						value.put("value", node.attributeValue(key));
					}
					props.put(key, value);
					break;
				case "IsSync":
					value = new JSONObject();
					if (null == node.attributeValue(key) || StringUtils.isEmpty(node.attributeValue(key))) {
						value.put("value", "false");
					} else {
						value.put("value", node.attributeValue(key));
					}
					props.put(key, value);
					break;
				default:
					break;
			}
		}
	}

	/***
	 * 设置“用户任务”、"多人任务节点"、"会签节点"、"数据映射"属性json
	 * 注：会签节点没有RevokeCondition属性，其他都有
	 * @param props 属性json
	 * @param node 当前节点
	 */
	private void setUserTaskAttributesFromXml(JSONObject props, Element node, String tagName) {
		List<String> keyList;
		// 用户任务属性列表
		keyList = Arrays.asList("TimerItemCollection", "CreateTrigger", "FinishTrigger",
				"AutoIgnoreNoParticipator", "Perm", ConstantUtil.OPERATION_COLLECTION, "AssistanceCollection",
				"ParticipatorCollection");
		if (NodeKeyAndCaptionEnum.MultiUserTask.getKey().equals(tagName)) {
			keyList = new ArrayList<>(keyList);
			keyList.add("InOrder");
			keyList.add("RevokeCondition");
		} else if (NodeKeyAndCaptionEnum.Countersign.getKey().equals(tagName)) {
			keyList = new ArrayList<>(keyList);
			keyList.add("PassCondition");
			keyList.add("PassType");
			keyList.add("ValidCondition");
			keyList.add("ValidType");
			keyList.add("FinishCondition");
			keyList.add("FinishType");
		} else if (NodeKeyAndCaptionEnum.DataMap.getKey().equals(tagName)) {
			keyList = new ArrayList<>(keyList);
			keyList.add("MidDataMap");
			keyList.add("SyncTriggerType");
			keyList.add("DataMapCondition");
			keyList.add("BillDataMapInfoCollection");
			keyList.add("RevokeCondition");
		} else if (NodeKeyAndCaptionEnum.UserTask.getKey().equals(tagName)) {
			keyList = new ArrayList<>(keyList);
			keyList.add("RevokeCondition");
		}

		JSONObject value;
		for (String k : keyList) {
			value = new JSONObject();
			switch (k) {
				case "TimerItemCollection":
					if (null != node.element(k)) {
						value.put("value", JSON.toJSONString(setTimerItemCollection(node)));
					} else {
						value.put("value", "");
					}
					break;
				case "RevokeCondition":
				case "CreateTrigger":
				case "FinishTrigger":
				case "SyncTriggerType":
				case "DataMapCondition":
					if (null == node.attributeValue(k) || StringUtils.isEmpty(node.attributeValue(k))) {
						value.put("value", "");
					} else {
						value.put("value", node.attributeValue(k));
					}
					break;
				case "AutoIgnoreNoParticipator":
				case "InOrder":
				case "MidDataMap":
					if (null == node.attributeValue(k) || StringUtils.isEmpty(node.attributeValue(k))) {
						value.put("value", "false");
					} else {
						value.put("value", node.attributeValue(k));
					}
					break;
				case "Perm":
					value.put("value", JSON.toJSONString(setPerm(node)));
					break;
				case ConstantUtil.OPERATION_COLLECTION:
					setOperationCollectionJsonFromXml(node, value, k);
					break;
				case "AssistanceCollection":
					setAssistanceCollection(node, value, k);
					break;
				case "ParticipatorCollection":
					value.put("value", JSON.toJSONString(setParticipatorCollection(node)));
					break;
				case "FinishCondition":
				case "PassCondition":
				case "ValidCondition":
					value.put("value", node.attributeValue(k));
					break;
				case "FinishType":
				case "PassType":
				case "ValidType":
					if (Objects.isNull(node.attributeValue(k))) {
						value.put("value", "false");
					} else {
						value.put("value", node.attributeValue(k));
					}
					break;
				case "BillDataMapInfoCollection":
					setBillDataMapInfoCollectionJsonFromXml(node, value, k);
					break;
				default:
					break;
			}
			props.put(k, value);
		}
	}

	/***
	 * 设置”审批任务“和"多人审批节点"和"子流程"自己的属性
	 * @param props 属性json
	 * @param node 当前节点
	 */
	private void setAuditAttributesFromXml(JSONObject props, Element node, String tagName) {
		List<String> keyList = Arrays.asList("TimerItemCollection", "CreateTrigger", "RevokeCondition", "FinishTrigger",
				"AutoIgnoreNoParticipator", "Perm", ConstantUtil.OPERATION_COLLECTION, "AssistanceCollection",
				"ParticipatorCollection");
		if (NodeKeyAndCaptionEnum.SubProcess.getKey().equals(tagName)) {
			keyList = new ArrayList<>(keyList);
			keyList.add("SyncMode");
			keyList.add("AutoStartAction");
			keyList.add("SubProcessKey");
		} else {
			keyList = new ArrayList<>(keyList);
			keyList.add("DenyToLastNode");
			keyList.add("DenyNodeKey");
		}

		JSONObject value;
		for (String k : keyList) {
			value = new JSONObject();
			switch (k) {
				case "TimerItemCollection":
					if (null != node.element(k)) {
						value.put("value", JSON.toJSONString(setTimerItemCollection(node)));
					} else {
						value.put("value", "");
					}
					break;
				case "RevokeCondition":
				case "CreateTrigger":
				case "FinishTrigger":
				case "DenyNodeKey":
				case "SubProcessKey":
				case "PassCondition":
				case "ValidCondition":
				case "FinishCondition":
				case "AutoStartAction":
					if (null == node.attributeValue(k) || StringUtils.isEmpty(node.attributeValue(k))) {
						value.put("value", "");
					} else {
						value.put("value", node.attributeValue(k));
					}
					break;
				case "DenyToLastNode":
				case "AutoIgnoreNoParticipator":
				case "FinishType":
				case "PassType":
				case "ValidType":
					if (null == node.attributeValue(k) || StringUtils.isEmpty(node.attributeValue(k))) {
						value.put("value", "false");
					} else {
						value.put("value", node.attributeValue(k));
					}
					break;
				case "Perm":
					value.put("value", JSON.toJSONString(setPerm(node)));
					break;
				case ConstantUtil.OPERATION_COLLECTION:
					setOperationCollectionJsonFromXml(node, value, k);
					break;
				case "AssistanceCollection":
					setAssistanceCollection(node, value, k);
					break;
				case "ParticipatorCollection":
					value.put("value", JSON.toJSONString(setParticipatorCollection(node)));
					break;
				case "SyncMode":
					if (null == node.attributeValue(k) || StringUtils.isEmpty(node.attributeValue(k))) {
						value.put("value", "Synchronized");
					} else {
						value.put("value", node.attributeValue(k));
					}
					break;
				default:
					break;
			}
			props.put(k, value);
		}
	}

	/***
	 * 设置泳道节点
	 * @param props 属性json
	 * @param node 当前节点
	 */
	private void setSwimline(JSONObject props, Element node) {
		List<String> keyList = Arrays.asList("Direction", "Position", "Size");
		JSONObject value;
		for (String k : keyList) {
			value = new JSONObject();
			switch (k) {
				case "Direction":
					if (null == node.attributeValue(k) || StringUtils.isEmpty(node.attributeValue(k))) {
						value.put("value", "Vertical");
					} else {
						value.put("value", node.attributeValue(k));
					}
					props.put(k, value);
					break;
				case "Position":
					getSplitAttr(props, node, k, ConstantUtil.X, 0);
					getSplitAttr(props, node, k, ConstantUtil.Y, 1);
					break;
				case "Size":
					getSplitAttr(props, node, k, "Width", 0);
					getSplitAttr(props, node, k, ConstantUtil.HEIGHT, 1);
					break;
				default:
					break;
			}
		}
	}

	/***
	 * 获取并设置泳道的X，Y，Width，Height属性值
	 * @param props 属性json
	 * @param node 当前节点
	 * @param k 当前属性
	 * @param attr 子属性，如，X，Y等
	 * @param pos 位置，0或1
	 */
	private void getSplitAttr(JSONObject props, Element node, String k, String attr, int pos) {
		if (node.attributeValue(k) != null) {
			JSONObject value;
			value = new JSONObject();
			value.put("value", node.attributeValue(k).split(",")[pos].split(":")[1]);
			props.put(attr, value);
		}
	}

	/***
	 * 设置”状态节点“属性json
	 * @param props 属性json
	 * @param node 当前节点
	 */
	private void setStateAttributesFromXml(JSONObject props, Element node) {
		List<String> keyList = Arrays.asList("Status", "CreateTrigger", "FinishTrigger", "Perm", ConstantUtil.FIELD_KEY,
				"FieldValue", "ParticipatorCollection", "RevokeCondition");
		JSONObject value;
		for (String k : keyList) {
			value = new JSONObject();
			switch (k) {
				case "Status":
				case "CreateTrigger":
				case "FinishTrigger":
				case ConstantUtil.FIELD_KEY:
				case "FieldValue":
				case "RevokeCondition":
					if (null == node.attributeValue(k) || StringUtils.isEmpty(node.attributeValue(k))) {
						value.put("value", "");
					} else {
						value.put("value", node.attributeValue(k));
					}
					break;
				case "Perm":
					value.put("value", JSON.toJSONString(setPerm(node)));
					break;
				case "ParticipatorCollection":
					value.put("value", JSON.toJSONString(setParticipatorCollection(node)));
					break;
				default:
					break;
			}
			props.put(k, value);
		}
	}

	/***
	 * 设置辅助节点集合json来自xml文件
	 * @param node 当前节点
	 * @param value 当前节点的值
	 * @param k 操作集合key，值为"AssistanceCollection"
	 */
	private void setAssistanceCollection(Element node, JSONObject value, String k) {
		JSONArray jsonArray = new JSONArray();
		if (null != node.element(k)) {
			// 用户任务列表
			List<Element> userTasks = node.element(k).elements("UserTask");
			// 会签节点列表
			List<Element> countersigns = node.element(k).elements("Countersign");
			// 审批任务列表
			List<Element> audits = node.element(k).elements("Audit");

			List<String> keys = Arrays.asList("id", "NodeType", ConstantUtil.KEY, ConstantUtil.CAPTION, "Perm", ConstantUtil.OPERATION_COLLECTION,
					"ParticipatorCollection", "TimerItemCollection", "OtherSet");
			if (!CollectionUtils.isEmpty(userTasks)) {
				setAssistanceItem(jsonArray, userTasks, keys, "UserTask");
			}

			if (!CollectionUtils.isEmpty(countersigns)) {
				setAssistanceItem(jsonArray, countersigns, keys, "Countersign");
			}

			if (!CollectionUtils.isEmpty(audits)) {
				setAssistanceItem(jsonArray, audits, keys, "Audit");
			}

		}
		value.put("value", JSON.toJSONString(jsonArray));
	}

	/***
	 * 设置辅助节点集合明细
	 * @param jsonArray json数组
	 * @param tasks 元素列表
	 * @param keys 关键字列表
	 * @param nodeType 节点类型：UserTask(用户任务),Countersign(会签节点),Audit(审批任务)
	 */
	private void setAssistanceItem(JSONArray jsonArray, List<Element> tasks, List<String> keys, String nodeType) {
		JSONObject jsonObj;
		JSONObject otherSetJsonObj;
		for (int i = 0; i < tasks.size(); i++) {
			jsonObj = new JSONObject();
			for (String key : keys) {
				switch (key) {
					case "id":
						jsonObj.put(key, (jsonArray.size() + i + 1));
						break;
					case "NodeType":
						jsonObj.put(key, nodeType);
						break;
					case ConstantUtil.KEY:
					case ConstantUtil.CAPTION:
						jsonObj.put(key, tasks.get(i).attributeValue(key));
						break;
					case "Perm":
						jsonObj.put(key, JSON.toJSONString(setPerm(tasks.get(i))));
						break;
					case ConstantUtil.OPERATION_COLLECTION:
						setOperationCollectionJsonFromXml(tasks.get(i), jsonObj, key);
						jsonObj.put(key, jsonObj.getString("value"));
						break;
					case "ParticipatorCollection":
						jsonObj.put(key, JSON.toJSONString(setParticipatorCollection(tasks.get(i))));
						break;
					case "TimerItemCollection":
						jsonObj.put(key, JSON.toJSONString(setTimerItemCollection(tasks.get(i))));
						break;
					case "OtherSet":
						List<String> otherSetKeys = Arrays.asList("ProcessKey", "TemplateKey", "CreateTrigger",
								"FinishTrigger", "PassCondition", "ValidCondition", "FinishCondition", "InUse",
								"AutoIgnoreNoParticipator", "PassType", "ValidType");
						otherSetJsonObj = new JSONObject();
						String otherSetValue;
						for (String otherSetKey : otherSetKeys) {
							otherSetValue = tasks.get(i).attributeValue(otherSetKey);
							if (StringUtils.isNotEmpty(otherSetValue)) {
								otherSetJsonObj.put(otherSetKey, otherSetValue);
							} else {
								otherSetJsonObj.put(otherSetKey, "");
							}
						}
						jsonObj.put(key, otherSetJsonObj);
						break;
					default:
						break;
				}
			}
			jsonArray.add(jsonObj);
		}
	}

	/***
	 * 设置操作集合json来自xml文件
	 * @param node 当前节点
	 * @param value 当前节点的值
	 * @param k 操作集合key，值为ConstantUtil.OPERATION_COLLECTION
	 */
	private void setOperationCollectionJsonFromXml(Element node, JSONObject value, String k) {
		if (null != node.element(k) && null != node.element(k).elements(ConstantUtil.OPERATION)) {
			JSONArray jsonArray = new JSONArray();
			JSONObject jsonObj;
			String actionValue = "";
			List<Element> operations = node.element(k).elements(ConstantUtil.OPERATION);
			List<String> keys = Arrays.asList(ConstantUtil.ACTION, ConstantUtil.KEY, ConstantUtil.CAPTION, ConstantUtil.ENABLE,
					ConstantUtil.VISIBLE, "Icon", "TemplateKey", "id");
			int count = 0;
			for (Element ele : operations) {
				jsonObj = new JSONObject();
				for (String key : keys) {
					switch (key) {
						case ConstantUtil.ACTION:
							if (null != ele.element(key)) {
								actionValue = ConvertSpecialStringUtil
										.convertSpecialString(ele.element(key).getData().toString().trim());
							}
							jsonObj.put(key, actionValue);
							break;
						case ConstantUtil.ENABLE:
							jsonObj.put(key, StringUtils.isNotEmpty(ele.attributeValue(key)) ?
									ele.attributeValue(key) : "ReadOnly()");
							break;
						case ConstantUtil.KEY:
						case ConstantUtil.CAPTION:
						case ConstantUtil.VISIBLE:
						case "Icon":
						case "TemplateKey":
							jsonObj.put(key, ele.attributeValue(key));
							break;
						case "id":
							jsonObj.put(key, ++count);
							break;
						default:
							break;
					}
				}
				jsonArray.add(jsonObj);
			}
			value.put("value", JSON.toJSONString(jsonArray));
		} else {
			value.put("value", "");
		}
	}

	/***
	 * 设置映射集合json来自xml文件
	 * @param node 当前节点
	 * @param value 当前节点的值
	 * @param k 映射集合key，值为"BillDataMapInfoCollection"
	 */
	private void setBillDataMapInfoCollectionJsonFromXml(Element node, JSONObject value, String k) {
		if (null != node.element(k) && null != node.element(k).elements("BillDataMapInfo")) {
			JSONArray jsonArray = new JSONArray();
			JSONObject jsonObj;
			List<Element> billDataMapInfos = node.element(k).elements("BillDataMapInfo");
			List<String> keys = Arrays.asList("DataMapKey", ConstantUtil.CAPTION, "AutoStart", ConstantUtil.FORM_KEY);
			int count = 0;
			for (Element ele : billDataMapInfos) {
				jsonObj = new JSONObject();
				for (String key : keys) {
					switch (key) {
						case "AutoStart":
							jsonObj.put(key, null != ele.attributeValue(key));
							break;
						case "DataMapKey":
						case ConstantUtil.CAPTION:
						case ConstantUtil.FORM_KEY:
							jsonObj.put(key, ele.attributeValue(key));
							break;
						case "id":
							jsonObj.put(key, ++count);
							break;
						default:
							break;
					}
				}
				jsonArray.add(jsonObj);
			}
			value.put("value", JSON.toJSONString(jsonArray));
		} else {
			value.put("value", "");
		}
	}

	/***
	 * 设置审批超时集合
	 * @param node 节点
	 * @return 审批超时集合的json数组设置
	 */
	private JSONArray setTimerItemCollection(Element node) {
		List<Element> timerAutoPassItems = null;
		List<Element> timerAutoDenyItems = null;
		List<Element> timerItemItems = null;
		if (null != node.element("TimerItemCollection") &&
				!CollectionUtils.isEmpty(node.element("TimerItemCollection").elements("TimerAutoPass"))) {
			timerAutoPassItems = node.element("TimerItemCollection").elements("TimerAutoPass");
		}
		if (null != node.element("TimerItemCollection") &&
				!CollectionUtils.isEmpty(node.element("TimerItemCollection").elements("TimerAutoDeny"))) {
			timerAutoDenyItems = node.element("TimerItemCollection").elements("TimerAutoDeny");
		}
		if (null != node.element("TimerItemCollection") &&
				!CollectionUtils.isEmpty(node.element("TimerItemCollection").elements("TimerItem"))) {
			timerItemItems = node.element("TimerItemCollection").elements("TimerItem");
		}

		List<String> keyList = Arrays.asList("TimerAutoPass", "TimerAutoDeny", "TimerItem");
		JSONArray array = new JSONArray();
		List<Element> elements = null;
		// "自动通过","自动驳回","循环自动任务","单次自动任务"
		String type = "";
		for (String k : keyList) {
			switch (k) {
				case "TimerAutoPass":
					type = "自动通过";
					elements = timerAutoPassItems;
					break;
				case "TimerAutoDeny":
					type = "自动驳回";
					elements = timerAutoDenyItems;
					break;
				case "TimerItem":
					type = "";
					elements = timerItemItems;
					break;
				default:
					break;
			}
			if (CollectionUtils.isEmpty(elements)) {
				continue;
			}
			JSONObject jsonObj;
			for (Element ele : elements) {
				jsonObj = new JSONObject();
				jsonObj.put("id", array.size() + 1);
				jsonObj.put(ConstantUtil.KEY, ele.attributeValue(ConstantUtil.KEY));
				jsonObj.put("Peroid", ele.attributeValue("Peroid"));

				if (StringUtils.isNotEmpty(type)) {
					jsonObj.put("type", type);
					if (null != ele.attribute("UserInfo")) {
						jsonObj.put("Suggestion", ele.attributeValue("UserInfo"));
					} else {
						jsonObj.put("Suggestion", "");
					}
					jsonObj.put("CycleInterval", "");
					jsonObj.put("Repeat", "");
				} else {
					if (null != ele.attribute("Repeat")) {
						jsonObj.put("type", "循环自动任务");
						if (null != ele.attribute("Repeat")) {
							jsonObj.put("Repeat", ele.attributeValue("Repeat"));
						} else {
							jsonObj.put("Repeat", "");
						}
					} else {
						jsonObj.put("type", "单次自动任务");
					}
					if (null != ele.attribute("CycleInterval")) {
						jsonObj.put("CycleInterval", ele.attributeValue("CycleInterval"));
					} else {
						jsonObj.put("CycleInterval", "");
					}
					if (null != ele.attribute("Trigger")) {
						jsonObj.put("Suggestion", ele.attributeValue("Trigger"));
					} else {
						jsonObj.put("Suggestion", "");
					}
				}
				array.add(jsonObj);
			}
		}
		return array;
	}

	/***
	 * 设置任务的权限设置
	 * @param node 节点
	 * @return 权限的json设置
	 */
	private JSONObject setPerm(Element node) {
		List<Element> optPermItems = null;
		List<Element> visiblePermItems = null;
		List<Element> enablePermItems = null;
		if (null != node.element("Perm") && null != node.element("Perm").element("OptPerm")) {
			optPermItems = node.element("Perm").element("OptPerm").elements("OptPermItem");
		}
		if (null != node.element("Perm") && null != node.element("Perm").element("VisiblePerm")) {
			visiblePermItems = node.element("Perm").element("VisiblePerm").elements("VisiblePermItem");
		}
		if (null != node.element("Perm") && null != node.element("Perm").element("EnablePerm")) {
			enablePermItems = node.element("Perm").element("EnablePerm").elements("EnablePermItem");
		}

		JSONObject value = new JSONObject();
		List<String> keyList = Arrays.asList("OptPerm", "VisiblePerm", "EnablePerm");
		JSONArray array;
		List<Element> elements = null;
		for (String k : keyList) {
			switch (k) {
				case "OptPerm":
					elements = optPermItems;
					break;
				case "VisiblePerm":
					elements = visiblePermItems;
					break;
				case "EnablePerm":
					elements = enablePermItems;
					break;
				default:
					break;
			}
			if (!CollectionUtils.isEmpty(elements)) {
				array = new JSONArray();
				JSONObject jsonObj;
				for (Element ele : elements) {
					jsonObj = new JSONObject();
					jsonObj.put(ConstantUtil.KEY, ele.attributeValue(ConstantUtil.KEY));
					array.add(jsonObj);
				}
				value.put(k, array);
			}
		}
		return value;
	}

	/***
	 * 设置参与者集合
	 * @param node 节点
	 * @return 参与者集合的json设置
	 */
	private JSONObject setParticipatorCollection(Element node) {
		List<Element> dictionaryItems = null;
		List<Element> queryItems = null;
		List<Element> midFormulaItems = null;
		if (null != node.element("ParticipatorCollection") &&
				null != node.element("ParticipatorCollection").elements("Dictionary")) {
			dictionaryItems = node.element("ParticipatorCollection").elements("Dictionary");
		}
		if (null != node.element("ParticipatorCollection") &&
				null != node.element("ParticipatorCollection").elements("Query")) {
			queryItems = node.element("ParticipatorCollection").elements("Query");
		}
		if (null != node.element("ParticipatorCollection") &&
				null != node.element("ParticipatorCollection").elements("MidFormula")) {
			midFormulaItems = node.element("ParticipatorCollection").elements("MidFormula");
		}

		JSONObject value = new JSONObject();
		List<String> keyList = Arrays.asList("Dictionary", "Query", "MidFormula");
		JSONArray jsonArray;
		JSONObject jsonObj;
		List<Element> queryParameterEles;
		for (String k : keyList) {
			jsonArray = new JSONArray();
			switch (k) {
				case "Dictionary":
					if (!CollectionUtils.isEmpty(dictionaryItems)) {
						// 字典标识：操作员、角色
						String itemKey;
						// 字典ID
						String[] itemIdArray;
						// 字典Code
						String[] codeArray;
						for (Element ele : dictionaryItems) {
							itemKey = ele.attributeValue("ItemKey");
							if (null != ele.attributeValue("ItemID")) {
								itemIdArray = ele.attributeValue("ItemID").split(":");
								if (itemIdArray.length > 0) {
									setDictionaryJson(jsonArray, itemKey, itemIdArray, "ItemID");
								}
							}
							if (null != ele.attributeValue("Code")) {
								codeArray = ele.attributeValue("Code").split(":");
								if (codeArray.length > 0) {
									setDictionaryJson(jsonArray, itemKey, codeArray, "Code");
								}
							}
						}
					}
					break;
				case "Query":
					if (!CollectionUtils.isEmpty(queryItems)) {
						int outCount = 0;
						for (Element ele : queryItems) {
							jsonObj = new JSONObject();
							jsonObj.put("id", ++outCount);
							// sql语句
							jsonObj.put("data", ConvertSpecialStringUtil
									.convertSpecialString(ele.getData().toString().trim()));
							if (null != ele.element("QueryParameterCollection")) {
								queryParameterEles = ele.element("QueryParameterCollection")
										.elements("QueryParameter");
								JSONObject tempObj;
								JSONArray jsonArrayTemp = new JSONArray();
								int count = 0;
								for (Element e : queryParameterEles) {
									tempObj = new JSONObject();
									tempObj.put("DataType", e.attributeValue("DataType"));
									tempObj.put("Formula", e.attributeValue("Formula")
											.replaceAll("<br>", "@br@"));
									tempObj.put("id", ++count);
									jsonArrayTemp.add(tempObj);
								}
								// 给QueryParameterCollection赋值
								jsonObj.put("QueryParameterCollection", jsonArrayTemp.toString());
							}
							jsonArray.add(jsonObj);
						}
					}
					break;
				case "MidFormula":
					if (!CollectionUtils.isEmpty(midFormulaItems)) {
						int count = 0;
						for (Element ele : midFormulaItems) {
							jsonObj = new JSONObject();
							jsonObj.put("value", ConvertSpecialStringUtil
									.convertSpecialString(ele.getData().toString().trim()));
							jsonObj.put("id", ++count);
							jsonArray.add(jsonObj);
						}
					}
					break;
				default:
					break;
			}
			value.put(k, jsonArray);
		}
		return value;
	}

	/***
	 * 设置Dictionary json值
	 * @param jsonArray 生成的Dictionary json数组
	 * @param itemKey 类型：操作员或角色
	 * @param itemIdArray 字典ID数组
	 * @param itemID 字典ID
	 */
	private void setDictionaryJson(JSONArray jsonArray, String itemKey, String[] itemIdArray, String itemID) {
		JSONObject jsonObj;
		for (String s : itemIdArray) {
			jsonObj = new JSONObject();
			jsonObj.put("id", (jsonArray.size() + 1));
			jsonObj.put("dictType", itemKey);
			jsonObj.put("codeType", itemID);
			jsonObj.put("codeValue", s);
			jsonArray.add(jsonObj);
		}
	}

	/***
	 * 处理节点之间的连线
	 * @param element 连线的节点
	 * @param paths 路径json对象
	 */
	private void hendleLines(Element element, JSONObject paths) {
		// 获取元素下的连线
		Element transitionCollection = element.element("TransitionCollection");
		if (null != transitionCollection) {
			List<String> keys = Arrays.asList("SequenceFlow", "Association", "ExceptionFlow");
			for (String key : keys) {
				List<Element> pathList = transitionCollection.elements(key);
				for (Element path : pathList) {
					if (null != path) {
						this.setPathsJson(path, paths);
					}
				}
			}
		}
	}

	/***
	 * 设置paths json值
	 * @param element SequenceFlow节点
	 * @param paths 路径json对象
	 */
	private void setPathsJson(Element element, JSONObject paths) {
		JSONObject pathsInner = new JSONObject();
		JSONObject temp;
		String from;
		String to = "";
		int x = 150;
		int y = 150;
		from =  element.getParent().getParent().attributeValue("Key");
		String toKey = element.attributeValue("TargetNodeKey");
		Element root = element.getParent().getParent().getParent();
		List<Element> elements = root.elements();
		// 线的开始节点Key
		String fromNodeKey = element.getParent().getParent().attributeValue(ConstantUtil.KEY);
		for (Element elem : elements) {
			if (StringUtils.isNotEmpty(elem.attributeValue(ConstantUtil.KEY))
					&& elem.attributeValue(ConstantUtil.KEY).equals(fromNodeKey)) {
				x = Integer.parseInt(elem.element("NodeGraphic").attributeValue(ConstantUtil.X));
				y = Integer.parseInt(elem.element("NodeGraphic").attributeValue(ConstantUtil.Y));
			}
			if (StringUtils.isNotEmpty(elem.attributeValue(ConstantUtil.KEY))
					&& elem.attributeValue(ConstantUtil.KEY).equals(toKey)) {
				to =  elem.attributeValue("Key");
			}
		}
		pathsInner.put("from", from);
		pathsInner.put("to", to);
		pathsInner.put("dots", new JSONArray());
		JSONObject textJson = new JSONObject();
		textJson.put("text", "");
		pathsInner.put("text", textJson);
		temp = new JSONObject();
		temp.put(ConstantUtil.x, x);
		temp.put(ConstantUtil.y, y);
		pathsInner.put("textPos", temp);
		// {props:{NodeType:{value:''},ID:{value:''},Key:{value:''},
		//         Caption:{value:''},TargetNodeKey:{value:''},State:{value:''},Condition:{value:''}}}
		JSONObject props = new JSONObject();
		// props中的所有Key
		List<String> keys = null;
		switch (element.getName()) {
			case "SequenceFlow":
				// 只有Audit或MultiAudit才有Condition属性
				if (isAuditOrMultiAudit(root, fromNodeKey)) {
					keys = Arrays.asList("NodeType", "ID", ConstantUtil.KEY, ConstantUtil.CAPTION, "TargetNodeKey", "State", "Condition");
				} else {
					keys = Arrays.asList("NodeType", "ID", ConstantUtil.KEY, ConstantUtil.CAPTION, "TargetNodeKey", "State");
				}
				break;
			case "Association":
			case "ExceptionFlow":
				keys = Arrays.asList("NodeType", "ID", ConstantUtil.KEY, ConstantUtil.CAPTION, "TargetNodeKey");
				break;
			default:
				break;
		}

		if (!CollectionUtils.isEmpty(keys)) {
			keys.forEach(key -> this.setPathsProps(key, element, props));
		}
		pathsInner.put("props", props);
		paths.put("path" + element.attributeValue("Key"), pathsInner);
	}

	/***
	 * 是否是审批节点或多人审批节点 或者选择节点
	 * @param root 根节点
	 * @param fromKey 源节点Key
	 * @return true：是；false：否
	 */
	private boolean isAuditOrMultiAudit(Element root, String fromKey) {
		AtomicBoolean result = new AtomicBoolean(false);
		root.elements().forEach(q -> {
			boolean isAuditOrMultiAudit = fromKey.equals(q.attributeValue(ConstantUtil.KEY))
					&& (NodeKeyAndCaptionEnum.Audit.getKey().equals(q.getName()) ||
					NodeKeyAndCaptionEnum.MultiAudit.getKey().equals(q.getName())
					|| NodeKeyAndCaptionEnum.Decision.getKey().equals(q.getName())
					|| NodeKeyAndCaptionEnum.Countersign.getKey().equals(q.getName()));
			if (isAuditOrMultiAudit) {
				result.set(true);
			}
		});
		return result.get();
	}

	/***
	 * 设置paths的props属性设置
	 * @param key 键值
	 * @param element 操作的节点元素
	 * @param props 属性对象
	 */
	private void setPathsProps(String key, Element element, JSONObject props) {
		JSONObject value = new JSONObject();
		String v = element.attributeValue(key);
		switch (key) {
			case "NodeType":
				value.put("value", element.getName());
				break;
			case "ID":
			case ConstantUtil.KEY:
			case ConstantUtil.CAPTION:
			case "TargetNodeKey":
				value.put("value", v);
				break;
			case "State":
			case "Condition":
				if (StringUtils.isNotEmpty(v)) {
					value.put("value", v);
				} else {
					value.put("value", "true");
				}
				break;
			default:
				break;
		}
		props.put(key, value);
	}

	//返回集合
	public Stack<String> resultStack() {
		return bpmStack;
	}
}
