package com.bokesoft.yes.design.bpm.po;

import com.alibaba.fastjson.JSONObject;
import com.bokesoft.erp.WebDesignerConfiguration;
import com.bokesoft.yes.design.cmd.XmlFileProcessor;
import com.bokesoft.yes.design.enums.NodeKeyAndCaptionEnum;
import com.bokesoft.yes.design.constant.ConstantUtil;
import com.bokesoft.yes.design.utils.XMLWriter;
import com.bokesoft.yes.helper.FilePathHelper;
import org.apache.commons.lang3.exception.ExceptionUtils;
import org.apache.commons.lang3.StringUtils;
import org.dom4j.Attribute;
import org.dom4j.Document;
import org.dom4j.Element;
import org.dom4j.io.OutputFormat;
import org.dom4j.io.SAXReader;
import org.apache.commons.collections.CollectionUtils;

import java.io.File;
import java.io.FileOutputStream;
import java.nio.file.Paths;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.logging.Logger;

/***
 * 路径
 * @date 2020-10-14
 * @author zhsy
 */
public class Path {
	private static final Logger logger = Logger.getLogger(Path.class.getName());

	/***节点类型：SequenceFlow，Association，ExceptionFlow*/
	private String NodeType;
	/***ID属性*/
	private String ID;
	/***Key属性*/
	private String Key;
	/***Caption属性*/
	private String Caption;
	/***目标节点Key属性*/
	private String TargetNodeKey;
	/***状态属性*/
	private String State = "";
	/***条件属性*/
	private String Condition = "";

	/***
	 * 保存
	 * @param filePath 文件路径
	 * @param frontFullJson 前端传过来的全部json
	 * @param nodeId 节点id
	 * @param operNodeJson 操作节点json
	 * @return 处理后的json
	 * @throws Exception 异常
	 */
	public JSONObject save(String filePath, JSONObject frontFullJson, String nodeId, JSONObject operNodeJson)
			throws Throwable {
		JSONObject resultJson = new JSONObject();
		FileOutputStream fileOutputStream = null;
		XMLWriter writer = null;
		try {
			//获取临时文件路径
			String tempPath = XmlFileProcessor.instance.getTmpFile(filePath);
			if (StringUtils.isBlank(tempPath)){
				tempPath = filePath;
			}
			// 创建SAX读取器
			SAXReader reader = new SAXReader();
			// 加载文档
			Document document = reader.read(new File(tempPath));
			// 获取根节点
			Element root = document.getRootElement();

			// 来源节点Key
			String fromNodeKey = operNodeJson.getString("from");
			// 获取实际的来源节点Key
			String from = frontFullJson.getJSONObject("states").getJSONObject(fromNodeKey)
					.getJSONObject("props").getJSONObject(ConstantUtil.KEY).getString("value");
			// 目标节点Key
			String toNodeKey = operNodeJson.getString("to");
			// 获取实际的目标节点Key
			String to = frontFullJson.getJSONObject("states").getJSONObject(toNodeKey)
					.getJSONObject("props").getJSONObject(ConstantUtil.KEY).getString("value");
			// "标签名"也即"节点类型"
			String tagName = operNodeJson.getJSONObject("props").getJSONObject("NodeType").getString("value");
			// 获取ID
			String id = operNodeJson.getJSONObject("props").getJSONObject("ID").getString("value");
			id = StringUtils.isNotEmpty(id) ? id : nodeId;
			id = id.replace("path", "");
			// 获取Key
			String key = operNodeJson.getJSONObject("props").getJSONObject(ConstantUtil.KEY).getString("value");
			key = StringUtils.isEmpty(key) ? nodeId : key;

			// 保存数据到xml中
			Path path = this.saveAttributesIntoXml(nodeId, operNodeJson, root, from, to, tagName, key);

			// 格式化输出流，同时指定编码格式。也可以在FileOutputStream中指定。
			OutputFormat format = getOutputFormat();
			String solutionPath = FilePathHelper.getWorkspacePath() + File.separator;
			String newFilePath = Paths.get(WebDesignerConfiguration.getDesignerDataPath(), "tmp", filePath.substring(solutionPath.length()).replace(File.separator, "__") + "." + System.currentTimeMillis()).toString();
			fileOutputStream = new FileOutputStream(newFilePath);
			writer = new XMLWriter(fileOutputStream, format);
			XmlFileProcessor.stackput(filePath,newFilePath);
			writer.write(document);
			// 设置前端默认值显示，这里对新增节点有用
			Map<String, String> commonAttributesMap
					= this.setCommonAttributesToJson(tagName, id, key, path.getCaption(), to);
			if (NodeKeyAndCaptionEnum.SequenceFlow.getKey().equals(tagName)) {
				commonAttributesMap.put("State", path.getState());
				// 审批任务和多人审批节点，Condition默认值为true，其他的节点没有这个Condition属性
				if (isAuditOrMultiAudit(root, from)) {
					commonAttributesMap.put("Condition", path.getCondition());
				}
			}

			//将内容写入json中，并返回前端页面属性框中显示
			this.setProps(commonAttributesMap, resultJson);
		} catch (Throwable e) {
			logger.warning("保存异常，异常为:" + ExceptionUtils.getStackTrace(e));
			throw e;
		} finally {
			this.closeFileStream(fileOutputStream, writer);
		}
		return resultJson;
	}

	/***
	 * 是否是审批节点或多人审批节点
	 * @param root 根节点
	 * @param fromKey 源节点Key
	 * @return true：是；false：否
	 */
	private boolean isAuditOrMultiAudit(Element root, String fromKey) {
		List<Element> elements = root.elements();
		for (Element element : elements) {
			boolean isAuditOrMultiAudit = fromKey.equals(element.attributeValue(ConstantUtil.KEY))
					&& (NodeKeyAndCaptionEnum.Audit.getKey().equals(element.getName()) ||
					NodeKeyAndCaptionEnum.MultiAudit.getKey().equals(element.getName())
					|| NodeKeyAndCaptionEnum.Decision.getKey().equals(element.getName())
					|| NodeKeyAndCaptionEnum.Countersign.getKey().equals(element.getName()));
			if (isAuditOrMultiAudit) {
				return true;
			}
		}
		return false;
	}

	/***
	 * 保存属性到xml文件中
	 * @param nodeId 节点id
	 * @param operNodeJson 操作节点json
	 * @param root 根节点
	 * @param from 源节点
	 * @param to 目标节点
	 * @param tagName 节点类型
	 * @param key 标识
	 * @return Path对象
	 */
	private Path saveAttributesIntoXml(String nodeId, JSONObject operNodeJson, Element root, String from, String to,
									   String tagName, String key) {
		Path path = new Path();
		// 获取根节点下的所有元素,若当前路径存在，先删除后添加
		List<Element> elements = root.elements();
		for (Element element : elements) {
			if (from.equals(element.attributeValue(ConstantUtil.KEY))) {
				Element transitionCollectionEle = element.element("TransitionCollection");
				if (null != transitionCollectionEle) {
					List<Element> sequenceFlowEleList = transitionCollectionEle.elements(tagName);
					if (null != sequenceFlowEleList && !sequenceFlowEleList.isEmpty()) {
						for (Element e : sequenceFlowEleList) {
							// 删除xml中已经存在的当前节点
							if (key.equals(e.attributeValue(ConstantUtil.KEY))) {
								transitionCollectionEle.remove(e);
							}
						}
					}
				} else {
					transitionCollectionEle = element.addElement("TransitionCollection");
				}

				// 设置元素公共部分
				Element node = this.setCommonAttributes(operNodeJson, transitionCollectionEle, tagName, nodeId, to);
				if (null != node) {
					path.setCaption(node.attributeValue(ConstantUtil.CAPTION));
				}

				// 只有SequenceFlow才有属性"State、Condition"
				if (!NodeKeyAndCaptionEnum.SequenceFlow.getKey().equals(tagName)) {
					return path;
				}

				// 设置属性"State、Condition"
				JSONObject propsJsonObj = operNodeJson.getJSONObject("props");
				Set<String> keys = propsJsonObj.keySet();
				String v = "";
				for (String k : keys) {
					v = propsJsonObj.getJSONObject(k).getString("value");
					if ("null".equals(v)) {
						v = "";
					}

					switch (k) {
						case "State":
							if (!"true".equalsIgnoreCase(v) && StringUtils.isNotEmpty(v)) {
								this.saveAttributeToXml(node, propsJsonObj, k);
								path.setState(v);
							} else {
								this.deleteXmlElementAttribute(node, node.attribute(k));
							}
							break;
						case "Condition":
							// 是Audit或MultiAudit，才有Condition属性设置
							if (isAuditOrMultiAudit(root, from)) {
								if (!"true".equalsIgnoreCase(v) && StringUtils.isNotEmpty(v)) {
									node.addAttribute(k, v);
									path.setCondition(v);
								} else {
									this.deleteXmlElementAttribute(node, node.attribute(k));
								}
							}
							break;
						default:
							break;
					}
				}

			}
		}
		return path;
	}

	/***
	 * 设置公共属性
	 * @param operNodeJson 操作的节点json
	 * @param element 元素节点
	 * @param tagName 待设置的元素标签名
	 * @param nodeId 节点id，也即xml中的属性Key
	 * @param targetNodeKey 目标节点Key
	 * @return 设置后的元素
	 */
	private Element setCommonAttributes(JSONObject operNodeJson, Element element, String tagName,
										String nodeId, String targetNodeKey) {
		// id
		String id = nodeId.replace("rect", "").replace("path", "");
		// 获取Key
		String key = operNodeJson.getJSONObject("props").getJSONObject(ConstantUtil.KEY).getString("value");
		key = StringUtils.isEmpty(key) ? nodeId : key;
		// caption
		String caption = operNodeJson.getJSONObject("props").getJSONObject(ConstantUtil.CAPTION).getString("value");

		// 设置默认的Caption
		if (StringUtils.isEmpty(caption)) {
			caption = NodeKeyAndCaptionEnum.valueOf(tagName).getCaption();
		}

		// 查找TransitionCollection节点下的当前路径
		Element node = null;
		List<Element> elements = element.elements(tagName);
		if (!CollectionUtils.isEmpty(elements)) {
			for (Element ele : elements) {
				if (key.equals(ele.attributeValue(ConstantUtil.KEY))) {
					node = ele;
					break;
				}
			}
		}

		if (null == node) {
			node = element.addElement(tagName);
		}

		// 保存ID,Key,Caption到xml中
		return node.addAttribute("ID", id).addAttribute(ConstantUtil.KEY, key)
				.addAttribute(ConstantUtil.CAPTION, caption).addAttribute("TargetNodeKey", targetNodeKey);
	}

	/***
	 * 保存属性到xml文件中
	 * @param node 当前node节点
	 * @param propsJsonObj 前端传过来的属性及对应的值json
	 * @param attribute 属性
	 */
	private void saveAttributeToXml(Element node, JSONObject propsJsonObj, String attribute) {
		String v = propsJsonObj.getJSONObject(attribute).getString("value");
		if (StringUtils.isNotEmpty(v)) {
			node.addAttribute(attribute, v);
		} else {
			deleteXmlElementAttribute(node, node.attribute(attribute));
		}
	}

	/***
	 * 设置公共属性的json，给前端使用
	 * @param tagName 节点名称
	 * @param id id属性
	 * @param key key属性
	 * @param caption caption顺序
	 * @param targetNodeKey 目标节点Key
	 * @return 设置后的map
	 */
	private Map<String, String> setCommonAttributesToJson(String tagName, String id, String key,
														  String caption, String targetNodeKey) {
		// 设置公共属性值啊
		Map<String, String> commonAttributesMap = new HashMap<>(7);
		commonAttributesMap.put("NodeType", tagName);
		commonAttributesMap.put("ID", id);
		commonAttributesMap.put(ConstantUtil.KEY, key);
		commonAttributesMap.put(ConstantUtil.CAPTION, caption);
		commonAttributesMap.put("TargetNodeKey", targetNodeKey);
		return commonAttributesMap;
	}

	/***
	 * 删除xml元素属性
	 * @param element 元素
	 * @param attribute 属性
	 */
	private void deleteXmlElementAttribute(Element element, Attribute attribute) {
		if (null != attribute) {
			element.remove(attribute);
		}
	}

	/***
	 * 设置属性值
	 * @param commonAttributesMap 公共属性map
	 * @param nodeJson 节点json
	 */
	private void setProps(Map<String, String> commonAttributesMap, JSONObject nodeJson) {
		String key;
		JSONObject v;
		for (Map.Entry<String, String> map : commonAttributesMap.entrySet()) {
			key = map.getKey();
			v = new JSONObject();
			v.put("value", map.getValue());
			nodeJson.put(key, v);
		}
	}

	/***
	 * 写xml格式化
	 * @return 格式化类
	 */
	private OutputFormat getOutputFormat() {
		// 格式化输出流，同时指定编码格式。也可以在FileOutputStream中指定。
		OutputFormat format = OutputFormat.createPrettyPrint();
		// 设置xml标签和Process标签之间没有空行
		format.setNewLineAfterDeclaration(false);
		format.setIndentSize(4);
		format.setExpandEmptyElements(false);
		format.setPadText(false);
		return format;
	}

	/***
	 * 关闭文件流
	 * @param fileOutputStream 文件输出流
	 * @param writer 写对象
	 */
	private void closeFileStream(FileOutputStream fileOutputStream, XMLWriter writer) {
		try {
			if (null != fileOutputStream) {
				fileOutputStream.close();
			}
			if (null != writer) {
				writer.close();
			}
		} catch (Exception e) {
			logger.warning("关闭writer异常，异常为:" + ExceptionUtils.getStackTrace(e));
		}
	}

	public String getNodeType() {
		return NodeType;
	}

	public void setNodeType(String nodeType) {
		NodeType = nodeType;
	}

	public String getID() {
		return ID;
	}

	public void setID(String ID) {
		this.ID = ID;
	}

	public String getKey() {
		return Key;
	}

	public void setKey(String key) {
		Key = key;
	}

	public String getCaption() {
		return Caption;
	}

	public void setCaption(String caption) {
		Caption = caption;
	}

	public String getTargetNodeKey() {
		return TargetNodeKey;
	}

	public void setTargetNodeKey(String targetNodeKey) {
		TargetNodeKey = targetNodeKey;
	}

	public String getState() {
		return State;
	}

	public void setState(String state) {
		State = state;
	}

	public String getCondition() {
		return Condition;
	}

	public void setCondition(String condition) {
		Condition = condition;
	}

}
