package com.bokesoft.yes.design.bpm.po;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.bokesoft.yes.design.bpm.util.ConvertSpecialStringUtil;
import com.bokesoft.yes.design.enums.NodeKeyAndCaptionEnum;
import com.bokesoft.yes.design.constant.ConstantUtil;
import org.apache.commons.lang3.exception.ExceptionUtils;
import org.apache.commons.lang3.StringUtils;
import org.dom4j.Attribute;
import org.dom4j.Element;
import org.apache.commons.collections.CollectionUtils;

import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.logging.Logger;
import java.util.regex.Pattern;

/***
 * 节点
 * @date 2020-10-14
 * @author zhsy
 */
public class Node {
	private static final Logger logger = Logger.getLogger(Node.class.getName());

	/***节点类型*/
	private String NodeType;
	/***ID属性*/
	private String ID;
	/***Key属性*/
	private String Key;
	/***Caption属性*/
	private String Caption;
	/***NodeGraphic元素*/
	private NodeGraphic NodeGraphic;
	/***Image属性*/
	private String Image;
	/***PastImage属性*/
//	private String PastImage;
	/***LastImage属性*/
//	private String LastImage;
	/***BPMState属性*/
	private String BPMState;
	/***WorkitemRevoke属性*/
	private String WorkitemRevoke;
	/***RetreatCaption属性*/
//	private String RetreatCaption;
	/***RevokeTrigger属性*/
//	private String RevokeTrigger;
	/***IsSync属性*/
	private String IsSync;

	/***
	 * 保存到xml文件中
	 * @param operNodeJson 操作节点json
	 * @param nodeId 节点id
	 * @param nodeType 节点类型
	 * @return 操作后的节点元素
	 * @throws Throwable 抛出异常
	 */
	public Element save(JSONObject operNodeJson, Element root, String nodeType, String nodeId) throws Throwable {
		Element node = null;
		try {
			// 设置元素公共部分
			node = this.saveCommonAttributesToXml(operNodeJson, root, nodeType, nodeId);

			// 属性"Image、PastImage、LastImage、BPMState、WorkitemRevoke、RetreatCaption、RevokeTrigger、IsSync"的设置
			this.saveImageAttrbutesToXml(operNodeJson, node);

		} catch (Throwable e) {
			logger.warning("保存公共属性到xml文件异常，异常为:" + ExceptionUtils.getStackTrace(e));
			throw e;
		}
		return node;
	}

	/***
	 * 保存自己的属性到xml文件中
	 * @param operNodeJson 操作节点json
	 * @param node 当前节点
	 */
	public void saveSelfAttributesToXml(JSONObject operNodeJson, Element node) throws Throwable {
	}

	/***
	 * 获取前端默认值设置json
	 * @return 设置后的json
	 */
	public JSONObject setFrontDefaultValues() {
		return new JSONObject();
	}

	/***
	 * 设置元素公共部分
	 * @param operNodeJson 操作的节点json
	 * @param root 根元素
	 * @param nodeType 节点类型
	 * @param nodeId 节点id，也即xml中的属性Key
	 * @return 设置后的元素
	 */
	private static Pattern NUMBER_PATTERN = Pattern.compile("[^0-9]");
	public Element saveCommonAttributesToXml(JSONObject operNodeJson, Element root, String nodeType, String nodeId) {
		// 获取id
		String id = nodeId.replaceAll("rect", "").replaceAll("path", "");
		// 获取Key
		String key = operNodeJson.getJSONObject("props").getJSONObject(ConstantUtil.KEY).getString("value");
		key = StringUtils.isEmpty(key) ? nodeId : key;
		// 获取Caption
		String caption = operNodeJson.getJSONObject("props").getJSONObject(ConstantUtil.CAPTION).getString("value");
		id = NUMBER_PATTERN.matcher(id).replaceAll("").trim();
		// 设置默认的Caption
		if (StringUtils.isEmpty(caption)) {
			caption = NodeKeyAndCaptionEnum.valueOf(nodeType).getCaption();
		}

		Element node = null;
		List<Element> elements = root.elements(nodeType);
		if (!CollectionUtils.isEmpty(elements)) {
			for (Element ele : elements) {
				if (key.equals(ele.attributeValue(ConstantUtil.KEY))) {
					node = ele;
					break;
				}
			}
		}

		if (null == node) {
			node = root.addElement(nodeType);
		}

		// 保存ID，Key，Caption属性到xml文件中
		node.addAttribute("ID", id).addAttribute(ConstantUtil.CAPTION, caption).addAttribute(ConstantUtil.KEY, key);

		// 添加NodeGraphic元素
		if (null == node.element("NodeGraphic")) {
			node.addElement("NodeGraphic");
		}

		// 保存NodeGraphic
		NodeGraphic graphic = saveNodeGraphic(operNodeJson, node);

		// 设置属性到当前对象中，为了默认值显示服务
		this.setNodeType(nodeType);
		this.setID(id);
		this.setCaption(caption);
		this.setKey(key);
		this.setNodeGraphic(graphic);
		return node;
	}

	/***
	 * 设置属性"Image、PastImage、LastImage、BPMState、WorkitemRevoke、RetreatCaption、RevokeTrigger、IsSync"值,从json获取数据
	 * @param operNodeJson 操作json
	 * @param node 当前元素节点
	 */
	public void saveImageAttrbutesToXml(JSONObject operNodeJson, Element node) {
		JSONObject propsJsonObj = operNodeJson.getJSONObject("props");
		Set<String> keys = propsJsonObj.keySet();
		String v;
		for (String k : keys) {
			v = propsJsonObj.getJSONObject(k).getString("value");
			if ("null".equals(v)) {
				v = "";
			}
			switch (k) {
				case "Image":
				case "PastImage":
				case "LastImage":
					boolean canSetAttrValue = false;
					// 如果属性为空，前台传值不为空时，才新增这个属性;
					// 如果属性不为空，前台传值为空时，则删除这个属性
					if (null == node.element("NodeGraphic").attribute(k)) {
						if (StringUtils.isNotEmpty(v)) {
							canSetAttrValue = true;
						}
					} else {
						// 前台传过来的值为空，那么删除掉这个属性
						if (StringUtils.isNotEmpty(v)) {
							canSetAttrValue = true;
						} else {
							deleteXmlElementAttribute(node.element("NodeGraphic"),
									node.element("NodeGraphic").attribute(k));
						}
					}
					// 设置属性及值
					if (canSetAttrValue) {
						node.element("NodeGraphic").addAttribute(k, v);
					}
					break;
				case "BPMState":
					if (StringUtils.isNotEmpty(v) && !"-1".equals(v)) {
						node.addAttribute(k, v);
					} else {
						if (null != node.attribute(k)) {
							deleteXmlElementAttribute(node, node.attribute(k));
						}
					}
					break;
				case "RetreatCaption":
				case "RevokeTrigger":
					if (StringUtils.isNotEmpty(v)) {
						node.addAttribute(k, v);
					} else {
						deleteXmlElementAttribute(node, node.attribute(k));
					}
					break;
				case "WorkitemRevoke":
					if ("false".equals(v)) {
						node.addAttribute(k, v);
					} else {
						deleteXmlElementAttribute(node, node.attribute(k));
					}
					break;
				case "IsSync":
					if ("true".equals(v)) {
						node.addAttribute(k, v);
					} else {
						deleteXmlElementAttribute(node, node.attribute(k));
					}
					break;
				default:
					break;
			}
		}
	}

	/***
	 * 获取前端默认值设置json
	 * @param otherAttributesMap 其他属性map
	 * @return 设置后的json
	 */
	public JSONObject getFrontDefaultValuesJson(Map<String, String> otherAttributesMap) {
		JSONObject nodeJson = new JSONObject();
		this.setFrontDefaultValue(this, otherAttributesMap, nodeJson);
		return nodeJson;
	}

	/***
	 * 设置审批超时集合xml
	 * @param node 当前节点
	 * @param v 审批超时集合的值
	 * @param k 审批超时集合Key
	 */
	public void setTimerItemCollectionXmlValue(Element node, String v, String k) {
		if ("已设置".equals(v)) {
			return;
		}
		if (null != node.element(k)) {
			node.remove(node.element(k));
		}
		node.addElement(k);
		if (StringUtils.isNotEmpty(v) && !"{}".equals(v)) {
			JSONArray array = JSON.parseArray(v);
			if (!CollectionUtils.isEmpty(array)) {
				JSONObject temp;
				String value;
				for (int j = 0; j < array.size(); j++) {
					temp = array.getJSONObject(j);
					switch (temp.getString("type")) {
						case "自动通过":
						case "自动驳回":
							String eleName = "自动通过".equals(temp.getString("type"))
									? "TimerAutoPass" : "TimerAutoDeny";
							Element timerAutoPass = node.element(k).addElement(eleName);
							List<String> keys = Arrays.asList(ConstantUtil.KEY, "Peroid", "UserInfo");
							for (String key : keys) {
								value = temp.getString(key);
								if ("UserInfo".equals(key)) {
									value = temp.getString("Suggestion");
									if (StringUtils.isNotEmpty(value)) {
										value = value.replaceAll("@br@", "\n")
												.replaceAll("&lt;br&gt;", "\n");
									}
								}
								if (StringUtils.isNotEmpty(value)) {
									timerAutoPass.addAttribute(key, value);
								}
							}
							break;
						case "循环自动任务":
						case "单次自动任务":
							Element timerItem = node.element(k).addElement("TimerItem");
							List<String> autoKeys =
									Arrays.asList(ConstantUtil.KEY, "Peroid", "CycleInterval", "Repeat", "Trigger");
							for (String key : autoKeys) {
								// "单次自动任务"的"Repeat"属性跳过，不设置
								if ("单次自动任务".equals(temp.getString("type")) && "Repeat".equals(key)) {
									continue;
								}
								value = temp.getString(key);
								if ("Trigger".equals(key)) {
									value = temp.getString("Suggestion");
									if (StringUtils.isNotEmpty(value)) {
										value = value.replaceAll("@br@", "\n")
												.replaceAll("&lt;br&gt;", "\n");
									}
								}
								if (StringUtils.isNotEmpty(value)) {
									timerItem.addAttribute(key, value);
								}
							}
							break;
						default:
							break;
					}
				}
			}
		}
	}

	/***
	 * 保存权限到xml文件中
	 * @param node 当前节点
	 * @param v 值
	 * @param k 键：值为：Perm
	 */
	public void savePermToXml(Element node, String v, String k) {
		if ("已设置".equals(v)) {
			return;
		}
		if (null != node.element(k)) {
			node.remove(node.element(k));
		}
		node.addElement(k);
		if (StringUtils.isNotEmpty(v)) {
			v = v.replaceAll("\"\"\\[", "\"[")
					.replaceAll("]\"\"", "]")
					.replaceAll("\"\\[", "[")
					.replaceAll("]\"", "]")
					.replaceAll("(\\\\r\\\\n)|\\\\n", "\\\n");
			JSONObject permJson = JSON.parseObject(v);
			if (null != permJson) {
				// 设置操作权限
				setXmlPerm(permJson, node, k, "OptPerm", "OptPermItem");
				// 设置可见性
				setXmlPerm(permJson, node, k, "VisiblePerm",
						"VisiblePermItem");
				// 设置可用性
				setXmlPerm(permJson, node, k, "EnablePerm",
						"EnablePermItem");
			}
		}
	}

	/***
	 * 设置Perm的相关属性
	 * @param permJson 权限json
	 * @param node 当前操作的节点
	 * @param k 当前属性名
	 * @param firstElementName xml中Perm元素下第一级元素名称
	 * @param secondElementName xml中Perm元素下第二级元素名称
	 */
	private void setXmlPerm(JSONObject permJson, Element node, String k,
							String firstElementName, String secondElementName) {
		JSONArray permArray = permJson.getJSONArray(firstElementName);
		if (!CollectionUtils.isEmpty(permArray)) {
			if (null != node.element(k).element(firstElementName)) {
				node.element(k).remove(node.element(k).element(firstElementName));
			}
			Element permElement = node.element(k).addElement(firstElementName);
			JSONObject eachObj;
			for (Object o : permArray) {
				eachObj = (JSONObject) o;
				permElement.addElement(secondElementName).addAttribute(ConstantUtil.KEY, eachObj.getString(ConstantUtil.KEY));
			}
		} else {
			node.element(k).remove(node.element(k).element(firstElementName));
		}
	}

	/***
	 * 设置操作集合xml值
	 * @param node 当前节点
	 * @param v 操作集合的值
	 * @param k 操作集合Key，即：OperationCollection
	 */
	public void setOperationCollectionXmlValue(Element node, String v, String k) {
		if ("已设置".equals(v)) {
			return;
		}
		if (null != node.element(k)) {
			node.remove(node.element(k));
		}
		node.addElement(k);
		if (StringUtils.isNotEmpty(v)) {
			v = v.replaceAll("\"\"\\[", "[")
					.replaceAll("]\"\"", "]")
					.replaceAll("\"\\[", "[")
					.replaceAll("]\"", "]")
					.replaceAll("(\\\\r\\\\n)|\\\\n", "\\\n");
			JSONArray valueJson = JSON.parseArray(v);
			if (!CollectionUtils.isEmpty(valueJson)) {
				JSONObject temp;
				Element operation;
				String value;
				for (int j = 0; j < valueJson.size(); j++) {
					temp = valueJson.getJSONObject(j);
					operation = node.element(k).addElement(ConstantUtil.OPERATION);
					List<String> keySet = Arrays.asList(ConstantUtil.KEY, ConstantUtil.CAPTION, ConstantUtil.ENABLE, ConstantUtil.VISIBLE,
							"Icon", "TemplateKey", ConstantUtil.ACTION);
					for (String key : keySet) {
						value = temp.getString(key);
						if (StringUtils.isNotEmpty(value)) {
							// 如果Enable值是ReadOnly()，不增加属性
							if (ConstantUtil.ENABLE.equals(key) && "ReadOnly()".equals(value)) {
								continue;
							}
							if (ConstantUtil.ACTION.equals(key)) {
								if (null != temp.get(key) && StringUtils.isNotEmpty(temp.get(key).toString())) {
									operation.addElement(key).addCDATA(temp.get(key).toString()
											.replaceAll("&quot;", "\"")
											.replaceAll("&apos;", "\'"));
								}
								continue;
							}
							operation.addAttribute(key, value);
						}
					}
				}
			}
		} else {
			// 如果未设置，默认有一个“提交”操作
			String operCollectionKey = node.getParent().attributeValue(ConstantUtil.KEY) + "_op1";
			node.element(k).addElement(ConstantUtil.OPERATION).addAttribute(ConstantUtil.KEY, operCollectionKey)
					.addAttribute(ConstantUtil.CAPTION, "提交")
					.addElement(ConstantUtil.ACTION)
					.addCDATA("CommitWorkitem(-1,1,\"\")");
		}
	}

	/***
	 * 设置辅助集合xml，来自前端json
	 * @param node 当前节点
	 * @param v 当前节点辅助集合值
	 * @param k 当前key,值为:AssistanceCollection
	 */
	public void setAssistanceCollectionFromFrontJson(Element node, String v, String k) {
		if ("已设置".equals(v)) {
			return;
		}

		if (null != node.element(k)) {
			node.remove(node.element(k));
		}
		node.addElement(k);
		if (StringUtils.isNotEmpty(v)) {
			v = v.replaceAll("\"\"\\[", "\"[")
					.replaceAll("]\"\"", "]")
					.replaceAll("\"\\[", "[")
					.replaceAll("]\"", "]")
					.replaceAll("\"\\{", "{")
					.replaceAll("}\"", "}")
					.replaceAll("(\\\\r\\\\n)|\\\\n", "@n@")
					.replaceAll("\\\\", "");
			JSONArray assistanceCollectionJson = JSON.parseArray(v);
			if (null != assistanceCollectionJson) {
				for (int i = 0; i < assistanceCollectionJson.size(); i++) {
					JSONObject obj = (JSONObject) assistanceCollectionJson.get(i);
					String nodeType = obj.getString("NodeType");
					String key_ = obj.getString(ConstantUtil.KEY);
					String caption_ = obj.getString(ConstantUtil.CAPTION);
					Element taskNode = node.element(k).addElement(nodeType);
					taskNode.addElement("NodeGraphic").addAttribute(ConstantUtil.HEIGHT, "0")
							.addAttribute("Width", "0");
					taskNode.addElement("NodeAttachment");
					taskNode.addAttribute(ConstantUtil.KEY, key_);
					taskNode.addAttribute(ConstantUtil.CAPTION, caption_);
					taskNode.addAttribute("ID", node.attributeValue("ID")
							+ "00" + (i + 1));
					//其他属性设置
					JSONObject otherSetJson = obj.getJSONObject("OtherSet");
					for (String kk : otherSetJson.keySet()) {
						String vv = otherSetJson.getString(kk);
						switch (kk) {
							case "ProcessKey":
							case "TemplateKey":
							case "CreateTrigger":
							case "FinishTrigger":
							case "PassCondition":
							case "ValidCondition":
							case "FinishCondition":
							case "PassType":
							case "ValidType":
								if (!"null".equals(vv) && StringUtils.isNotEmpty(vv)) {
									taskNode.addAttribute(kk, vv);
								}
								break;
							case "InUse":
								if ("false".equals(vv)) {
									taskNode.addAttribute(kk, vv);
								}
								break;
							case "AutoIgnoreNoParticipator":
								if ("true".equals(vv)) {
									taskNode.addAttribute(kk, vv);
								}
								break;
							default:
								break;
						}
					}
					// 权限设置
					JSONObject permJson = obj.getJSONObject("Perm");
					if (null != permJson) {
						if (null != taskNode.element("Perm")) {
							taskNode.remove(taskNode.element("Perm"));
						}
						taskNode.addElement("Perm");
						// 设置操作权限
						setXmlPerm(permJson, taskNode, "Perm", "OptPerm",
								"OptPermItem");
						// 设置可见性
						setXmlPerm(permJson, taskNode, "Perm", "VisiblePerm",
								"VisiblePermItem");
						// 设置可用性
						setXmlPerm(permJson, taskNode, "Perm", "EnablePerm",
								"EnablePermItem");
					}
					// 操作集合
					JSONArray operationCollectionJson = obj.getJSONArray(ConstantUtil.OPERATION_COLLECTION);
					this.setOperationCollectionXmlValue(taskNode,
							operationCollectionJson.toJSONString(), ConstantUtil.OPERATION_COLLECTION);
					// 参与者集合
					JSONObject participatorCollectionJson = obj.getJSONObject("ParticipatorCollection");
					if (null != participatorCollectionJson) {
						if (null != taskNode.element("ParticipatorCollection")) {
							taskNode.remove(taskNode.element("ParticipatorCollection"));
						}
						taskNode.addElement("ParticipatorCollection");
						// 设置"字典"
						setXmlParticipatorCollection(participatorCollectionJson, taskNode,
								"ParticipatorCollection", "Dictionary");
						// 设置"查询集"
						setXmlParticipatorCollection(participatorCollectionJson, taskNode,
								"ParticipatorCollection", "Query");
						// 设置"中间层表达式"
						setXmlParticipatorCollection(participatorCollectionJson, taskNode,
								"ParticipatorCollection", "MidFormula");
					}
					// 超时集合
					JSONArray timeJson = obj.getJSONArray("TimerItemCollection");
					if (null != timeJson) {
						setTimerItemCollectionXmlValue(taskNode,
								timeJson.toJSONString(), "TimerItemCollection");
					}
				}
			}
		}
	}

	/***
	 * 保存参与者集合到xml中
	 * @param node 当前节点
	 * @param v 值
	 * @param k 键，值为：ParticipatorCollection
	 */
	public void saveParticipatorCollectionToXml(Element node, String v, String k) {
		if ("已设置".equals(v)) {
			return;
		}
		if (null != node.element(k)) {
			node.remove(node.element(k));
		}
		node.addElement(k);
		if (StringUtils.isNotEmpty(v)) {
			v = v.replaceAll("\"\"\\[", "\"[")
					.replaceAll("]\"\"", "]")
					.replaceAll("\"\\[", "[")
					.replaceAll("]\"", "]")
					.replaceAll("(\\\\r\\\\n)|\\\\n", "\\\n");
			JSONObject valueJson = JSON.parseObject(v);
			if (null != valueJson) {
				// 设置"字典"
				setXmlParticipatorCollection(valueJson, node, k, "Dictionary");
				// 设置"查询集"
				setXmlParticipatorCollection(valueJson, node, k, "Query");
				// 设置"中间层表达式"
				setXmlParticipatorCollection(valueJson, node, k, "MidFormula");
			}
		}
	}

	/***
	 * 设置映射集合xml值
	 * @param node 当前节点
	 * @param v 映射集合的值
	 * @param k 映射集合Key，即：DataMapCollection
	 */
	public void setDataMapCollectionXmlValue(Element node, String v, String k) {
		if (null != node.element(k)) {
			node.remove(node.element(k));
		}
		node.addElement(k);
		if (StringUtils.isNotEmpty(v)) {
			v = v.replaceAll("\"\"\\[", "[")
					.replaceAll("]\"\"", "]")
					.replaceAll("\"\\[", "[")
					.replaceAll("]\"", "]")
					.replaceAll("(\\\\r\\\\n)|\\\\n", "\\\n");
			JSONArray valueJson = JSON.parseArray(v);
			if (!CollectionUtils.isEmpty(valueJson)) {
				JSONObject temp;
				Element billDataMapInfo;
				String value;
				for (int j = 0; j < valueJson.size(); j++) {
					temp = valueJson.getJSONObject(j);
					billDataMapInfo = node.element(k).addElement("BillDataMapInfo");
					List<String> keySet = Arrays.asList("DataMapKey", ConstantUtil.CAPTION, "AutoStart", ConstantUtil.FORM_KEY);
					for (String key : keySet) {
						value = temp.getString(key);
						if (StringUtils.isNotEmpty(value)) {
							// 如果AutoStart值是false，不增加属性
							if ("AutoStart".equals(key) && "false".equals(value)) {
								continue;
							}
							billDataMapInfo.addAttribute(key, value);
						}
					}
				}
			}
		}
	}

	/***
	 * 设置ParticipatorCollection的相关属性
	 * @param participatorCollectionJson 参与者集合json
	 * @param node 当前操作的节点
	 * @param k 当前属性名
	 * @param elementName 元素名称
	 */
	private void setXmlParticipatorCollection(JSONObject participatorCollectionJson, Element node, String k,
											  String elementName) {
		JSONArray jsonArray = participatorCollectionJson.getJSONArray(elementName);
		if (null != jsonArray && !jsonArray.isEmpty()) {
			if (null != node.element(k).elements(elementName)) {
				List<Element> elements = node.element(k).elements(elementName);
				elements.forEach(q -> node.element(k).remove(q));
			}

			//操作员/角色map
			Map<String, Map<String, String>> operatorMap;
			// 角色map
			Map<String, Map<String, String>> roleMap;
			//操作员/角色map
			operatorMap = new HashMap<>();
			operatorMap.put("Operator_ItemID", null);
			operatorMap.put("Operator_Code", null);
			// 角色map
			roleMap = new HashMap<>();
			roleMap.put("Role_ItemID", null);
			roleMap.put("Role_Code", null);

			JSONArray eachArray;
			JSONObject jsonObject;
			for (String key : participatorCollectionJson.keySet()) {
				eachArray = participatorCollectionJson.getJSONArray(key);
				switch (key) {
					case "Dictionary":
						if (!CollectionUtils.isEmpty(eachArray)) {
							for (int i = 0; i < eachArray.size(); i++) {
								jsonObject = eachArray.getJSONObject(i);
								if ("Operator".equals(jsonObject.getString("dictType"))) {
									getDictionaryMap(jsonObject, jsonObject.getString("dictType"),
											jsonObject.getString("codeType"), operatorMap);
								} else {
									getDictionaryMap(jsonObject, jsonObject.getString("dictType"),
											jsonObject.getString("codeType"), roleMap);
								}
							}
						}
						break;
					case "Query":
						if (null != node.element(k).elements(key)) {
							node.element(k).elements(key).forEach(q -> node.element(k).remove(q));
						}
						if (!CollectionUtils.isEmpty(eachArray)) {
							for (int i = 0; i < eachArray.size(); i++) {
								jsonObject = eachArray.getJSONObject(i);
								Element queryEle =
										node.element(k).addElement(key)
												.addCDATA(ConvertSpecialStringUtil
														.replaceSpecialString(jsonObject.getString("data")));
								if (jsonObject.get("QueryParameterCollection") instanceof String) {
									continue;
								}
								JSONArray array = jsonObject.getJSONArray("QueryParameterCollection");
								if (!CollectionUtils.isEmpty(array)) {
									Element queryParameterCollectionEle
											= queryEle.addElement("QueryParameterCollection");
									for (Object o : array) {
										JSONObject each = (JSONObject) o;
										queryParameterCollectionEle.addElement("QueryParameter")
												.addAttribute("DataType", each.getString("DataType"))
												.addAttribute("Formula",
														ConvertSpecialStringUtil
																.replaceSpecialString(each.getString("Formula")));
									}
								}
							}
						}
						break;
					case "MidFormula":
						if (null != node.element(k).elements(key)) {
							node.element(k).elements(key).forEach(q -> node.element(k).remove(q));
						}
						if (!CollectionUtils.isEmpty(eachArray)) {
							for (int i = 0; i < eachArray.size(); i++) {
								jsonObject = eachArray.getJSONObject(i);
								node.element(k).addElement(key)
										.addCDATA(ConvertSpecialStringUtil
												.replaceSpecialString(jsonObject.getString("value")));
							}
						}
						break;
					default:
						break;
				}
			}

			if ("Dictionary".equals(elementName)) {
				// 设置Dictionary的xml值
				setDictionaryXmlValue(node, k, elementName, operatorMap, "Operator");
				setDictionaryXmlValue(node, k, elementName, roleMap, "Role");
			}

		} else {
			node.element(k).remove(node.element(k).element(elementName));
		}
	}

	/***
	 * 设置字典类型的值
	 * @param json 操作的json
	 * @param dictType 字典类型
	 * @param codeType code类型
	 * @param dictMap 字典map
	 */
	private void getDictionaryMap(JSONObject json, String dictType, String codeType,
								  Map<String, Map<String, String>> dictMap) {
		// 字典ID/字典Code Map
		Map<String, String> map = new HashMap<>(10);
		String codeValue = json.getString("codeValue");
		if (null != dictMap.get(dictType + "_" + codeType)
				&& StringUtils.isNotEmpty(dictMap.get(dictType + "_" + codeType).get(codeType))) {
			map.put(codeType, dictMap.get(dictType + "_" + codeType).get(codeType) + ":" + codeValue);
		} else {
			map.put(codeType, codeValue);
		}
		dictMap.put(dictType + "_" + codeType, map);
	}

	/***
	 * 设置Dictionary的xml值
	 * @param node 当前节点
	 * @param k 参与者集合Key
	 * @param elementName 字典元素名
	 * @param map 操作或角色map
	 * @param dictType 字典类型：Operator，Role
	 */
	private void setDictionaryXmlValue(Element node, String k, String elementName,
									   Map<String, Map<String, String>> map, String dictType) {
		String itemId = null != map.get(dictType + "_ItemID") ? map.get(dictType + "_ItemID").get("ItemID") : "";
		String code = null != map.get(dictType + "_Code") ? map.get(dictType + "_Code").get("Code") : "";
		if (StringUtils.isNotEmpty(itemId) || StringUtils.isNotEmpty(code)) {
			Element dictionaryEle = node.element(k).addElement(elementName)
					.addAttribute("ItemKey", dictType);
			if (StringUtils.isNotEmpty(itemId)) {
				dictionaryEle.addAttribute("ItemID", itemId);
			}
			if (StringUtils.isNotEmpty(code)) {
				dictionaryEle.addAttribute("Code", code);
			}
		}
	}

	/***
	 * 设置前端默认值
	 * @param node 节点对象
	 * @param otherAttributesMap 要设置默认值的其他属性map
	 * @param nodeJson 当前节点json
	 */
	private void setFrontDefaultValue(Node node, Map<String, String> otherAttributesMap, JSONObject nodeJson) {
		// hashMap的默认长度
		int length = 4;
		if (otherAttributesMap!=null && otherAttributesMap.size()>0) {
			length += otherAttributesMap.size();
		}

		// 设置公共属性，包括：NodeType,ID,Key,Caption
		Map<String, String> commonAttributesMap = this.setCommonAttributesToJson(node, length);

		// 将其他要设置默认值的属性添加到map中
		if (otherAttributesMap!=null && otherAttributesMap.size()>0) {
			commonAttributesMap.putAll(otherAttributesMap);
		}

		//将内容写入json中，并返回前端页面属性框中显示
		this.setProps(commonAttributesMap, nodeJson);
	}

	/***
	 * 保存属性到xml文件中
	 * @param node 当前node节点
	 * @param propsJsonObj 前端传过来的属性及对应的值json
	 * @param attribute 属性
	 */
	public void saveAttributeToXml(Element node, JSONObject propsJsonObj, String attribute) {
		String v = propsJsonObj.getJSONObject(attribute).getString("value");
		if ("null".equals(v) || "undefined".equals(v)) {
			v = "";
		}
		if (StringUtils.isNotEmpty(v)) {
			node.addAttribute(attribute, v);
		} else {
			deleteXmlElementAttribute(node, node.attribute(attribute));
		}
	}

	/***
	 * 设置节点的CDATA属性值
	 * @param node 节点
	 * @param propsJsonObj 属性及属性值所在的json
	 * @param attribute 属性
	 */
	public void saveAttributeCdataToXml(Element node, JSONObject propsJsonObj, String attribute) {
		if (null != node.element(attribute)) {
			node.remove(node.element(attribute));
		}
		String v = propsJsonObj.getJSONObject(attribute).getString("value");
		if ("null".equals(v)) {
			v = "";
		}
		if (StringUtils.isNotEmpty(v)) {
			node.addElement(attribute).addText("<![CDATA[" + v + "]]>");
		}
	}

	/***
	 * 保存NodeGraphic的相关属性
	 * @param operNodeJson 操作节点json
	 * @param node 当前节点
	 * @return NodeGraphic对象
	 */
	private NodeGraphic saveNodeGraphic(JSONObject operNodeJson, Element node) {
		NodeGraphic graphic = new NodeGraphic();
		// 赋值height,width,x,y
		Integer height, width, x, y;
		JSONObject attr = operNodeJson.getJSONObject("attr");
		height = attr.getInteger("height");
		width = attr.getInteger("width");
		x = attr.getInteger(ConstantUtil.x);
		y = attr.getInteger(ConstantUtil.y);
		graphic.setHeight(height);
		graphic.setWidth(width);
		graphic.setX(x);
		graphic.setY(y);
		if (null != node.element("NodeGraphic")) {
			node.remove(node.element("NodeGraphic"));
		}
		// 保存NodeGraphic的相关属性到xml文件中
		node.addElement("NodeGraphic")
				.addAttribute(ConstantUtil.HEIGHT, graphic.getHeight() < 48 ? "48" : String.valueOf(height))
				.addAttribute("Width", graphic.getWidth() < 48 ? "48" : String.valueOf(width))
				.addAttribute(ConstantUtil.X, graphic.getX() < 150 ? "150" : String.valueOf(x))
				.addAttribute(ConstantUtil.Y, String.valueOf(y));
		return graphic;
	}

	/***
	 * 设置公共属性的json，给前端使用
	 * @param node 节点对象
	 * @param length hashMap的默认长度
	 * @return 设置后的map
	 */
	private Map<String, String> setCommonAttributesToJson(Node node, int length) {
		// 设置公共属性值啊
		Map<String, String> commonAttributesMap = new HashMap<>(length);
		commonAttributesMap.put("NodeType", node.getNodeType());
		commonAttributesMap.put("ID", node.getID());
		commonAttributesMap.put(ConstantUtil.KEY, node.getKey());
		commonAttributesMap.put(ConstantUtil.CAPTION, node.getCaption());
		return commonAttributesMap;
	}

	/***
	 * 设置属性值
	 * @param commonAttributesMap 公共属性map
	 * @param nodeJson 节点json
	 */
	private void setProps(Map<String, String> commonAttributesMap, JSONObject nodeJson) {
		String key;
		JSONObject v;
		for (Map.Entry<String, String> map : commonAttributesMap.entrySet()) {
			key = map.getKey();
			v = new JSONObject();
			v.put("value", map.getValue());
			nodeJson.put(key, v);
		}
	}

	/***
	 * 删除xml元素属性
	 * @param element 元素
	 * @param attribute 属性
	 * @return true：删除成功；false:删除失败
	 */
	public boolean deleteXmlElementAttribute(Element element, Attribute attribute) {
		if (null != attribute) {
			return element.remove(attribute);
		}
		return false;
	}

	public Node() {
	}

	public static Logger getLogger() {
		return logger;
	}

	public String getNodeType() {
		return NodeType;
	}

	public void setNodeType(String nodeType) {
		NodeType = nodeType;
	}

	public String getID() {
		return ID;
	}

	public void setID(String ID) {
		this.ID = ID;
	}

	public String getKey() {
		return Key;
	}

	public void setKey(String key) {
		Key = key;
	}

	public String getCaption() {
		return Caption;
	}

	public void setCaption(String caption) {
		Caption = caption;
	}

	public com.bokesoft.yes.design.bpm.po.NodeGraphic getNodeGraphic() {
		return NodeGraphic;
	}

	public void setNodeGraphic(com.bokesoft.yes.design.bpm.po.NodeGraphic nodeGraphic) {
		NodeGraphic = nodeGraphic;
	}

	public String getImage() {
		return Image;
	}

	public void setImage(String image) {
		Image = image;
	}

	public String getBPMState() {
		return BPMState;
	}

	public void setBPMState(String BPMState) {
		this.BPMState = BPMState;
	}

	public String getWorkitemRevoke() {
		return WorkitemRevoke;
	}

	public void setWorkitemRevoke(String workitemRevoke) {
		WorkitemRevoke = workitemRevoke;
	}

	public String getIsSync() {
		return IsSync;
	}

	public void setIsSync(String isSync) {
		IsSync = isSync;
	}
}
