package com.bokesoft.erp.all.controller;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.serializer.SerializerFeature;
import com.bokesoft.yes.design.bpm.util.BpmOperToXmlUtil;
import com.bokesoft.yes.design.bpm.util.GetDataUtil;
import com.bokesoft.yes.design.vo.JsonQuestVo;
import com.bokesoft.yes.design.vo.ResponseResult;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.exception.ExceptionUtils;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;

import java.util.List;
import java.util.logging.Logger;

/**
 * B/s模式工作流属性设置实controller
 *
 * @author zhsy
 * @date 2020-08-18
 */
@Controller
@RequestMapping("/bpmAttributes")
public class BpmAttributesController {

	private static final Logger logger = Logger.getLogger(BpmAttributesController.class.getName());

	/***操作类型，1:节点和路径的新增和修改；2：属性的修改；3：删除*/
	/***新增操作类型OPER_TYPE_ONE*/
	public static final int OPER_TYPE_ONE = 1;
	/***修改操作类型OPER_TYPE_TWO*/
	public static final int OPER_TYPE_TWO = 2;
	/***删除操作类型OPER_TYPE_THREE*/
	public static final int OPER_TYPE_THREE = 3;

	/***BPM操作类*/
	private final BpmOperToXmlUtil bpmOperUtil;
	/***获取数据操作类*/
	private final GetDataUtil getDataUtil;

	public BpmAttributesController() {
		bpmOperUtil = new BpmOperToXmlUtil();
		getDataUtil = new GetDataUtil();
	}

	/***
	 * 处理节点和路径
	 * @param jsonQuestVo 请求参数对象
	 * @return 响应结果
	 */
	@PostMapping(value = "/handleNodesAndPaths")
	@ResponseBody
	public Object handleNodesAndPaths(@RequestBody JsonQuestVo jsonQuestVo) {
		ResponseResult<JSONObject> response = new ResponseResult<>();
		try {
			// 校验请求参数
			if (validateRquestParams(jsonQuestVo, response)) {
				return response;
			}

			switch (jsonQuestVo.getOperType()) {
				case OPER_TYPE_ONE:
				case OPER_TYPE_TWO:
				case OPER_TYPE_THREE:
					response = bpmOperUtil.handleNodesAndPaths(jsonQuestVo, jsonQuestVo.getOperType());
					break;
				default:
					response.setCode(-1);
					response.setMsg("未知操作");
					return response;
			}
		} catch (Exception e) {
			logger.warning("操作失败，异常信息为:" + ExceptionUtils.getStackTrace(e));
			response.setCode(999);
			response.setMsg("操作失败，失败信息为:" + e.getMessage());
			return response;
		}
		return JSONObject.toJSONString(response,
				SerializerFeature.WriteMapNullValue,
				SerializerFeature.WriteNullStringAsEmpty,
				SerializerFeature.DisableCircularReferenceDetect);
	}

	/***
	 * 更新xml
	 * @param jsonQuestVo 请求参数对象
	 * @return 响应结果
	 */
	@PostMapping(value = "/updateXml")
	@ResponseBody
	public ResponseResult<JSONObject> updateXml(@RequestBody JsonQuestVo jsonQuestVo) {
		ResponseResult<JSONObject> response = new ResponseResult<>();
		String filePath = jsonQuestVo.getFilePath();
		if (StringUtils.isEmpty(filePath)) {
			response.setCode(1);
			response.setMsg("请求参数filePath值为空，不能操作");
			return response;
		}

		String content = jsonQuestVo.getContent();
		if (StringUtils.isEmpty(content)) {
			response.setCode(1);
			response.setMsg("请求参数content值为空，不能操作");
			return response;
		}

		try {
			response = bpmOperUtil.updateXml(jsonQuestVo);
		} catch (Exception e) {
			logger.warning("更新xml失败，异常信息为:" + ExceptionUtils.getStackTrace(e));
			response.setCode(999);
			response.setMsg("更新xml失败，失败信息为:" + e.getMessage());
		}
		return response;
	}

	/***
	 * 部署工作流
	 * @param jsonQuestVo 请求参数对象
	 * @return 响应结果
	 */
	@PostMapping(value = "/deployBpm")
	@ResponseBody
	public ResponseResult<JSONObject> deployBpm(@RequestBody JsonQuestVo jsonQuestVo) {
		ResponseResult<JSONObject> response = new ResponseResult<>();
		try {
			response = bpmOperUtil.deployBpm(jsonQuestVo);
		} catch (Exception e) {
			logger.warning("部署工作流失败，异常信息为:" + ExceptionUtils.getStackTrace(e));
			response.setCode(999);
			response.setMsg("部署工作流失败，失败信息为:" + e.getMessage());
		}
		return response;
	}

	/***
	 * 流程单据关联设置
	 * @param jsonQuestVo 请求参数对象
	 * @return 响应结果
	 */
	@PostMapping(value = "/workflowAndBillsConfig")
	@ResponseBody
	public ResponseResult<List<JSONObject>> workflowAndBillsConfig(@RequestBody JsonQuestVo jsonQuestVo) {
		ResponseResult<List<JSONObject>> response = new ResponseResult<>();
		try {
			return bpmOperUtil.workflowAndBillsConfig(jsonQuestVo);
		} catch (Exception e) {
			logger.warning("流程单据关联设置失败，异常信息为:" + ExceptionUtils.getStackTrace(e));
			response.setCode(999);
			response.setMsg("流程单据关联设置失败，失败信息为:" + e.getMessage());
		}
		return response;
	}

	/***
	 * 新增版本
	 * @param jsonQuestVo 请求参数对象
	 * @return 响应结果
	 */
	@PostMapping(value = "/newVersion")
	@ResponseBody
	public ResponseResult<JSONObject> newVersion(@RequestBody JsonQuestVo jsonQuestVo) {
		ResponseResult<JSONObject> response = new ResponseResult<>();
		try {
			return bpmOperUtil.newVersion(jsonQuestVo);
		} catch (Exception e) {
			logger.warning("新增版本失败，异常信息为:" + ExceptionUtils.getStackTrace(e));
			response.setCode(999);
			response.setMsg("新增版本失败，失败信息为:" + e.getMessage());
		}
		return response;
	}

	/***
	 * 获取当前文件的所有节点元素
	 * @param jsonQuestVo 请求参数对象
	 * @return 响应结果
	 */
	@PostMapping(value = "/getAllNodes")
	@ResponseBody
	public ResponseResult<JSONArray> getAllNodes(@RequestBody JsonQuestVo jsonQuestVo) {
		ResponseResult<JSONArray> response = new ResponseResult<>();
		try {
			if (StringUtils.isEmpty(jsonQuestVo.getFilePath())) {
				response.setCode(1);
				response.setMsg("请求参数filePath值为空，不能操作");
				return response;
			}
			return getDataUtil.getAllNodes(jsonQuestVo);
		} catch (Exception e) {
			logger.warning("获取当前文件的所有节点元素失败，异常信息为:" + ExceptionUtils.getStackTrace(e));
			response.setCode(999);
			response.setMsg("获取当前文件的所有节点元素失败，失败信息为:" + e.getMessage());
		}
		return response;
	}

	/***
	 * 获取所有的流程标识
	 * @return 流程标识结果
	 */
	@PostMapping(value = "/getInline")
	@ResponseBody
	public ResponseResult<JSONArray> getInline() {
		ResponseResult<JSONArray> response = new ResponseResult<>();
		try {
			return getDataUtil.getInline();
		} catch (Exception e) {
			logger.warning("获取所有的流程标识失败，异常信息为:" + ExceptionUtils.getStackTrace(e));
			response.setCode(999);
			response.setMsg("获取所有的流程标识失败，失败信息为:" + e.getMessage());
		}
		return response;
	}

	/***
	 * 获取状态集合
	 * @param jsonQuestVo 条件对象
	 * @return 状态集合结果
	 */
	@PostMapping(value = "/getStatusList")
	@ResponseBody
	public ResponseResult<JSONArray> getStatus(@RequestBody JsonQuestVo jsonQuestVo) {
		ResponseResult<JSONArray> response = new ResponseResult<>();
		try {
			if (StringUtils.isEmpty(jsonQuestVo.getFilePath())) {
				response.setCode(1);
				response.setMsg("请求参数filePath值为空，不能操作");
				return response;
			}
			return getDataUtil.getStatusList(jsonQuestVo);
		} catch (Exception e) {
			logger.warning("获取状态集合失败，异常信息为:" + ExceptionUtils.getStackTrace(e));
			response.setCode(999);
			response.setMsg("获取状态集合失败，失败信息为:" + e.getMessage());
		}
		return response;
	}

	/***
	 * 获取绑定标识列表
	 * @param jsonQuestVo 请求参数对象
	 * @return 响应结果
	 */
	@PostMapping(value = "/getBpmBindingKeyList")
	@ResponseBody
	public ResponseResult<JSONObject> getBpmBindingKeyList(@RequestBody JsonQuestVo jsonQuestVo) {
		ResponseResult<JSONObject> response = new ResponseResult<>();
		try {
			if (StringUtils.isEmpty(jsonQuestVo.getFilePath())) {
				response.setCode(1);
				response.setMsg("请求参数filePath值为空，不能操作");
				return response;
			}
			return getDataUtil.getBpmBindingKeyList(jsonQuestVo);
		} catch (Exception e) {
			logger.warning("获取绑定标识列表失败，异常信息为:" + ExceptionUtils.getStackTrace(e));
			response.setCode(999);
			response.setMsg("获取绑定标识列表失败，失败信息为:" + e.getMessage());
		}
		return response;
	}

	/***
	 * 获取流程标识列表
	 * @param jsonQuestVo 请求参数对象
	 * @return 响应结果
	 */
	@PostMapping(value = "/getProcessKeyList")
	@ResponseBody
	public ResponseResult<JSONArray> getProcessKeyList(@RequestBody JsonQuestVo jsonQuestVo) {
		ResponseResult<JSONArray> response = new ResponseResult<>();
		try {
			if (StringUtils.isEmpty(jsonQuestVo.getFilePath())) {
				response.setCode(1);
				response.setMsg("请求参数filePath值为空，不能操作");
				return response;
			}
			return getDataUtil.getProcessKeyList(jsonQuestVo);
		} catch (Exception e) {
			logger.warning("获取流程标识列表失败，异常信息为:" + ExceptionUtils.getStackTrace(e));
			response.setCode(999);
			response.setMsg("获取流程标识列表失败，失败信息为:" + e.getMessage());
		}
		return response;
	}

	/***
	 * 获取流所有表单列表
	 * @return 响应结果
	 */
	@PostMapping(value = "/getAllFormKeyList")
	@ResponseBody
	public ResponseResult<JSONArray> getAllFormKeyList() {
		ResponseResult<JSONArray> response = new ResponseResult<>();
		try {
			return getDataUtil.getAllFormKeyList();
		} catch (Exception e) {
			logger.warning("获取流程标识列表失败，异常信息为:" + ExceptionUtils.getStackTrace(e));
			response.setCode(999);
			response.setMsg("获取流程标识列表失败，失败信息为:" + e.getMessage());
		}
		return response;
	}

	/**
	 * 获取流程单据关联设置数据
	 *
	 * @return 响应结果
	 */
	@RequestMapping("/getWorkflowAndBillsTableData")
	@ResponseBody
	public Object getWorkflowAndBillsTableData(@RequestBody JsonQuestVo jsonQuestVo) {
		ResponseResult<JSONArray> response = new ResponseResult<>();
		try {
			if (StringUtils.isEmpty(jsonQuestVo.getFilePath())) {
				response.setCode(1);
				response.setMsg("请求参数filePath值为空，不能操作");
				return response;
			}
			return bpmOperUtil.getWorkflowAndBillsTableData(jsonQuestVo.getFilePath());
		} catch (Exception e) {
			logger.warning("获取流程单据关联设置数据失败，异常信息为:" + ExceptionUtils.getStackTrace(e));
			response.setCode(999);
			response.setMsg("获取流程单据关联设置数据失败，失败信息为:" + e.getMessage());
		}
		return response;
	}

	/**
	 * 根据Key获取权限列表
	 *
	 * @param key 标识
	 * @return 响应结果
	 */
	@RequestMapping("/getPermListByKey")
	@ResponseBody
	public Object getPermListByKey(@RequestParam("key") String key) {
		ResponseResult<JSONArray> response = new ResponseResult<>();
		try {
			if (StringUtils.isEmpty(key)) {
				response.setCode(1);
				response.setMsg("请求参数key值为空，不能操作");
				return response;
			}
			return getDataUtil.getPermListByKey(key);
		} catch (Exception e) {
			logger.warning("根据Key获取权限列表失败，异常信息为:" + ExceptionUtils.getStackTrace(e));
			response.setCode(999);
			response.setMsg("根据Key获取权限列表失败，失败信息为:" + e.getMessage());
		}
		return response;
	}

	/***
	 * 校验请求参数
	 * @param jsonQuestVo 请求参数对象
	 * @param response 响应结果
	 * @return 校验结果，true(不合法)和false(合法)
	 */
	private boolean validateRquestParams(@RequestBody JsonQuestVo jsonQuestVo, ResponseResult<JSONObject> response) {
		if (null == jsonQuestVo.getOperType() || 0 == jsonQuestVo.getOperType()) {
			response.setCode(1);
			response.setMsg("请求参数operType值为空，不能操作");
			return true;
		}

		if (StringUtils.isEmpty(jsonQuestVo.getFilePath())) {
			response.setCode(1);
			response.setMsg("请求参数filePath值为空，不能操作");
			return true;
		}

		if (StringUtils.isEmpty(jsonQuestVo.getContent())) {
			response.setCode(1);
			response.setMsg("请求参数content值为空，不能操作");
			return true;
		}

		if (StringUtils.isEmpty(jsonQuestVo.getNodeId())) {
			response.setCode(1);
			response.setMsg("请求参数nodeId值为空，不能操作");
			return true;
		}
		return false;
	}
}
