/**
 * 主题相关的处理, 包括针对 mainframe 和 login 页面的 CSS 变量设置, 以及必要的其他内容处理
 */
import { loadjs, loadless, loadcss, base64, cookie, currentPage, normalizeUxImageUrl } from "../util/util.js";

var lessLoaded = false;

/**
 * 默认的 Less 变量, 可以在 less 文件中不定义直接使用;
 * 对于不需要这些变量的主题, 在 less 文件中直接覆盖即可.
 */
const defaultThemeLessVars = {
    "@yigoBrandColor": "#6264a7", //Yigo 默认的紫色
    "@yigoBorderRadius": "0px"
}

window.addEventListener('YIUI-ux-theme-changed', function(e){
    /**
     * 同步 ThemeOption 信息到 Cookie
     * e.detail, 主题数据结构，结构如下 
     * {
     *      themeOption: {
     *          themeName: 主题,
     *          colorGroupName: 颜色,
     *          fontSchemeName: 字体
     *      }
     * }
     */
    var val = JSON.stringify(e.detail);
    val = base64.encoder(val).replace(/=/g,"");
    cookie.setItem('yigo-ux-local', val, Infinity);
}, false);

/**
 * 加载主题
 * @param {*} themeOption 参考 com.bokesoft.yigo.ux.defination.draft.webapp.mainframe.themes.ThemeOption
 * @param {*} avaliableThemes 所有可用的主题, 数组类型, 参考 com.bokesoft.yigo.ux.defination.draft.webapp.mainframe.themes.ThemePackage
 * @param {*} additionalVars 额外 CSS/LESS 变量, Key-Value 格式, Key 为变量名, Value 为变量值
 * @param {*} extendThemeCssUrls 扩展 CSS/LESS 文件的 URL 地址, 数组类型
 */
export function loadTheme(themeOption, avaliableThemes, additionalVars, extendThemeCssUrls) {
    const themeName = themeOption && themeOption.themeName;

    var themePkg = findThemePackage(avaliableThemes, themeName);
    if (!themePkg) {
        return false;
    }

    var urlVars = {};
    var themeVars = defaultThemeLessVars;

    //处理额外的 Less/CSS 变量
    additionalVars = additionalVars || {};
    for (var key in additionalVars) {
        if (key.startsWith("@")) {
            //less
            themeVars[key] = additionalVars[key];
            urlVars[key] = additionalVars[key];
        } else if (key.startsWith("--")) {
            //CSS
            document.documentElement.style.setProperty(key, additionalVars[key]);
        } else {
            //Skip it
        }
    }

    //处理色彩搭配
    const colorGroup = findColorGroup(themePkg, themeOption.colorGroupName) || {};
    const colorPalette = themePkg.colorPalette;
    if ("None" == colorPalette) {
        //无色彩搭配
    } else if ("Monochromatic" == colorPalette && colorGroup.brandColor) {
        //单色搭配
        themeVars["@yigoBrandColor"] = colorGroup.brandColor;
        urlVars["@yigoBrandColor"] = colorGroup.brandColor;
    } else {
        //TODO 不同类别的处理不同
    }
    //处理字体方案
    const fontScheme = findFontScheme(themePkg, themeOption.fontSchemeName);
    if (fontScheme) {
        const baseFontSize = fontScheme.baseFontSize;
        if (baseFontSize) {
            document.documentElement.style.setProperty('--yigoFontSize', baseFontSize);
        }
    }

    //设置 theme 全局参数
    window.less = Object.assign({}, window.less || {}, {errorReporting:'console'});
    window.less.globalVars = Object.assign(window.less.globalVars || {}, themeVars);

    var variablesCssUrl = themePkg.variablesCssUrl
    //因为 less 存在自己处理的缓存(useFileCache 在 v2 之后默认为 true), 因此需要保证不同色彩的 less 文件 url 不同
    variablesCssUrl = buildUrlWithParams(variablesCssUrl, urlVars);

    //加载变量 CSS/LESS
    loadStyleFile(variablesCssUrl);

    //加载扩展 CSS/LESS 文件
    if (extendThemeCssUrls && extendThemeCssUrls.length) {
        for (var i = 0; i < extendThemeCssUrls.length; i++) {
            const extCssUrl = extendThemeCssUrls[i];
            if (extCssUrl) {
                loadStyleFile(extCssUrl);
            }
        }
    }

    // 应用主题后，处理操作
    applyThemePostLoad(themePkg);

    return true;
}

function findThemePackage(avaliableThemes, themeName) {
    for (var i = 0; i < avaliableThemes.length; i++) {
        var themePkg = avaliableThemes[i];
        if (themeName == themePkg.name) {
            return themePkg;
        }
    }
    return null;
}

function findColorGroup(themePackage, colorGroupName) {
    const colorGroups = themePackage.colorGroups || [];
    return _findByName(colorGroups, colorGroupName)
}
function findFontScheme(themePackage, fontSchemeName) {
    const fontSchemes = themePackage.fontSchemes || [];
    return _findByName(fontSchemes, fontSchemeName)
}
function _findByName(items, itemName) {
    var fallbackItem = null;
    for (var i = 0; i < items.length; i++) {
        var item = items[i];
        if (itemName == item.name) {
            return item;
        }
        if (!fallbackItem && item.fallback) {
            fallbackItem = item;
        }
    }
    //默认返回, 如果没有默认, 返回第一个
    return fallbackItem || items[0]/*[0] 可能为空*/;
}

function buildUrlWithParams(url, vars) {
    if(!url){
        return url;
    }
    var params = [];
    for (var key in vars) {
        const val = vars[key];
        params.push(key + "=" + encodeURIComponent(val));
    }
    if (url.indexOf("?") > 0) {
        url = url + "&" + params.join("&");
    } else {
        url = url + "?" + params.join("&");
    }
    return url;
}

function loadStyleFile(url) {
    if(!url){
        return;
    }
    var rawUrl = url;
    const qsStart = rawUrl.indexOf("?");  //Query String 开始的位置
    if (qsStart > 0) {
        rawUrl = rawUrl.substring(0, qsStart);
    }
    const hashStart = rawUrl.indexOf("#");
    if (hashStart > 0) {
        rawUrl = rawUrl.substring(0, hashStart);
    }
    if (rawUrl.endsWith(".less")) {
        loadless(url);
        lessLoaded = true;
    } else {
        loadcss(url);
    }
}
/**
 * 等待主题生效（或 3 s 超时）
 * @returns {Promise<void>}
 */
function waitThemeReady(readyCallback) {
  return new Promise((resolve) => {
    const startTime = Date.now();
    const check = () => {
      const mask = window
        .getComputedStyle(document.documentElement)
        .getPropertyValue('--yigo-theme-ready-mask')
        .trim();

      if (mask === 'yigo' || Date.now() - startTime > 3000) {
        resolve(readyCallback.apply(window));          // 条件满足，结束等待
      } else {
        setTimeout(check, 20);
      }
    };
    check();
  });
}
/**
 * 应用主题
 * 
 * @param {*} readyCallback 
 */
export async function applyTheme(readyCallback) {
    if (lessLoaded) {
        //载入less.min.js（编译less文件）
        await loadjs(window.Yes_ENV_WebRoot + "ux/lib/less.js/4.2.0/less.min.js");
    }
    await waitThemeReady(readyCallback);
}

/**
 * 按照 siteDecoration 定义对登录页面进行处理
 * @param {com.bokesoft.yigo.ux.defination.classic.decorations.DefaultSiteDecoration} siteDecoration 
 */
export function decorateSite(siteDecoration) {
    siteDecoration = siteDecoration || {};

    if (siteDecoration.faviconResourceUrl) {
        let links = document.head.querySelectorAll("link");
        let found = false;
        links.forEach((link) => {
            if (link.rel.indexOf(`icon`) != -1) {
                found = true;
                link.href = href;
            }
        });
                
        if (!found) {
            let iconLink = document.createElement(`link`);
            iconLink.setAttribute(`rel`, `icon`);
            iconLink.setAttribute(`href`, normalizeUxImageUrl(siteDecoration.faviconResourceUrl, false));
            document.head.appendChild(iconLink);
        }
    }
}

/**
 * 按照 loginDecoration 定义对登录页面进行处理
 * @param {com.bokesoft.yigo.ux.defination.classic.decorations.DefaultLoginDecoration} loginDecoration 
 */
export function decorateLogin(loginDecoration) {
    loginDecoration = loginDecoration || {};

    if (loginDecoration.windowTitle) {
        document.title = loginDecoration.windowTitle;
    }
    if (loginDecoration.coverBackgroundImageUrl){ // 登录页封面背景图 (左侧)
        document.body.style.setProperty('--yigoLogin_Left_BackgroundImageUrl', normalizeUxImageUrl(loginDecoration.coverBackgroundImageUrl, true));
    }
    if (loginDecoration.backgroundImageUrl){
        document.body.style.setProperty('--yigoLogin_BackgroundImageUrl', normalizeUxImageUrl(loginDecoration.backgroundImageUrl, true));
    }
    if (loginDecoration.logoImageUrl) {
        document.body.style.setProperty('--yigoLogin_Logo_ImageUrl', normalizeUxImageUrl(loginDecoration.logoImageUrl, true));
    }
    if (loginDecoration.logoBackgroundColor) {
        const div_login_left = document.getElementsByClassName("login-left")[0];
        if (div_login_left) {
            div_login_left.style.setProperty("background-color", loginDecoration.logoBackgroundColor);
        }
    }
    if (loginDecoration.logoSloganHtml) {
        const login_left_h2 = document.getElementsByClassName("login-left")[0].getElementsByTagName("h2");
        // 若登录页有多个 h2(副标题)，则保留第一个h2，并赋值，删除其它 h2 的 dom
        for (let i = 0; i < login_left_h2.length; i++) {
            if (i == 0) {
                login_left_h2[i].innerHTML = loginDecoration.logoSloganHtml;
            } else {
                login_left_h2[i].remove();
            }
        }
    }
    if (loginDecoration.logoFooterHtml) {
        const div_login_bottom = document.getElementsByClassName("login-bottom")[0];
        if (div_login_bottom) {
            div_login_bottom.innerHTML = loginDecoration.logoFooterHtml;
        }
    }
    if (loginDecoration.titleHtml) {
        const p_title = document.getElementsByClassName("title")[0];
        if (p_title) {
            p_title.innerHTML = loginDecoration.titleHtml;
        }
    }
    if (loginDecoration.descrHtml) {
        const p_login_info = document.getElementsByClassName("login-info")[0];
        if (p_login_info) {
            p_login_info.innerHTML = loginDecoration.descrHtml;
        }
    }
    if (loginDecoration.extendedFields && loginDecoration.extendedFields.length > 0) {
        window.addEventListener("YIUI-loaded", async e => {
            const Field = await import("../extended-fields/index.js");
            Field.extendedFieldsSetting(loginDecoration.extendedFields);
        });
    }
}

/**
 * 获取 logo 的 dom 节点 <img/>
 */
const getLogoImg = () => {
    let _logoContainer = document.querySelector(".logo");
    if (!_logoContainer) return null;

    let _logoImg = _logoContainer.querySelector("img");
    if (!_logoImg) {
        _logoImg = document.createElement("img");
        _logoContainer.appendChild(_logoImg);
    };
    return _logoImg;
}

/**
 * 动态设置 logo img 的 src 属性, 且控制其可见性
 */
const setLogoImg = ( logoImg, imgUrl ) => {
    if (imgUrl) {
        logoImg.src = imgUrl;
        logoImg.style.display = "inline-block";
    } else {
        logoImg.style.display = "none";
    }
}

/**
 * 按照 mainFrameDecoration 定义对 Yigo 主界面框架进行处理
 * @param {com.bokesoft.yigo.ux.defination.classic.decorations.DefaultMainFrameDecoration} mainFrameDecoration 
 */
export function decorateMainFrame(mainFrameDecoration) {
    mainFrameDecoration = mainFrameDecoration || {};

    if (mainFrameDecoration.windowTitle) {
        document.title = mainFrameDecoration.windowTitle;
    }

    let logoImg = getLogoImg();

    let logoImageUrl = mainFrameDecoration.logoImageUrl;
    if (logoImageUrl) {
        logoImageUrl = normalizeUxImageUrl(logoImageUrl, false);
        setLogoImg(logoImg, logoImageUrl);
    }

    let logoSmallImageUrl = mainFrameDecoration.logoSmallImageUrl;
    if (logoSmallImageUrl) {
        logoSmallImageUrl = normalizeUxImageUrl(logoSmallImageUrl, false);
    }

    window.addEventListener("YIUI-loaded", e => {
        YIUI.APP.MainFrame.leftSideToggleListener(leftSideStatus=>{
            if (leftSideStatus === "sm") {
                setLogoImg(logoImg, logoSmallImageUrl);
            } else {
                setLogoImg(logoImg, logoImageUrl);
            }
        });
    });

    if (mainFrameDecoration.bannerTitleHtml) {
        //直接在 navbar 的 export 方法 init 中以参数方式实现, 此处不在需要处理
    }
}

/**
 * 应用主题后，处理操作
 * @param {Object} themePkg，当前主题设置
 * @returns 
 */
async function applyThemePostLoad(themePkg) {
    let entryScriptUrl = themePkg && themePkg.entryScriptUrl;
    if (!entryScriptUrl) {
        return;
    }

    let urlRoot = '../../../'; // 导入脚本路径从yigo目录开始计算
    const { postLoad } = await import(urlRoot + entryScriptUrl);
    if (typeof postLoad != 'function') {
        throw new Error(`主题脚本(${entryScriptUrl})需要导出postLoad函数。`);
    }

    let themeEvent = {
        themePkg: themePkg,
        isLogin: currentPage && currentPage.isLogin(),
        isMainFrame: currentPage && currentPage.isMainFrame()
    };

    window.addEventListener("YIUI-loaded", e => {
        postLoad(themeEvent);
    });
}

export async function loadNavbar() {
    // 检查设置
    if (window.YigoClassicUxSetting) { // 有YigoUX
        if (!YigoClassicUxSetting.avaliableNavbars || YigoClassicUxSetting.avaliableNavbars.length<=0) {
            window.addEventListener("YIUI-loaded", (e) => {
                YIUI.errorHandle(new Error(YIUI.I18N.getString("ERROR_OPTION_NO_AVAILABLE_NAVBAR", '未配置有效的 Navbar 实现')));
            });
        }
    }

    // 加载navbar
    let navbarEntry = "../navbar-default/index.js"
    if (window.YigoClassicUxSetting && YigoClassicUxSetting.navbarOption) {
        //引入 navbar 的入口 js
        const navbarName = YigoClassicUxSetting.navbarOption.navbarName;
        const avaliableNavbars = YigoClassicUxSetting.avaliableNavbars;

        for (var i=0; i<avaliableNavbars.length; i++){
            var navbarPkg = avaliableNavbars[i];
            if (navbarName==navbarPkg.name){
                navbarEntry = navbarPkg.entryScriptUrl;
                navbarEntry = "../../../" + navbarEntry; /* "../../../" 用于确保 navbarEntry 可以从 /${context.path} 开始计算相对路径 */
                break;
            }
        }
    }

    const navbar = await import(navbarEntry);
    loadRely(()=>{
        navbar.init();
        YIUI.MainTree = {}
    })
    function loadRely(readyCallback){
        window.addEventListener("YIUI-loaded", e => {
            readyCallback();
        });
    }
}