package com.bokesoft.distro.tech.yigosupport.extension.utils.yigo;

import com.bokesoft.yes.mid.base.ServerSetting;
import com.bokesoft.yigo.mid.scheduler.QuartzManager;
import org.apache.commons.lang3.reflect.FieldUtils;
import org.apache.commons.lang3.reflect.MethodUtils;
import org.quartz.*;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.lang.reflect.InvocationTargetException;


public class AdditionalSchedulerUtil {
    private static final Logger log = LoggerFactory.getLogger(AdditionalSchedulerUtil.class);

    /**
     * @param jobName 任务名称
     * @param jobGroup 任务分组
     * @param jobClass {@link Job}的实现类
     * @param cronExpression 定时任务的表达式
     * @throws SchedulerException Quartz API 或 Yigo {@link QuartzManager}内部错误
     */
    public static void registerJob(String jobName,String jobGroup,Class<? extends Job> jobClass,String cronExpression)
            throws SchedulerException {
        JobDetail jobDetail = JobBuilder.newJob(jobClass).withIdentity( new JobKey(jobName,jobGroup))
                .build();
        CronTrigger trigger = TriggerBuilder.newTrigger()
                .withSchedule(CronScheduleBuilder.cronSchedule(cronExpression))
                .build();
        AdditionalSchedulerUtil.registerJob(jobDetail,trigger);
    }

    /**
     * 创建定时任务
     * @param jobDetail {@link JobDetail}实例
     * @param trigger {@link Trigger}实例
     * @throws SchedulerException Quartz API 或 Yigo {@link QuartzManager}内部错误
     */
    public static void registerJob(JobDetail jobDetail, Trigger trigger) throws SchedulerException {
        if(checkJobExist(jobDetail.getKey().getName(),jobDetail.getKey().getGroup())){
            log.info("Scheduler Job already exist, JobKey:[name:{},group:{}]", jobDetail.getKey().getName(), jobDetail.getKey().getGroup());
            return;
        }
        ServerSetting serverSetting = ServerSetting.getInstance();
        if (serverSetting != null && serverSetting.isReadOnly()) {
            return;
        }
        try {
            Scheduler scheduler = getScheduler();
            scheduler.scheduleJob(jobDetail, trigger);
            scheduler.start();
            String triggerInfo = trigger.getDescription();
            if(trigger instanceof CronTrigger){
                triggerInfo = "cron-expression="+((CronTrigger)trigger).getCronExpression();
            }
            log.info("Add Scheduler Job:{}, JobImpl:{}, TriggerInfo:{}",
                    jobDetail.getKey(),jobDetail.getJobClass(),triggerInfo );
        } catch (IllegalAccessException | NoSuchMethodException | InvocationTargetException | ClassNotFoundException e) {
            throw new SchedulerException(e);
        }
    }

    /**
     * 删除定时任务
     * @param jobName 任务名称
     * @param jobGroup 任务分组
     * @throws SchedulerException Quartz API 或 Yigo {@link QuartzManager}内部错误
     */
    public static void deleteJob(String jobName,String jobGroup) throws SchedulerException {
        if(!checkJobExist(jobName,jobGroup)){
            log.info("Scheduler Job not exist, JobKey:[name:{},group:{}]", jobName, jobGroup);
            return;
        }
        ServerSetting serverSetting = ServerSetting.getInstance();
        if (serverSetting != null && serverSetting.isReadOnly()) {
            return;
        }
        try {
            Scheduler scheduler = getScheduler();
            scheduler.deleteJob(new JobKey(jobName,jobGroup));
            log.info("Delete Scheduler, JobKey:[name:{},group:{}]", jobName, jobGroup);
        } catch (IllegalAccessException | NoSuchMethodException | InvocationTargetException | ClassNotFoundException e) {
            throw new SchedulerException(e);
        }
    }

    /**
     * 检查定时任务是否存在
     * @param jobName 任务名称
     * @param jobGroup 任务分组
     * @return
     * @throws SchedulerException Quartz API 或 Yigo {@link QuartzManager}内部错误
     */
    public static boolean checkJobExist(String jobName,String jobGroup) throws SchedulerException {
        try {
            Scheduler scheduler = getScheduler();
            return scheduler.checkExists(new JobKey(jobName,jobGroup));
        } catch (IllegalAccessException | NoSuchMethodException | InvocationTargetException | ClassNotFoundException e) {
            throw new SchedulerException(e);
        }
    }

    private static Object getQuartzManager() throws ClassNotFoundException, InvocationTargetException, NoSuchMethodException, IllegalAccessException {
        Class<?> quartzManagerClass;
        Object quartzManager;
        try{
            quartzManagerClass = Class.forName("com.bokesoft.yigo.mid.scheduler.QuartzManager");
            quartzManager= MethodUtils.invokeStaticMethod(quartzManagerClass,"getInstance");

        }catch (ClassNotFoundException e){
            quartzManagerClass = Class.forName("com.bokesoft.yes.mid.scheduler.QuartzManager");
            quartzManager= MethodUtils.invokeStaticMethod(quartzManagerClass,"getInstance");
        }
        return quartzManager;
    }

    private static Scheduler getScheduler() throws SchedulerException, ClassNotFoundException, InvocationTargetException, NoSuchMethodException, IllegalAccessException {
        Object quartzManager= getQuartzManager();
        SchedulerFactory schedulerFactory = (SchedulerFactory) FieldUtils.readField(
                quartzManager,"schedulerFactory", true);
        return schedulerFactory.getScheduler();
    }

}
