package com.bokesoft.distro.tech.yigosupport.extension.coordinate.impl;

import com.bokesoft.distro.tech.commons.basis.coordinate.SemaphoreEventBus;
import com.bokesoft.distro.tech.commons.basis.coordinate.intf.ISemaphoreChannel;
import com.bokesoft.distro.tech.commons.basis.coordinate.struct.Semaphore;
import com.bokesoft.distro.tech.yigosupport.extension.coordinate.intf.RedisCommands;
import com.bokesoft.distro.tech.yigosupport.extension.coordinate.intf.RedisFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import redis.clients.jedis.JedisPubSub;

import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

public class RedisSemaphoreChannel implements ISemaphoreChannel {

    private final static String CHANNEL_PREFIX = "YIGO_SEMAPHORE.";
    private final static String CHANNEL_PATTERN = CHANNEL_PREFIX + "*";

    private static final Logger log = LoggerFactory.getLogger(RedisSemaphoreChannel.class);

    private final RedisFactory factory;

    private final RedisConsumer consumer;
    private final ScheduledExecutorService executor = Executors.newSingleThreadScheduledExecutor(r -> {
        Thread th = new Thread(r);
        th.setDaemon(true);
        th.setName("RedisSemaphoreChannel." + CHANNEL_PREFIX);
        return th;
    });


    private final SemaphoreEventBus eventBus;

    public RedisSemaphoreChannel(SemaphoreEventBus eventBus, RedisFactory factory) {
        this.eventBus = eventBus;
        this.factory = factory;
        this.consumer = new RedisConsumer(factory);

        executor.execute(consumer);
    }

    @Override
    public void doSend(Semaphore semaphore) {
        try (RedisCommands cmd = factory.getRedis()) {
            cmd.publish(CHANNEL_PREFIX + semaphore.key, semaphore.toJSONString());
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }


    private class RedisConsumer extends JedisPubSub implements Runnable {


        private final RedisFactory factory;

        private RedisConsumer(RedisFactory factory) {
            this.factory = factory;
        }

        @Override
        public void onPSubscribe(String pattern, int subscribedChannels) {
            super.onPSubscribe(pattern, subscribedChannels);
            log.info("信号量监听成功!");
        }

        @Override
        public void onPMessage(String pattern, String channel, String message) {
            super.onPMessage(pattern, channel, message);
            try {
                String sema = channel.substring(CHANNEL_PREFIX.length());
                Semaphore semaphore = Semaphore.fromJSONString(message);
                eventBus.onReceiver(semaphore);
            } catch (Exception e) {
                log.warn(String.format("信号量[%s] 处理失败! message= %s ", channel, message), e);
            }
        }

        @Override
        public void onPUnsubscribe(String pattern, int subscribedChannels) {
            super.onPUnsubscribe(pattern, subscribedChannels);
            log.warn("信号量监听被取消!");
        }

        @Override
        public void run() {
            log.info("开始监听信号量...");
            try (RedisCommands commands = factory.getRedis()) {
                //阻塞方法,一旦返回,说明监听中断
                commands.psubscribe(this, CHANNEL_PATTERN);
            } catch (Exception e) {
                log.warn("信号量监听异常中断", e);
            }
            log.warn("信号量监听,重试中...");
            //循环监听
            executor.schedule(this, 3, TimeUnit.SECONDS);
        }
    }


}
