package com.bokesoft.distro.tech.yigosupport.extension.filter;

import com.bokesoft.distro.tech.commons.basis.trace.TraceUtil;
import com.bokesoft.yes.common.trace.TraceTools;
import com.bokesoft.yigo.common.trace.OverflowStrategy;
import com.bokesoft.yigo.common.trace.TraceConstants;
import com.bokesoft.yigo.common.trace.TraceSystemManager;
import jodd.util.Wildcard;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.exception.ExceptionUtils;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.slf4j.event.Level;

import javax.servlet.*;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.util.Arrays;
import java.util.List;

public class TracePrepareFilter implements Filter {
	private static final Logger LOGGER = LoggerFactory.getLogger(TracePrepareFilter.class);

	private static final String YES_HEADINFOS = "headinfos";
	private static final String YES_SYS_OPT_ID = "Sys_Opt_ID";
	public static final String PARA_OVERFLOW_LIMIT = "PARA_OVERFLOW_LIMIT";
	public static final String PARA_OVERFLOW_STRATEGY = "PARA_OVERFLOW_STRATEGY";

	private int overflowLimit = 30;
	private OverflowStrategy overflowStrategy = OverflowStrategy.LOG_AND_DROP;

	// FIXME,和yigo-springboot-support/YigoInstanceReadyConfig#DEFAULT_INDEP_URLS 相同,应该考虑合并同类
	private final List<String> STATIC_RESOURCE_URLS = Arrays.asList(new String[]{
			"*.jpg", "*.png", "*.git", "*.woff2", "*.js", "*.css", "*.less", "*.js.map"
	});

	@Override
	public void init(FilterConfig filterConfig) throws ServletException {
		String limitStr = filterConfig.getInitParameter(PARA_OVERFLOW_LIMIT);
		if(StringUtils.isNotBlank(limitStr)){
			try {
				overflowLimit = Integer.parseInt(limitStr);
			}catch (NumberFormatException e){
				overflowLimit = 30;
			}
		}
		String overflowStrategyStr = filterConfig.getInitParameter(PARA_OVERFLOW_STRATEGY);
		if(StringUtils.isNotBlank(overflowStrategyStr)){
			switch (overflowStrategyStr){
				case "DROP":
					overflowStrategy = OverflowStrategy.DROP;
				case "LOG_AND_DROP":
					overflowStrategy = OverflowStrategy.LOG_AND_DROP;
				case "LOG_AND_RESET":
					overflowStrategy = OverflowStrategy.LOG_AND_RESET;
				default:
					overflowStrategy = OverflowStrategy.LOG_AND_DROP;
			}
		}
	}

	@Override
	public void doFilter(ServletRequest request, ServletResponse response, FilterChain chain)
			throws IOException, ServletException {
		HttpServletRequest req = (HttpServletRequest) request;
		HttpServletResponse resp = (HttpServletResponse) response;
		String url = req.getServletPath();
		if(isStaticResourceUrl(url)){
		    // 静态资源,忽略trace跟踪
			chain.doFilter(request, response);
		}else {
			String spanId = TraceTools.createNanoId();
			String traceId = null;
			traceId = req.getHeader(TraceConstants.X_B3_TRACEID);
			if (StringUtils.isBlank(traceId)) {
				String headinfoStr = request.getParameter(YES_HEADINFOS);
				if (StringUtils.isNotBlank(headinfoStr)) {
					JSONObject headInfos = new JSONObject(headinfoStr);
					if (!headInfos.isNull(YES_SYS_OPT_ID)) {
						traceId = headInfos.getString(YES_SYS_OPT_ID);
					}
				}
			}

			if (StringUtils.isBlank(traceId)) {
				traceId = TraceTools.createNanoId();
			}

			String parentSpanId = req.getHeader(TraceConstants.X_B3_SPANID);
			if (StringUtils.isBlank(parentSpanId)) {
				parentSpanId = "";
			}

			// 记录trace信息
			TraceSystemManager.initializeTraceContext(traceId, spanId, parentSpanId);
			TraceSystemManager.initializeDataTrace(overflowLimit, overflowStrategy);
			try {
				TraceSystemManager.withTraceLog(() -> {
					chain.doFilter(request, response);
					return null;
				}, req.getRequestURI(), false, LOGGER, Level.INFO);
			} catch (Throwable te) {
				TraceSystemManager.setEnforceError();
				ExceptionUtils.rethrow(te);
			} finally {
				// 打印线程缓存的数据信息
				TraceSystemManager.printDataRecord();
				// 清除trace信息
				TraceSystemManager.clearTraceContext();
				// 清除线程中缓存的数据信息
				TraceSystemManager.clearDataTrace();
			}
		}
	}

	private boolean isStaticResourceUrl(String url) {
		for(String urlPattern: STATIC_RESOURCE_URLS){
			//如果命中独立请url。放入缓存,并之直接返回
			if(Wildcard.match(url, urlPattern)){
				return true;
			}
		}
		return false;
	}
}
