package com.bokesoft.distro.tech.yigosupport.extension.exttools.sdk;

import java.io.UnsupportedEncodingException;
import java.util.ArrayList;
import java.util.List;
import java.util.function.Function;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.bokesoft.distro.tech.commons.basis.MiscUtil;
import com.bokesoft.distro.tech.yigosupport.extension.exception.GeneralUserException;
import com.bokesoft.distro.tech.yigosupport.extension.exttools.funs.EncryptExp;
import com.bokesoft.distro.tech.yigosupport.extension.utils.codecs.AESUtils;
import com.bokesoft.yigo.struct.datatable.DataTable;
import com.bokesoft.yigo.struct.document.Document;

/**
 * {@link EncryptExp} 的实现后端
 */
public class EncryptImpl {
	/** 头标识字符串与加密后内容的连接字符 */
	private static final char HEADER_JOIN_CHAR = ':';
	
	/**
	 * 加密结果前缀 - AES 加密方式
	 */
	public static final String ENCRYPT_PREFIX_AES = "=*";
	
	private static final Logger log = LoggerFactory.getLogger(EncryptImpl.class);
	
	private final Configer configer;

	/**
	 * 通过加密配置参数构建
	 * @param configer
	 */
	public EncryptImpl(Configer configer){
		this.configer = configer;
	}
	
	private List<String> AES_KEYS = null;
	private List<String> lazyloadAESKey() throws UnsupportedEncodingException {
		if (null==AES_KEYS) {
			synchronized (EncryptImpl.class) {
				if (null==AES_KEYS) {
					AES_KEYS = new ArrayList<>();
					
					List<String> keys = null;
					if (null!=configer) {
						keys = configer.getEncryptKeys();
					}
					
					if (null!=keys) {
						for (String seed : keys) {
							if (StringUtils.isNotBlank(seed)) {
								AES_KEYS.add(AESUtils.getSecretKey(seed.trim()));
							}
						}
					}
					/* 不再指定系统默认的 Key, 强制要求配置 Configer
					//最后加上系统默认的 key
					String defSeed = System.class.getName();
					AES_KEYS.add(AESUtils.getSecretKey(defSeed));
					*/
					if (AES_KEYS.isEmpty()){
						throw new UnsupportedOperationException(
							"Seeds not configurated, use 'EncryptImpl.initConfiger(configer)'");
					}
				}
			}
		}
		return AES_KEYS;
	}
	private List<String> HEADERS = null;
	private List<String> lazyloadHeader(){
		if (null==HEADERS) {
			synchronized (EncryptImpl.class) {
				if (null==HEADERS) {
					HEADERS = new ArrayList<>();

					List<String> headers = null;
					if (null!=configer){
						headers = configer.getHeaders();
					}

					if (null!=headers){
						for (String hd : headers) {
							if (StringUtils.isNotBlank(hd)) {
								if (hd.indexOf(HEADER_JOIN_CHAR)>=0){
									throw new UnsupportedOperationException(
										"Headers configuration error, with invalid char '"+HEADER_JOIN_CHAR+"': " + hd);
								}
								HEADERS.add(hd.trim()+HEADER_JOIN_CHAR);
							}
						}
					}

					//最后会加上一个空的 header
					HEADERS.add("");
				}
			}
		}
		return HEADERS;
	}
	
	/**
	 * 加密数据
	 * @param value 加密内容
	 * @param prefix 代表加密方式的前缀
	 * @return 加密后字符串
	 */
	public String encrypt(String value, final String prefix) {
		if (null==value) {
			return null;
		}

		//先尝试解密
		try{
			value = decrypt(value);
		}catch(Exception e){
			//Ignore it
			if (log.isDebugEnabled()){
				log.debug("在加密前尝试解密 '{}' 错误: {}", value, e.getMessage());
			}
		}

		String header = lazyloadHeader().get(0);
		if (ENCRYPT_PREFIX_AES.equals(prefix)) {
			try {
				return header+encryptAES(value);
			} catch (UnsupportedEncodingException e) {
				throw MiscUtil.toRuntimeException(e);
			}
		}
		//找不到合适的加密类型
		throw new GeneralUserException("不支持的加密方式: "+prefix);
	}
	/**
	 * 解密数据
	 * @param value 解密内容
	 * @return 解密后字符串
	 */
	public String decrypt(String value) {
		if (null==value) {
			return null;
		}

		List<String> headers = lazyloadHeader();
		for(String hd: headers){
			if (value.startsWith(hd)){
				value = value.substring(hd.length());
				break;
			}
		}

		if (value.startsWith(ENCRYPT_PREFIX_AES)) {
			try {
				return decryptAES(value);
			} catch (UnsupportedEncodingException e) {
				throw MiscUtil.toRuntimeException(e);
			}
		}
		throw new GeneralUserException("加密数据格式错误");
	}
	
	/**
	 * 针对 Yigo 上下文 document 中某个表的某个字段进行加密
	 * @param doc 当前document,数据文档实例
	 * @param tableName 表key
	 * @param fieldName 字段key
	 * @param prefix 代表加密方式的前缀
	 */
	public void encryptDocument(Document doc, String tableName, String fieldName, final String prefix) {
		processDocument(doc, tableName, fieldName, "字段加密", true, (value) -> {
			return encrypt(value, prefix);
		});
	}

	/**
	 * 针对 Yigo 上下文 document 中某个表的某个字段进行解密
	 * @param doc 当前document,数据文档实例
	 * @param tableName 表key
	 * @param fieldName 字段key
	 */
	public void decryptDocument(Document doc, String tableName, String fieldName) {
		processDocument(doc, tableName, fieldName, "字段解密", false, (value) -> {
			try {
				return decrypt(value);
			} catch (Exception e) {
				//无法正常解密, 直接返回原文
				return value;
			}
		});
	}
	
	private String encryptAES(final String value) throws UnsupportedEncodingException {
		List<String> keys = lazyloadAESKey();
		String key = keys.get(0);
		try {
			String enVal = AESUtils.encrypt(value, key);
			return ENCRYPT_PREFIX_AES + enVal;
		} catch (Exception e) {
			throw MiscUtil.toRuntimeException(e);
		}
	}
	
	private String decryptAES(final String value) throws UnsupportedEncodingException {
		String rawValue = value.substring(ENCRYPT_PREFIX_AES.length());
		List<String> keys = lazyloadAESKey();
		int size = keys.size();
		for(int i=0; i<size; i++) {
			String key = keys.get(i);
			try {
				String deVal = AESUtils.decrypt(rawValue, key);
				return deVal;
			} catch (Exception e) {
				log.warn("使用第 "+i+ "个密钥解密错误: " + e.getMessage());
			}
		}
		return value;
	}
	
	private static void processDocument(
		Document doc, String tableName, String fieldName, String descr, boolean changeState, Function<String,String> runner) {
		DataTable tbl = doc.get(tableName);
		if (null==tbl) {
			throw new GeneralUserException("当前上下文中找不到数据表 ["+tableName+"]");
		}

		int size = tbl.size();
		if (size <= 0){
			return;
		}

		int fieldIdx = tbl.getMetaData().findColumnIndexByKey(fieldName, true);
		for(int i=0; i<size; i++) {
			String val = tbl.getString(i, fieldName);
			if(null!=val) {
				log.info("正在处理 '{}': {}.{} 的 第 [{}] 行, {} ...",
				         descr, tableName, fieldName, i, changeState?"<记录修改状态>":"<保留修改状态>");
				val = runner.apply(val);
				tbl.setObject(i, fieldIdx, val, changeState);
			}
		}
	}

	/**
	 * 后台加密公式使用的配置信息
	 */
	public interface Configer {
		/**
		 * 密钥字符串, 支持多项, 第一个为主密钥(用于加密), 其他则仅用于解密
		 * @return
		 */
		List<String> getEncryptKeys();

		/**
		 * 头标识字符串, 支持多项, 第一个为主标识(用于加密), 其他则仅用于解密
		 * @return
		 */
		List<String> getHeaders();
	}
}
