package com.bokesoft.distro.tech.yigosupport.extension.coordinate.impl;

import com.bokesoft.base.bokebase.instance.ProcessInstanceUtil;
import com.bokesoft.distro.tech.commons.basis.coordinate.intf.ILeaderStatusDetector;
import com.bokesoft.distro.tech.commons.basis.coordinate.struct.LeaderStatusConfig;
import com.bokesoft.distro.tech.yigosupport.extension.coordinate.intf.RedisCommands;
import com.bokesoft.distro.tech.yigosupport.extension.coordinate.intf.RedisFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;


public class RedisLeaderStatusDetector implements ILeaderStatusDetector {


    final static Logger log = LoggerFactory.getLogger(RedisLeaderStatusDetector.class);

    private final RedisFactory factory;

    private final ScheduledExecutorService executor;

    final String instanceId = ProcessInstanceUtil.getInstanceId();



    private final Map<String, LeaderStatus> leaderStatusMap = new HashMap<>();

    public RedisLeaderStatusDetector(RedisFactory factory) {
        this(factory, Executors.newSingleThreadScheduledExecutor(r -> {
            Thread th = new Thread(r);
            th.setDaemon(true);
            th.setName(RedisLeaderStatusDetector.class.getName());
            th.setUncaughtExceptionHandler((t, e) -> e.printStackTrace());
            return th;
        }));
    }

    public RedisLeaderStatusDetector(RedisFactory factory,
                                     ScheduledExecutorService executor) {

        this.factory = factory;
        this.executor = executor;
    }

    @Override
    public boolean isLeader(String tag) {
        return findStatus(tag).isLeader();
    }

    @Override
    public void register(String tag, LeaderStatusConfig config) {
        if (leaderStatusMap.containsKey(tag)) {
            throw new IllegalArgumentException("重复注册的业务标识: " + tag);
        }
        LeaderStatus status = new LeaderStatus(tag,config);
        leaderStatusMap.put(tag, status);
        executor.submit(status);
    }


    private LeaderStatus findStatus(String tag) {
        LeaderStatus status = leaderStatusMap.getOrDefault(tag, null);
        if (status == null) {
            throw new IllegalArgumentException(String.format("不存在的主节点状态标签 %s", tag));
        }
        return status;
    }


    private class LeaderStatus implements Runnable {

        /* 成为主节点的次数,主节点每一次轮询自增一次 */
        private int leaderRound;

        /*释放主节点,为了防止强制释放导致没有主节点,释放操作是延迟的*/
        private boolean releaseLeader = false;

        private Future future;

        final String key;
        final int heartbeatSecond;
        final long maxHeartbeatCount;
        final int retryCount;
        final int sleepSecondAfterError;

        private LeaderStatus(String key,LeaderStatusConfig config) {
            checkParam(config);
            this.key = key;
            this.heartbeatSecond = config.getHeartbeatSecond();
            this.maxHeartbeatCount = config.getMaxHeartbeatCount();
            this.retryCount = config.getRetryCount();
            this.sleepSecondAfterError = config.getSleepSecondAfterError();
        }

        private void checkParam(LeaderStatusConfig config) {
            if (config.getHeartbeatSecond() <= 0) {
                throw new IllegalArgumentException("心跳秒数不能小于等于0!");
            }
            if (config.getMaxHeartbeatCount() <= 0) {
                config.setMaxHeartbeatCount(Long.MAX_VALUE);
            }
            if (config.getRetryCount() <= 0) {
                config.setRetryCount(1);
            }
            if (config.getSleepSecondAfterError() <= 0) {
                config.setSleepSecondAfterError(config.getHeartbeatSecond());
            }

        }

        @Override
        public synchronized void run() {
            for (int i = 0; i < retryCount; i++) {
                long second = 0;
                try {
                    second = runForSecond();
                } catch (Exception e) {
                    second = -1;
                    log.warn("节点状态更新失败!", e);
                }
                if (second >= 0) {
                    log.debug("准备 {} 秒后更新节点状态...", second);
                    this.future = executor.schedule(this, second, TimeUnit.SECONDS);
                    return;
                }
                if (i + 1 < retryCount) {
                    log.warn("[{}节点] instanceId={} 更新节点状态异常,准备第 {} 次重试...", isLeader() ? "主" : "从", instanceId, i + 1);
                }
            }
            log.warn("[{}节点] instanceId={} 更新节点状态异常,准备睡眠 {} 秒后重试...", isLeader() ? "主" : "从", instanceId, sleepSecondAfterError);
            this.future = executor.schedule(this, sleepSecondAfterError, TimeUnit.SECONDS);
        }

        private long runForSecond() {
            try (RedisCommands redis = factory.getRedis()) {
                if (isLeader()) {
                    if (instanceId.equals(redis.get(key))) {//判断当前主节点是否自己
                        if (!releaseLeader && leaderRound <= maxHeartbeatCount) {//判断是否需要释放主节点或主节点轮次是否超过上限
                            log.debug("[主节点] instanceId={} 第 {} 次发送心跳!", instanceId, leaderRound);
                            redis.expire(key, heartbeatSecond);//发送心跳
                            leaderRound++;
                            return heartbeatSecond / 2;//下次执行的时间必须小于心跳过期时间
                        } else {//释放主节点的做法就是不发送心跳,等锁自然释放,此时进入准备释放状态
                            log.debug("[主节点] instanceId={} 准备释放主节点...", instanceId);
                            leaderRound++;
                            return redis.ttl(key);//准备释放锁
                        }
                    } else {//降级成从节点时需要重置状态
                        log.debug("[主节点] instanceId={} 成为 [从节点]!", instanceId);
                        leaderRound = 0;
                        releaseLeader = false;
                        return redis.ttl(key);
                    }
                } else {
                    redis.setNxEx(key, instanceId, heartbeatSecond);//竞争主节点
                    if (instanceId.equals(redis.get(key))) {
                        log.debug("[从节点] instanceId={} 成为 [主节点]!", instanceId);
                        leaderRound++;
                        return heartbeatSecond / 2;
                    } else {
                        log.debug("[从节点] instanceId={} 开始等待主节点失效...", instanceId);
                        return redis.ttl(key);//保证下次轮询时,肯定可以竞争锁
                    }
                }
            } catch (Exception e) {
                throw new RuntimeException(e);
            }
        }


        public synchronized boolean isLeader() {
            return leaderRound > 0;
        }


        public synchronized void releaseLeader() {
            releaseLeader = true;
        }
    }


}
