package com.bokesoft.distro.tech.yigosupport.extension;

import java.util.ArrayList;
import java.util.List;
import java.util.ServiceLoader;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.bokesoft.distro.tech.yigosupport.extension.impl.BaseStaticWrapperMidFuncRunner;
import com.bokesoft.distro.tech.yigosupport.extension.intf.IFunctionProviderService;
import com.bokesoft.yigo.parser.IFunImplCluster;
import com.bokesoft.yigo.parser.IFunctionProvider;

/**
 * 基于 Java 的 {@link ServiceLoader} 机制动态寻找指定的 {@link IFunctionProviderService} 实现，
 * 将其包装成 Yigo 中间层公式({@link IFunctionProvider}).
 * <hr/>
 * 注意：在使用 {@link ServiceLoaderRegisteringStartListener} 之后, 已不需要下面的注册步骤:<br/>
 * <s>在 Yigo 2.0 中使用时, 需要手工在 solution 的 `Enhance.xml` 中注册，例如:
 * <pre>
 * &lt;ExtMidFunction&gt;
 *     &lt;MidFunction Provider="com.bokesoft.distro.tech.yigosupport.extension.ServiceLoaderMidFunctionProvider" Description="自动加载中间层扩展公式"/&gt;
 * &lt;/ExtMidFunction&gt;
 * </pre></s>
 */
public class ServiceLoaderMidFunctionProvider implements IFunctionProvider{
	private static final Logger log = LoggerFactory.getLogger(ServiceLoaderMidFunctionProvider.class);
	
	@Override
	public IFunImplCluster[] getClusters() {
		List<IFunImplCluster> result = new ArrayList<>();
		
		ServiceLoader<IFunctionProviderService> fpss = ServiceLoader.load(IFunctionProviderService.class);
		for(IFunctionProviderService fps: fpss) {
			String info = "Loading IFunctionProviderService '"+fps.getClass().getName()+"'";
			List<String> allFuncInfos = new ArrayList<String>();
			try {
				log.info(info + " ("+fps.getClass().getProtectionDomain().getCodeSource().getLocation().getPath()+") ...");
			}catch(Exception ex) {
				log.info(info + " ...");
			}

			IFunctionProvider fp = fps.getFunctionProvider();
			
			if (null!=fp) {
				IFunImplCluster[] clusters = fp.getClusters();
				if (null!=clusters) {
					for (int i = 0; i < clusters.length; i++) {
						IFunImplCluster c = clusters[i];
						logFunctions(c, allFuncInfos);
						result.add(c);
					}
				}
			}
			
			if (allFuncInfos.size()>0) {
				String allFuncs = StringUtils.join(allFuncInfos, "\n\t");
				log.info(info + " completed .\n\t" + allFuncs);
			}else {
				log.info(info + " completed .");
			}
		}
		
		return result.toArray(new IFunImplCluster[] {});
	}
	
	private void logFunctions(IFunImplCluster c, List<String> infos) {
		Object[][] table = c.getImplTable();
		for(Object[] funcPair: table) {
			String funcInfo = funcPair.length>0?funcPair[0]+"":"<INVALID>";
			Object funcImpl = funcPair.length>1?funcPair[1]:null;
			if (null==funcImpl) {
				funcInfo = funcInfo + ": " + null;
			}else if (funcImpl instanceof BaseStaticWrapperMidFuncRunner) {
				BaseStaticWrapperMidFuncRunner r = (BaseStaticWrapperMidFuncRunner)funcImpl;
				funcInfo = funcInfo + ": " + r.getWrapperInfo();
			}else {
				funcInfo = funcInfo + ": " + funcImpl.getClass().getName();
			}
			log.debug(">>> " + funcInfo);
			infos.add(funcInfo);
		}
	}
}
