package com.bokesoft.distro.tech.yigosupport.deployment.resource;

import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.List;
import java.util.Map;

import org.apache.commons.io.FileUtils;
import org.apache.commons.io.FilenameUtils;
import org.apache.commons.lang3.exception.ExceptionUtils;
import org.apache.commons.lang3.time.DateFormatUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.bokesoft.distro.tech.yigosupport.deployment.resource.intf.IResourceIO;
import com.bokesoft.yigo.meta.base.IMetaResolver;
import com.bokesoft.yigo.meta.factory.IMetaResolverFactory;

/**
 * “来自资源”的 Solution MetaResolverFactory 抽象类, 基于 {@link IResourceIO} 抽象具体资源的获取过程
 */
public abstract class AbstractResourceMultiSolutionMetaResolverFactory implements IMetaResolverFactory {
	private static final Logger log = LoggerFactory.getLogger(AbstractResourceMultiSolutionMetaResolverFactory.class);
	
	private static final char SEPARATOR = IResourceIO.SEPARATOR.charAt(0);

	/** 抽象具体资源的获取过程 */
	private IResourceIO resourceIO;
	/** 缓存具体资源包含的所有资源文件路径 */
	private List<String> resources;
	/** 缓存当前资源实例的 ID, 主要用于日志 */
	private String instanceId;

	/**
	 * 具体资源的读写实现, 需要在子类中提供
	 * @return
	 */
	protected abstract IResourceIO getResourceIO();

	@Override
	public void initParas(Map<String, Object> paras) {
		try {
			this.resourceIO = getResourceIO();
			this.resources = this.resourceIO.list();

			this.instanceId = this.resourceIO.getInstanceId();
		} catch (IOException e) {
			ExceptionUtils.rethrow(e);
		}
	}

	@Override
	public String getSolutionPath() {
		String slnPath = null;
		try{
			IMetaResolver resolver = newMetaResolver("");
			String slnXml = resolver.getPath("Solution.xml");
			//Solution.xml 所在目录即为 Solution Path
			File fSlnXml = new File(slnXml);
			if (fSlnXml.exists()){
				slnPath = fSlnXml.getParentFile().getCanonicalPath();
				log.info("[{}]: Solution.xml 是有效文件, 已获取有效的 Solution Path '{}'", this.instanceId, slnPath);
			}else{
				slnPath = null;
				log.warn("[{}]: 文件 {} 不存在, 无法获取有效的 Solution Path", this.instanceId, fSlnXml);
			}
		}catch(Exception ex){
			log.warn("[{}]: 获取 Solution Path 出错: {}.", this.instanceId, ex.getMessage());
			if (log.isDebugEnabled()){
				log.debug("["+this.instanceId+"] 详细错误信息: "+ex.getMessage(), ex);
			}
			slnPath = null;
		}

		if (null!=slnPath){
			return slnPath;
		}

		try {
			String safeName = FilenameUtils.normalize( (null==this.instanceId?"":this.instanceId) )
                             .replaceAll("[\\\\/:*?\"<>|]", "_");
			Path path = Files.createTempDirectory(
				this.getClass().getSimpleName() + "-"
			    + DateFormatUtils.ISO_8601_EXTENDED_DATE_FORMAT.format(System.currentTimeMillis())
				+ "." + safeName + ".");
			File tmpDir = path.toFile();
			FileUtils.forceMkdir(tmpDir);
			String tmp = tmpDir.getCanonicalPath();
			log.warn("[{}]: 不建议使用的 API 调用 - getSolutionPath, 返回无效值 '{}'", this.instanceId, tmp);
			return tmp;
		} catch (IOException e) {
			return ExceptionUtils.rethrow(e);
		}
	}

	@Override
	public IMetaResolver newMetaResolver(String prefix) {
		ResourceMultiSolutionResolver resolver = new ResourceMultiSolutionResolver(resources, resourceIO, prefix);
		log.info("create newMetaResolver [{}]: prefix='{}' .", this.instanceId, prefix);
		return resolver;
	}

	@Override
	public IMetaResolver newFileMetaResolver(String prefix) {
		throw new UnsupportedOperationException(
			"New IMetaResolver with absolute identifier is not supported"
			+"(instance='"+this.instanceId+"', prefix='"+prefix+"')");
	}

	@Override
	public char getSeparator() {
		return SEPARATOR;
	}

}
