package com.bokesoft.distro.tech.commons.basis.coordinate;

import com.bokesoft.distro.tech.commons.basis.coordinate.impl.SimpleSemaphoreChannel;
import com.bokesoft.distro.tech.commons.basis.coordinate.impl.SimpleSemaphoreThrottle;
import com.bokesoft.distro.tech.commons.basis.coordinate.intf.ISemaphoreChannel;
import com.bokesoft.distro.tech.commons.basis.coordinate.intf.ISemaphoreChannelFactory;
import com.bokesoft.distro.tech.commons.basis.coordinate.intf.ISemaphoreConsumer;
import com.bokesoft.distro.tech.commons.basis.coordinate.struct.Semaphore;
import com.bokesoft.distro.tech.commons.basis.coordinate.struct.ThrottleConfig;
import org.apache.commons.lang3.exception.ExceptionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.HashMap;
import java.util.Map;
import java.util.function.Consumer;

public class SemaphoreEventBus {

    private static SemaphoreEventBus INSTANCE;

    private static ISemaphoreChannelFactory factory = (eventBus) -> new SimpleSemaphoreChannel(eventBus);

    public static void setFactory(ISemaphoreChannelFactory factory) {
        if (factory == null) {
            throw new IllegalArgumentException();
        }
        SemaphoreEventBus.factory = factory;
    }

    public static SemaphoreEventBus getINSTANCE() {
        if (INSTANCE == null) {
            synchronized (SemaphoreEventBus.class) {
                if (INSTANCE == null) {
                    try {
                        INSTANCE = new SemaphoreEventBus(factory);
                    } catch (Exception e) {
                        log.error("SemaphoreEventBus 初始化失败,使用默认本地实现!", e);
                        return ExceptionUtils.rethrow(e);
                    }
                }
            }
        }
        return INSTANCE;
    }

    private final ISemaphoreChannel channel;
    private final static Logger log = LoggerFactory.getLogger(SemaphoreEventBus.class);

    private ThrottleConfig sendThrottle;
    private ThrottleConfig receiveThrottle;

    private final Map<String, Sender> senderMap = new HashMap<>();
    private final Map<String, Receiver> receiverMap = new HashMap<>();

    public SemaphoreEventBus(ISemaphoreChannelFactory factory) throws Exception {
        this.channel = factory.createChannel(this);
    }

    public final void registerSender(String semaphoreKey, ThrottleConfig config) {
        senderMap.put(semaphoreKey, new Sender(semaphoreKey, config == null ? sendThrottle : config));
    }

    public Sender getSender(String semaphoreKey) {
        return senderMap.get(semaphoreKey);
    }

    public boolean canSend(String semaphoreKey) {
        return senderMap.containsKey(semaphoreKey);
    }


    public final void registerReceiver(String semaphoreKey, ISemaphoreConsumer consumer, ThrottleConfig config) {
        Receiver receiver = new Receiver(semaphoreKey, consumer, config == null ? receiveThrottle : config);
        receiverMap.put(semaphoreKey, receiver);
    }

    public final Receiver getReceiver(String semaphoreKey) {
        return receiverMap.get(semaphoreKey);
    }

    public ThrottleConfig getSendThrottle() {
        return sendThrottle;
    }

    public void setSendThrottle(ThrottleConfig sendThrottle) {
        this.sendThrottle = sendThrottle;
    }

    public ThrottleConfig getReceiveThrottle() {
        return receiveThrottle;
    }

    public void setReceiveThrottle(ThrottleConfig receiveThrottle) {
        this.receiveThrottle = receiveThrottle;
    }

    public void postSend(Semaphore semaphore) {
       Sender sender=getSender(semaphore.key);
       if(sender!=null){
           sender.doSend(semaphore);
       }
    }

    public void postSend(String semaphore) {
        this.postSend(new Semaphore(semaphore));
    }

    public void onReceiver(Semaphore semaphore) {
        Receiver receiver = this.getReceiver(semaphore.key);
        if (receiver != null) {
            receiver.onSemaphore(semaphore);
        }
    }

    private class Receiver {
        public final String key;
        public final Consumer<Semaphore> consumer;
        private final ThrottleConfig config;
        private final SimpleSemaphoreThrottle throttle;

        public Receiver(String key, Consumer<Semaphore> consumer, ThrottleConfig config) {
            this.key = key;
            this.consumer = consumer;
            this.config = config;
            this.throttle = config == null ? null : new SimpleSemaphoreThrottle(config);
        }

        public void onSemaphore(Semaphore semaphore) {
            if (throttle != null) {
                throttle.onFilter(semaphore, consumer);
            } else {
                consumer.accept(semaphore);
            }
        }
    }

    private final class Sender {
        public final String key;
        private final ThrottleConfig config;
        private final SimpleSemaphoreThrottle throttle;

        public Sender(String key, ThrottleConfig config) {
            this.key = key;
            this.config = config;
            this.throttle = config == null ? null : new SimpleSemaphoreThrottle(config);

        }

        public void doSend(Semaphore semaphore) {
            if (throttle != null) {
                throttle.onFilter(semaphore, channel::doSend);
            } else {
                channel.doSend(semaphore);
            }
        }

    }
}
