package com.bokesoft.distro.tech.commons.basis.io.internal;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.ObjectOutputStream;
import org.apache.commons.lang3.exception.ExceptionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;


public class LocalFileTempWriter<T> {
    private static final Logger log = LoggerFactory.getLogger(LocalFileTempWriter.class);

    /**
     * 当前writer对应对存储对象class
     */
    private Class<?> clazz;
    /**
     * 临时目录,一般输出流开启时,文件所在的目录
     */
    private String tmpDir;
    /**
     * 准备目录,当文件准备被执行时,所在的准备区,此时输出流已关闭
     */
    private String readyDir;
    /**
     * 错误目录,当临时区文件反序列错误时,将无法解析的文件放在错误区
     */
    private String errDir;
    /**
     * 输出流所指向的文件
     */
    private String fileName;
    /**
     * 文件的区块时间,单位毫秒
     */
    private long blockTime;
    /**
     * writer的创建时间
     */
    private long createTime;
    /**
     * 输出流对象
     */
    private ObjectOutputStream oos;

    public String getFileName() {
        return fileName;
    }

    public long getBlockTime() {
        return blockTime;
    }

    public LocalFileTempWriter(String fileName, Class<T> clazz, WriterParam param) {
        try {
            this.clazz = clazz;
            this.tmpDir = param.getTmpDir();
            this.readyDir = param.getReadyDir();
            this.errDir = param.getErrDir();
            this.blockTime = param.getBlockTime();
            this.fileName = fileName;
            this.createTime = System.currentTimeMillis();
            File file = new File(tmpDir, fileName);
            this.oos = new ObjectOutputStream(new FileOutputStream(file));
        }catch (IOException e){
            ExceptionUtils.rethrow(e);
        }
    }

    /**
     * 将对象通过writer中的输出流,写入文件
     * @param value
     * @throws IOException
     */
    public void writeObject(T value) throws IOException {
        // 防止还有写入时,被关闭
        synchronized (this.oos) {
            this.oos.writeObject(value);
            // 考虑性能,flush动作放在close前
            // this.oos.flush();
        }
    }

    /**
     * 将文件转化成准备状态: 关闭输出流,且将文件移至 ready目录中
     * @throws IOException
     */
    public void tryFlush() throws IOException {
        if(System.currentTimeMillis() > createTime+blockTime*2 ) {
            synchronized (this.oos) {
                this.oos.flush();
                this.oos.close();
            }

            TempWriterManager.WRITERS_CACHE.remove(this.fileName);
            if (log.isDebugEnabled()){
                log.debug("LocalFileTempStorage 文件 {} 已写入完毕,可以迁移.", this.fileName);
            }
        }
    }


}
