package com.bokesoft.distro.tech.commons.basis.flightrecord.struc.rediscache;

import com.bokesoft.base.bokebase.rediscache.CacheDescription;
import com.bokesoft.base.bokebase.rediscache.struct.DataType;
import com.bokesoft.base.bokebase.rediscache.struct.KeyType;
import com.bokesoft.base.bokebase.rediscache.struct.StoreType;
import com.bokesoft.distro.tech.commons.basis.flightrecord.utils.YFRUtils;
import org.apache.commons.lang3.exception.ExceptionUtils;

import java.io.*;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Strings.isNullOrEmpty;

public class CacheDescriptionDTO implements Externalizable {

    /**
     * 缓存key
     */
    private String key;
    /**
     * key的类型 ： FULL_KEY表示完整的key， PREFIX表示key的前缀
     */
    private KeyType keyType;
    /**
     * 缓存描述
     */
    private String description;
    /**
     * 缓存存储类型
     */
    private StoreType storeType;
    /**
     * 缓存值类型涉及到的类
     */
    private List<ClassInfo> refClasses;
    /**
     * 缓存值描述解释器
     */
    private byte[] valueDescriberBytes;

    private DataType dataType;

    public static CacheDescriptionDTO toDTO(CacheDescription cacheDescription){
        CacheDescriptionDTO cacheDescriptionDTO = new CacheDescriptionDTO();
        cacheDescriptionDTO.setKey(cacheDescription.getKey());
        cacheDescriptionDTO.setDescription(cacheDescription.getDescription());
        cacheDescriptionDTO.setKeyType(cacheDescription.getKeyType());
        cacheDescriptionDTO.setStoreType(cacheDescription.getStoreType());
        cacheDescriptionDTO.setDataType(cacheDescription.getDataType());

        ArrayList<Class> refClasses=new ArrayList<>();
        if(cacheDescription.getValueDescriber()!=null){
            refClasses.add(cacheDescription.getValueDescriber().getClass());
            ByteArrayOutputStream buf=new ByteArrayOutputStream();
            try {
                new ObjectOutputStream(buf).writeObject(cacheDescription.getValueDescriber());
            } catch (IOException e) {
                return ExceptionUtils.rethrow(e);
            }
            cacheDescriptionDTO.setValueDescriberBytes(buf.toByteArray());
        }
        if(cacheDescription.getRefClasses()!=null){
            refClasses.addAll(cacheDescription.getRefClasses());
        }
        cacheDescriptionDTO.setRefClasses(refClasses.stream().distinct().map(c -> {
            if (YFRUtils.isJREClass(c)) {
                return null;
            } else {
                return new ClassInfo(c.getName(), YFRUtils.getClassBytes(c));
            }
                }).filter(Objects::nonNull)
                .sorted(Comparator.comparing(ClassInfo::getClassName))
                .collect(Collectors.toList()));
        return cacheDescriptionDTO;
    }

    @Override
    public void writeExternal(ObjectOutput out) throws IOException {
        checkArgument(!isNullOrEmpty(key));
        checkArgument(keyType != null);
        checkArgument(!isNullOrEmpty(description));
        out.writeUTF(key);
        out.writeObject(keyType);
        out.writeUTF(description);
        out.writeObject(storeType);
        if (refClasses != null) {
            out.writeInt(refClasses.size());
            refClasses.stream()
                    .sorted(Comparator.comparing(ClassInfo::getClassName))
                    .forEach(ci -> {
                        try {
                            out.writeObject(ci);
                        } catch (IOException e) {
                            ExceptionUtils.rethrow(e);
                        }
                    });
        } else {
            out.writeInt(0);
        }
        if (valueDescriberBytes != null) {
            out.writeInt(valueDescriberBytes.length);
            out.write(valueDescriberBytes);
        } else {
            out.writeInt(0);
        }
        out.writeObject(dataType);
        out.flush();
    }

    @Override
    public void readExternal(ObjectInput in) throws IOException, ClassNotFoundException {
        this.key = in.readUTF();
        this.keyType = (KeyType) in.readObject();
        this.description = in.readUTF();
        this.storeType = (StoreType) in.readObject();
        this.refClasses = new ArrayList<>();
        for (int i = 0, len = in.readInt(); i < len; i++) {
            refClasses.add((ClassInfo) in.readObject());
        }
        this.valueDescriberBytes = new byte[in.readInt()];
        in.readFully(this.valueDescriberBytes);
        this.dataType = (DataType) in.readObject();
    }

    public String getKey() {
        return key;
    }

    public void setKey(String key) {
        this.key = key;
    }

    public KeyType getKeyType() {
        return keyType;
    }

    public void setKeyType(KeyType keyType) {
        this.keyType = keyType;
    }

    public String getDescription() {
        return description;
    }

    public void setDescription(String description) {
        this.description = description;
    }

    public StoreType getStoreType() {
        return storeType;
    }

    public void setStoreType(StoreType storeType) {
        this.storeType = storeType;
    }

    public List<ClassInfo> getRefClasses() {
        return refClasses;
    }

    public void setRefClasses(List<ClassInfo> refClasses) {
        this.refClasses = refClasses;
    }

    public byte[] getValueDescriberBytes() {
        return valueDescriberBytes;
    }

    public void setValueDescriberBytes(byte[] valueDescriberBytes) {
        this.valueDescriberBytes = valueDescriberBytes;
    }

    public DataType getDataType() {
        return dataType;
    }

    public void setDataType(DataType dataType) {
        this.dataType = dataType;
    }

    @Override
    public String toString() {
        return "CacheDescriptionDTO{" +
                "key='" + key + '\'' +
                ", keyType=" + keyType +
                ", description='" + description + '\'' +
                ", storeType=" + storeType +
                ", refClasses=" + refClasses +
                ", dataType=" + dataType +
                '}';
    }


}
