package com.bokesoft.distro.tech.commons.basis.dependency;
import java.util.*;
import com.bokesoft.distro.tech.commons.basis.exception.DependencyCyclicException;
import org.apache.commons.lang3.exception.ExceptionUtils;

/**
 * 依赖排序算法
 */
public class DependencySortCore {
    /**
     * 依赖排序
     * @param sortableList 需要排序的集合
     * @return 返回排序后的集合
     */
    public static <T> List<T> sort(List<T> sortableList) {
        Map<Integer, String> nodeNoPool = new HashMap<>(suitableSize(sortableList.size()));
        int[][] positions=initDAG(sortableList,nodeNoPool);
        int[] sortedPosition = findOrder(nodeNoPool.size(), positions,nodeNoPool);
        //根据下标查找className
        List<T> sortedRes = new LinkedList<>();
        for (Integer pos : sortedPosition) {
            String className=nodeNoPool.get(pos);
            for (T o : sortableList) {
                if(getClassName(o).equals(className)){
                    sortedRes.add(o);
                    break;
                }
            }
        }
        return sortedRes;
    }

    private static String getClassName(Object node){
        Class cls=node.getClass();
        String clsName=cls.getName();
        if(clsName.contains("$$")){//CGLIB动态生成的代理类
            return cls.getSuperclass().getName();
        }
        return clsName;
    }

    private static <T> Set<String> getAllNodeInDAG(List<T> nodes) {
        Set<String> result = new LinkedHashSet<>(suitableSize(nodes.size()));
        for (T node : nodes) {
            result.add(getClassName(node));
        }
        for (T node : nodes) {
            if (node instanceof IDependencySortable) {
                IDependencySortable dependCalNode = (IDependencySortable) node;
                if (null != dependCalNode.getDependentClasses()) {
                    Collections.addAll(result, dependCalNode.getDependentClasses());
                }
                if (null != dependCalNode.getDependencyClasses()) {
                    Collections.addAll(result, dependCalNode.getDependencyClasses());
                }
            }
        }
        return result;
    }

    /**
     * 初始化下标
     * @param nodeSet 需要排序实例的HASH集合
     * @param nodeNoPool pos:name 映射集合
     * @param nodeNamePool name:pos 映射集合
     */
    private static void markPos(Set<String> nodeSet,
                                Map<Integer, String> nodeNoPool,
                                Map<String, Integer> nodeNamePool) {
        int pos = 0;
        for (String nodeName : nodeSet) {
            nodeNoPool.put(pos, nodeName);
            nodeNamePool.put(nodeName, pos);
            pos++;
        }
    }

    /**
     * 初始化二维数组 有向无环，参考拓扑排序
     * @param nodes 需要排序的集合
     * @param nodeNoPool pos:name 映射集合
     * @return 返回领接矩阵
     */
    protected static <T> int[][] initDAG(List<T> nodes, Map<Integer, String> nodeNoPool) {
        Set<String> nodeSet = getAllNodeInDAG(nodes);
        Map<String, Integer> nodeNamePool = new HashMap<>(suitableSize(nodes.size()));
        markPos(nodeSet, nodeNoPool, nodeNamePool);
        List<int[]> dependencySet = new LinkedList<>();
        for (Object node : nodes) {
            if (node instanceof IDependencySortable) {
                int nodePos = nodeNamePool.get(getClassName(node));
                IDependencySortable dependCalNode = (IDependencySortable) node;
                if (null != dependCalNode.getDependentClasses()) {
                    for (String bdClass : dependCalNode.getDependentClasses()) {
                        dependencySet.add(new int[] { nodeNamePool.get(bdClass), nodePos });
                    }
                }
                if (null != dependCalNode.getDependencyClasses()) {
                    for (String dClass : dependCalNode.getDependencyClasses()) {
                        dependencySet.add(new int[] { nodePos, nodeNamePool.get(dClass) });
                    }
                }
            }
        }
        int[][] wholeDependencies = new int[dependencySet.size()][];
        int dpos = 0;
        for (int[] ints : dependencySet) {
            wholeDependencies[dpos] = ints;
            dpos++;
        }
        return wholeDependencies;
    }

    private static int suitableSize(int size) {
        if (size < 8) {
            return 16;
        }
        return size * 2;
    }

    private static int[] findOrder(int numCourses, int[][] prerequisites,Map<Integer, String> nodeNoPool) {
        if (numCourses == 0){
            return new int[0];
        }
        // HashSet 作为邻接矩阵
        @SuppressWarnings("unchecked")
        Set<Integer>[] graph = new HashSet[numCourses];
        for (int i = 0; i < numCourses; i++) {
            graph[i] = new HashSet<>();
        }
        for (int[] p : prerequisites) {
            graph[p[1]].add(p[0]);
        }
        // 标记数组
        int[] mark = new int[numCourses];
        // 结果栈
        Stack<Integer> stack = new Stack<>();
        List<Integer> dagLink=new LinkedList<>();
        StringBuilder cycleMsg=new StringBuilder("Cyclic dependency found:[");
        for (int i = 0; i < numCourses; i++) {
            if (!isNotCycle(graph, mark, i, stack,dagLink)){
                for (int k = 0; k < dagLink.size(); k++) {
                    String objClassName=nodeNoPool.get(dagLink.get(k));
                    if(k>0){
                        cycleMsg.append(" <- ");
                    }
                    cycleMsg.append(objClassName);
                }
                cycleMsg.append("]");
                ExceptionUtils.wrapAndThrow(new DependencyCyclicException(cycleMsg.toString()));
            }
        }
        int[] res = new int[numCourses];
        for (int i = 0; i < numCourses; i++) {
            res[i] = stack.pop();
        }
        return res;
    }

    /**
     * 判断是否成环
     * @param graph 邻接矩阵
     * @param mark 标记数组
     * @param i 当前位置
     * @param stack 结果栈
     * @return 判断是否成环的结果,true表示没有成环
     */
    private static boolean isNotCycle(Set<Integer>[] graph, int[] mark, int i, Stack<Integer> stack,List<Integer> dagLink) {
        dagLink.add(i);
        if (mark[i] == -1){
            return true;
        }
        if (mark[i] == 1){
            return false;
        }
        mark[i] = 1;
        for (int neighbor : graph[i]) {
            if (!isNotCycle(graph, mark, neighbor, stack,dagLink)){
                return false;
            }
        }
        mark[i] = -1;
        stack.push(i);
        return true;
    }

}
