package com.bokesoft.distro.tech.bootsupport.starter.wrapper;

import com.bokesoft.distro.tech.bootsupport.starter.execctl.ExecutionTimeoutManager;
import com.bokesoft.distro.tech.bootsupport.starter.execctl.model.StartTimeObject;
import com.bokesoft.distro.tech.bootsupport.starter.utils.ServletUtil;
import com.bokesoft.distro.tech.commons.basis.flightrecord.FlightRecordFacade;
import com.bokesoft.distro.tech.commons.basis.trace.TraceUtil;
import com.bokesoft.yigo.common.trace.TraceRecorder;
import org.springframework.web.context.request.RequestAttributes;
import org.springframework.web.context.request.RequestContextHolder;
import org.springframework.web.context.request.ServletRequestAttributes;

import javax.servlet.http.HttpServletRequest;
import java.sql.*;
import java.util.Map;
import java.util.Properties;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executor;

public class WrappedConnection implements Connection {
    public static Map<Connection, String> dbTraceCache = new ConcurrentHashMap<>();

    private Connection inner;
    private String datasourceid;
    private String instanceId;
    private String trace;

    private boolean dbTraceCacheEnabled;
    private boolean execTimeoutControlEnabled;

    public WrappedConnection(String datasourceid, Connection conn, String instanceId, boolean dbTraceCacheEnabled, boolean execTimeoutControlEnabled) {
        this.inner = conn;
        this.instanceId = instanceId;
        this.datasourceid = datasourceid;

        this.dbTraceCacheEnabled = dbTraceCacheEnabled;
        if (dbTraceCacheEnabled) {
            trace = buildThreadTrace(Thread.currentThread());
            dbTraceCache.put(this, trace);
        }
        this.execTimeoutControlEnabled = execTimeoutControlEnabled;
        FlightRecordFacade.logConnectionCreate(datasourceid, instanceId);
    }
    private String buildThreadTrace(Thread currentThread) {
        StringBuffer sb = new StringBuffer("Thread[" + currentThread.getName() + "," + currentThread.getId() + "];");
        sb.append("\n");
        buildTraceInfo(sb);
        int limit = 50;
        for (StackTraceElement traceElement : currentThread.getStackTrace()) {
            if (limit > 0) {
                sb.append(traceElement.getClassName() + ":" + traceElement.getMethodName() + "#" + traceElement.getLineNumber());
                sb.append(";\n");
            }
            limit--;
        }
        return sb.toString();
    }

    private void buildTraceInfo(StringBuffer sb) {
        sb.append("TraceId:"+ TraceRecorder.getTraceId() + " .");
        sb.append("\n");

        RequestAttributes requestAttributes = RequestContextHolder.getRequestAttributes();
        if(null != requestAttributes) {
            if(requestAttributes instanceof ServletRequestAttributes) {
                HttpServletRequest request = ((ServletRequestAttributes) requestAttributes).getRequest();
                String servletPath = ServletUtil.getFullServletPath(request);
                sb.append("request servlet path:" + servletPath+" .");
                sb.append("\n");
                String headInfoStr = request.getParameter("headinfos");
                sb.append("request headinfos:"+headInfoStr+" .");
                sb.append("\n");
            }
        }
    }

    public Connection getInnerConnection() {
        return inner;
    }

    public <T> T unwrap(Class<T> iface) throws SQLException {
        return inner.unwrap(iface);
    }

    public boolean isWrapperFor(Class<?> iface) throws SQLException {
        return inner.isWrapperFor(iface);
    }

    public Statement createStatement() throws SQLException {
        Statement stat = inner.createStatement();
        setQueryTimeout(stat);
        return stat;
    }

    public PreparedStatement prepareStatement(String sql) throws SQLException {
        PreparedStatement stat = inner.prepareStatement(sql);
        setQueryTimeout(stat);
        return stat;
    }

    public CallableStatement prepareCall(String sql) throws SQLException {
        CallableStatement stat = inner.prepareCall(sql);
        setQueryTimeout(stat);
        return stat;
    }

    public String nativeSQL(String sql) throws SQLException {
        return inner.nativeSQL(sql);
    }

    public void setAutoCommit(boolean autoCommit) throws SQLException {
        inner.setAutoCommit(autoCommit);
    }

    public boolean getAutoCommit() throws SQLException {
        return inner.getAutoCommit();
    }

    public void commit() throws SQLException {
        inner.commit();
    }

    public void rollback() throws SQLException {
        inner.rollback();
    }

    public void close() throws SQLException {
        if (dbTraceCacheEnabled) {
            dbTraceCache.remove(this);
        }
        FlightRecordFacade.logConnectionclose(datasourceid, instanceId);
        inner.close();
    }

    public boolean isClosed() throws SQLException {
        return inner.isClosed();
    }

    public DatabaseMetaData getMetaData() throws SQLException {
        return inner.getMetaData();
    }

    public void setReadOnly(boolean readOnly) throws SQLException {
        inner.setReadOnly(readOnly);
    }

    public boolean isReadOnly() throws SQLException {
        return inner.isReadOnly();
    }

    public void setCatalog(String catalog) throws SQLException {
        inner.setCatalog(catalog);
    }

    public String getCatalog() throws SQLException {
        return inner.getCatalog();
    }

    public void setTransactionIsolation(int level) throws SQLException {
        inner.setTransactionIsolation(level);
    }

    public int getTransactionIsolation() throws SQLException {
        return inner.getTransactionIsolation();
    }

    public SQLWarning getWarnings() throws SQLException {
        return inner.getWarnings();
    }

    public void clearWarnings() throws SQLException {
        inner.clearWarnings();
    }

    public Statement createStatement(int resultSetType, int resultSetConcurrency) throws SQLException {
        Statement stat = inner.createStatement(resultSetType, resultSetConcurrency);
        setQueryTimeout(stat);
        return stat;
    }

    public PreparedStatement prepareStatement(String sql, int resultSetType, int resultSetConcurrency)
            throws SQLException {
        PreparedStatement stat = inner.prepareStatement(sql, resultSetType, resultSetConcurrency);
        setQueryTimeout(stat);
        return stat;
    }

    public CallableStatement prepareCall(String sql, int resultSetType, int resultSetConcurrency)
            throws SQLException {
        CallableStatement stat = inner.prepareCall(sql, resultSetType, resultSetConcurrency);
        setQueryTimeout(stat);
        return stat;
    }

    public Map<String, Class<?>> getTypeMap() throws SQLException {
        return inner.getTypeMap();
    }

    public void setTypeMap(Map<String, Class<?>> map) throws SQLException {
        inner.setTypeMap(map);
    }

    public void setHoldability(int holdability) throws SQLException {
        inner.setHoldability(holdability);
    }

    public int getHoldability() throws SQLException {
        return inner.getHoldability();
    }

    public Savepoint setSavepoint() throws SQLException {
        return inner.setSavepoint();
    }

    public Savepoint setSavepoint(String name) throws SQLException {
        return inner.setSavepoint(name);
    }

    public void rollback(Savepoint savepoint) throws SQLException {
        inner.rollback(savepoint);
    }

    public void releaseSavepoint(Savepoint savepoint) throws SQLException {
        inner.releaseSavepoint(savepoint);
    }

    public Statement createStatement(int resultSetType, int resultSetConcurrency, int resultSetHoldability)
            throws SQLException {
        Statement stat = inner.createStatement(resultSetType, resultSetConcurrency, resultSetHoldability);
        setQueryTimeout(stat);
        return stat;
    }

    public PreparedStatement prepareStatement(String sql, int resultSetType, int resultSetConcurrency,
                                              int resultSetHoldability) throws SQLException {
        PreparedStatement stat = inner.prepareStatement(sql, resultSetType, resultSetConcurrency, resultSetHoldability);
        setQueryTimeout(stat);
        return stat;
    }

    public CallableStatement prepareCall(String sql, int resultSetType, int resultSetConcurrency,
                                         int resultSetHoldability) throws SQLException {
        CallableStatement stat = inner.prepareCall(sql, resultSetType, resultSetConcurrency, resultSetHoldability);
        setQueryTimeout(stat);
        return stat;
    }

    public PreparedStatement prepareStatement(String sql, int autoGeneratedKeys) throws SQLException {
        PreparedStatement stat = inner.prepareStatement(sql, autoGeneratedKeys);
        setQueryTimeout(stat);
        return stat;
    }

    public PreparedStatement prepareStatement(String sql, int[] columnIndexes) throws SQLException {
        PreparedStatement stat = inner.prepareStatement(sql, columnIndexes);
        setQueryTimeout(stat);
        return stat;
    }

    public PreparedStatement prepareStatement(String sql, String[] columnNames) throws SQLException {
        PreparedStatement stat = inner.prepareStatement(sql, columnNames);
        setQueryTimeout(stat);
        return stat;
    }

    public Clob createClob() throws SQLException {
        return inner.createClob();
    }

    public Blob createBlob() throws SQLException {
        return inner.createBlob();
    }

    public NClob createNClob() throws SQLException {
        return inner.createNClob();
    }

    public SQLXML createSQLXML() throws SQLException {
        return inner.createSQLXML();
    }

    public boolean isValid(int timeout) throws SQLException {
        return inner.isValid(timeout);
    }

    public void setClientInfo(String name, String value) throws SQLClientInfoException {
        inner.setClientInfo(name, value);
    }

    public void setClientInfo(Properties properties) throws SQLClientInfoException {
        inner.setClientInfo(properties);
    }

    public String getClientInfo(String name) throws SQLException {
        return inner.getClientInfo(name);
    }

    public Properties getClientInfo() throws SQLException {
        return inner.getClientInfo();
    }

    public Array createArrayOf(String typeName, Object[] elements) throws SQLException {
        return inner.createArrayOf(typeName, elements);
    }

    public Struct createStruct(String typeName, Object[] attributes) throws SQLException {
        return inner.createStruct(typeName, attributes);
    }

    public void setSchema(String schema) throws SQLException {
        inner.setSchema(schema);
    }

    public String getSchema() throws SQLException {
        return inner.getSchema();
    }

    public void abort(Executor executor) throws SQLException {
        inner.abort(executor);
    }

    public void setNetworkTimeout(Executor executor, int milliseconds) throws SQLException {
        inner.setNetworkTimeout(executor, milliseconds);
    }

    public int getNetworkTimeout() throws SQLException {
        return inner.getNetworkTimeout();
    }

    /**
     * 设置查询超时时间
     */
    private void setQueryTimeout(Statement stat) throws SQLException {
        if (execTimeoutControlEnabled) {
            stat.setQueryTimeout(getDBExecutionTimeout());
        }
    }

    /**
     * 获取查询的超时时间
     */
    private int getDBExecutionTimeout() {
        StartTimeObject startTimeObject = null;
        try {
            //创建单个查询开始时间对象
            startTimeObject = StartTimeObject.buildSqlExectionStartTimeObject();
            //添加单个查询开始时间对象
            ExecutionTimeoutManager.addStartTimeObject(startTimeObject);
            long dbExecutionTimeout = ExecutionTimeoutManager.allocRemainedTime(instanceId);
            return (int) Math.ceil(dbExecutionTimeout / 1000);
        } finally {
            ExecutionTimeoutManager.removeStartTimeObject(startTimeObject);
        }
    }
}
