package com.bokesoft.distro.tech.bootsupport.starter.runtime;

import java.io.File;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.Supplier;

import com.bokesoft.distro.tech.bootsupport.starter.api.YigoBootPreparator;
import com.bokesoft.distro.tech.bootsupport.starter.i18n.StringTable;
import com.bokesoft.distro.tech.commons.basis.dependency.DependencySortCore;
import com.bokesoft.yigo.common.util.SimpleStringFormat;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.exception.ExceptionUtils;
import org.apache.commons.lang3.reflect.FieldUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ConfigurableApplicationContext;

import com.bokesoft.distro.tech.bootsupport.starter.api.YigoAdditionalInitiator;
import com.bokesoft.distro.tech.bootsupport.starter.runtime.model.InstanceStatus;
import com.bokesoft.distro.tech.bootsupport.starter.runtime.model.InstanceStatus.Status;
import com.bokesoft.distro.tech.bootsupport.starter.utils.YigoPropPreparation;
import com.bokesoft.distro.tech.bootsupport.starter.utils.YigoPropPreparationHelper.PrepareResult;
import com.bokesoft.distro.tech.bootsupport.starter.utils.YigoServiceInitHelper;
import com.bokesoft.distro.tech.yigosupport.extension.ServiceLoaderRegisteringStartListener;
import com.bokesoft.distro.tech.yigosupport.extension.utils.yigo.ServiceLoaderRegisteringUtil;
import com.bokesoft.distro.tech.yigosupport.extension.utils.yigo.SessionUtils;
import com.bokesoft.yes.mid.base.SvrInfo;
import com.bokesoft.yigo.meta.enhance.MetaEnhance;
import com.bokesoft.yigo.meta.enhance.MetaExtStartListener;
import com.bokesoft.yigo.meta.enhance.MetaStartListener;
import com.bokesoft.yigo.meta.factory.IMetaFactory;
import com.bokesoft.yigo.mid.base.DefaultContext;

/**
 * Yigo 运行实例管理, 用于结合 SpringBoot 后的启动和重新加载
 */
@SuppressWarnings("deprecation")
public class YigoInstanceManager {
    private static final Logger log = LoggerFactory.getLogger(YigoInstanceManager.class);

    public static boolean BOOTING_WITH_SPRING_BOOT = true;

    /**
     * 以当前类加载时间作为系统启动时间
     */
    private static long initStartTime = System.currentTimeMillis();

    /** yigo服务运行状态 */
    private InstanceStatus instanceStatus;

    private Map<String, YigoAdditionalInitiator> additionalInitiators;

    /** 记录yigo服务是否加载完成 */
    private boolean initializationComplete = false;

    /**
     * 记录不同来源的系统不可用原因
     * key 提交不可用原因的来源程序的类, 通常是一些模块级状态是否正常的检查程序, 比如读写分离部署时读库状态检查程序
     * value 不可用原因
     */
    private final Map<Class,String> UNAVALIABLE_REASONS = new ConcurrentHashMap<>();

    /** 启动阶段第一次确定的 WorkPath */
    private String initWorkPath;

    public YigoInstanceManager(){
        instanceStatus = new InstanceStatus(Status.UNAVALIABLE, "Yigo 环境不可用", initStartTime);
    }

    /**
     * 获取 Yigo 实例的当前执行状态
     * @return 返回实例的当前执行状态
     */
    public InstanceStatus getInstanceStatus() {
        if(UNAVALIABLE_REASONS.isEmpty()){
            // 如果没有unavaliable事件,则返回原始值
            return instanceStatus;
        }else {
            // 如果有unavaliable事件,则返回 Status.UNAVALIABLE 状态
            String reason = StringUtils.join(UNAVALIABLE_REASONS.values(),";\n");
            log.warn("Yigo服务出不可用状态,不可用原因有以下这些:\n"+reason);
            return new InstanceStatus(Status.UNAVALIABLE, reason, initStartTime);
        }
    }

    /**
     * 获取Yigo服务是否加载完成
     * @return
     */
    public boolean isInitializationComplete(){
        return initializationComplete;
    }

    /**
     * 在  SpringBoot 环境下准备 Yigo 执行环境
     * @param ctx spring的应用上下文
     * @param reload 重新加载标识
     * @return 当前环境是否在 "silent" 状态
     */
    public boolean bootPrepare(ApplicationContext ctx,boolean reload){
        final boolean[] result = new boolean[]{false};
        trans(Status.PREPARING, "Yigo 环境初始化", Status.WAITING, ()->{
            log.info("准备 Yigo 环境初始化 ...");
            //初始化工作目录: 将 Yigo 的配置文件处理到临时目录下
            PrepareResult prepareResult = YigoPropPreparation.doPrepare((ConfigurableApplicationContext) ctx);
            log.info("Yigo 环境初始化: workPath={} .", prepareResult.getWorkPath());
            this.initWorkPath = prepareResult.getWorkPath();
            result[0] = prepareResult.isSilentStart();

            //执行启动前准备事件
            doBootPrepare(ctx, prepareResult);

            //Yigo 初始化加载
            try {
                YigoServiceInitHelper.initYigoService(prepareResult,reload);
            } catch (Exception e) {
                ExceptionUtils.rethrow(e);
            }

            //获取附加初始化程序
            Map<String, YigoAdditionalInitiator> addInits = ctx.getBeansOfType(YigoAdditionalInitiator.class);
            log.info("Yigo 环境初始化: 加载附加初始化程序 {} 个 - [{}] .", addInits.size(), addInits.keySet().toString());
            this.additionalInitiators = addInits;

            return "Yigo 环境初始化完成, 包括 "+this.additionalInitiators.size()+" 个附加初始化程序";
        });
        return result[0];
    }

    private void doBootPrepare(ApplicationContext ctx,PrepareResult prepareResult){
        // 获取Yigo启动前准备事件
        Map<String, YigoBootPreparator> bootPreparatorMaps = ctx.getBeansOfType(YigoBootPreparator.class);
        // 如果不存在Yigo启动前准备事件则跳过
        if(null == bootPreparatorMaps || bootPreparatorMaps.isEmpty()){
            return;
        }
        log.info("Yigo 环境初始前: 加载附加准备程序 {} 个 - [{}] .", bootPreparatorMaps.size(), bootPreparatorMaps.keySet().toString());
        List<YigoBootPreparator> bootPreparators = new ArrayList<>(bootPreparatorMaps.values());
        // 按依赖排序
        bootPreparators = DependencySortCore.sort(bootPreparators);
        // 排序后按顺序执行
        for(YigoBootPreparator preparator:bootPreparators){
            preparator.prepare(prepareResult);
        }
    }

    /**
     * 在 SpringBoot 环境下加载 Yigo 程序
     */
    public void bootLoad(){
        trans(Status.LOADING, "加载 Yigo 程序", Status.READY, ()->{
            log.info("准备加载 Yigo 程序 ...");
            SessionUtils.processWithContext(null, (yigoCtx) -> {
                enabledServiceLoaderRegister(yigoCtx);
                log.info("加载 Yigo 程序: 注册 ServiceLoader 扩展程序完成 .");
                return null;
            });

            //Yigo 扩展组件的初始化过程
            for(Entry<String, YigoAdditionalInitiator> en: this.additionalInitiators.entrySet()) {
                SessionUtils.processWithContext(null, (yigoCtx) -> {
                    log.debug("加载 Yigo 程序: YigoAdditionalInitiator '{}' starting ...", en.getKey());
                    YigoAdditionalInitiator init = en.getValue();
                    init.init(yigoCtx);
                    log.info("加载 Yigo 程序: YigoAdditionalInitiator '{}' started.", en.getKey());
                    return true;
                });
            }
            resetWorkPath();
            log.info("加载 Yigo 程序完成 .");
            initializationComplete = true;
            return "加载 Yigo 程序完成";
        });
    }

    /**
     * 重置工作目录(设置为与 {@link #bootPrepare(ApplicationContext, boolean)} 后的工作目录一致)
     */
    public void resetWorkPath(){
        //FIXME 在 reload 的情况下, 因 yigo 另启线程初始化环境参数, 导致 workdir 不正确. 必须重新指定以保证 license 可以正确读取
        SvrInfo.setWorkDir(this.initWorkPath + File.separator);
        log.info("重新设置 WorkDir='{}' .", this.initWorkPath);
    }

    /**
     * 执行指定操作, 跟踪状态变化
     * @param begin 操作开始时的状态
     * @param transDescr 操作的描述
     * @param end 操作完成时的状态
     * @param callback 定义执行操作的回调方法
     */
    public void trans(Status begin, String transDescr, Status end, Supplier<String> callback){
        try{
            log.info("开始执行 '{}'({}->{}) ...", transDescr, begin, end);
            this.instanceStatus = new InstanceStatus(begin, transDescr, initStartTime);

            String msg = callback.get();
            log.info("执行 '{}'({}->{}) 成功: {}.", transDescr, begin, end, msg);
            this.instanceStatus = new InstanceStatus(end, msg, initStartTime);
        }catch(Exception ex){
            String errMsg = "执行 '" + transDescr + "' 失败: "+ex.getMessage();
            log.error(errMsg + " .", ex);
            this.instanceStatus = new InstanceStatus(Status.ERROR, errMsg, initStartTime);

            ExceptionUtils.rethrow(ex);
        }
    }

    /**
     * 记录不同来源的系统不可用原因
     * @param source 提交不可用原因的来源程序的类, 通常是一些模块级状态是否正常的检查程序, 比如读写分离部署时读库状态检查程序
     * @param reason 不可用原因描述
     */
    public void markUnavaliableReason(Class source, String reason){
        UNAVALIABLE_REASONS.put(source, reason);
    }

    /**
     * 记录不同来源的系统不可用原因
     * @param source 提交不可用原因的来源程序的类, 通常是一些模块级状态是否正常的检查程序, 比如读写分离部署时读库状态检查程序
     * @param exception 异常实例
     */
    public void markUnavaliableReason(Class source, Exception exception){
        markUnavaliableReason(source, errorI18N(exception));
    }

    private String errorI18N( Exception exception) {
        return StringTable.i18N(null,StringTable.MSG_SYS_EXEC_ERROR,exception.getClass(),exception.getMessage());
    }

    /**
     * 撤销不可用原因记录
     * @param source 提交不可用原因的来源程序的类
     */
    public void cleanUnavaliableEvent(Class source){
        UNAVALIABLE_REASONS.remove(source);
    }

    /**
     * FIXME 因 ERP 实现中复制了一份 functionmap, 导致 {@link ServiceLoaderRegisteringUtil#invoke(DefaultContext)} 注册的二开扩展函数无法被 ERP 使用
     * <br/>
     * 目前处理思路: 在 ERP 中通过 Enhance.xml 排序的方式提前执行 bootsupport 的二开注册过程, 同时通过判断避免重复注册
     * @param yigoCtx yigo上下文
     * @throws Throwable
     */
    private void enabledServiceLoaderRegister(DefaultContext yigoCtx) throws Throwable {
        IMetaFactory mf = yigoCtx.getVE().getMetaFactory();
        MetaEnhance enhance = mf.getEnhance("");
        if (null == enhance) {
            enhance = new MetaEnhance();
            FieldUtils.writeField(mf,"solutionEnhance",enhance,true);
        }
        MetaExtStartListener extStartListener = enhance.getStartListener();
        if(null != extStartListener) {
            // 如果已经在Enhance.xml注册过ServiceLoaderRegisteringStartListener,则不再继续使用 ServiceLoaderRegisteringUtil来注册
            for (MetaStartListener startListener : extStartListener) {
                if (startListener.getImpl().equals(ServiceLoaderRegisteringStartListener.class.getName())) {
                    return;
                }
            }
        }
        ServiceLoaderRegisteringUtil.invoke(yigoCtx);
    }

}