package com.bokesoft.distro.tech.bootsupport.starter.execctl.utils;

import com.bokesoft.distro.tech.bootsupport.starter.config.ExecTimeoutCtrlConfig;
import com.bokesoft.distro.tech.bootsupport.starter.execctl.config.dataobject.DataObjectCtrl;
import com.bokesoft.distro.tech.bootsupport.starter.execctl.config.dataobject.DataObjectGroup;
import com.bokesoft.distro.tech.bootsupport.starter.execctl.config.service.ServiceCtrl;
import com.bokesoft.distro.tech.bootsupport.starter.execctl.config.service.ServiceGroup;
import com.bokesoft.distro.tech.bootsupport.starter.execctl.exception.ExecTimeoutException;
import com.bokesoft.distro.tech.bootsupport.starter.execctl.model.StartTimeObject;
import com.bokesoft.distro.tech.bootsupport.starter.execctl.model.TimeoutConfig;
import jodd.util.Wildcard;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.time.DateFormatUtils;

import java.util.*;

public class ExecutionControlUtils {

    private static long defaultDbSqlTimeout;

    public static void setDefaultDbSqlTimeout(long defaultDbSqlTimeout) {
        ExecutionControlUtils.defaultDbSqlTimeout = defaultDbSqlTimeout;
    }

    /**
     * 日期格式
     */
    private static final String DATE_PATTERN = "yyyy-MM-dd HH:mm:ss.SSS";


    /**
     * 计算剩余分配时间
     */
    public static long calcAllocRemainedTime(ExecTimeoutCtrlConfig executionTimeoutControl, List<StartTimeObject> parallelSTOContainer, List<StartTimeObject> unParallelSTOContainer,
                                             Stack<String> dataObjectPaths, String serviceId, String instanceId) {
        //获取相关的开始时间对象
        List<StartTimeObject> startTimeObjects = getReleatedStartTimeObjectList(parallelSTOContainer, unParallelSTOContainer, instanceId);
        //获取配置的TimeoutConfig
        TimeoutConfig timeoutConfig = getTimeoutConfig(executionTimeoutControl, serviceId, dataObjectPaths);

        if (null != startTimeObjects && startTimeObjects.size() > 0) {
            long minEndTime = 0;
            List<String> endTimeLogs = new ArrayList<>();
            for (StartTimeObject startTimeObject : startTimeObjects) {
                StartTimeObject.StartTimeObjectType type = startTimeObject.getType();
                String executionPoint = startTimeObject.getExecutionPoint();
                Long timeoutMs = getTimeoutMs(timeoutConfig, type, executionPoint);
                if (null == timeoutMs || 0 == timeoutMs) {
                    endTimeLogs.add(startTimeObject.getType() + "-EndTime:nolimit");
                    continue;
                }
                long startTime = startTimeObject.getStartTime();
                long endTime = startTime + timeoutMs;
                endTimeLogs.add(type + "-EndTime:" + DateFormatUtils.format(endTime, DATE_PATTERN));
                if (0 == minEndTime) {
                    minEndTime = endTime;
                } else {
                    minEndTime = Math.min(minEndTime, endTime);
                }
            }
            if (0 != minEndTime) {
                long minTimeoutMs = minEndTime - System.currentTimeMillis();
                if (minTimeoutMs < 0) {
                    String throwMsg = getThrowExecutionTimeoutMsg(endTimeLogs);
                    throw new ExecTimeoutException(throwMsg);
                }
                return minTimeoutMs;
            }
        }
        return 0;
    }


    /**
     * 获取相关的开始时间对象集合
     * 并行集合会根据instanceId挑选StartTimeObject
     * 串行集合以计算整条链的超时的原则，所以全部涵盖在计算范围内
     */
    private static List<StartTimeObject> getReleatedStartTimeObjectList(List<StartTimeObject> parallelSTOContainer, List<StartTimeObject> unParallelSTOContainer, String instanceId) {
        List<StartTimeObject> releatedStartTimeObjects = new ArrayList<>();

        if (null != parallelSTOContainer && parallelSTOContainer.size() > 0) {
            if (StringUtils.isBlank(instanceId)) {
                releatedStartTimeObjects.addAll(parallelSTOContainer);
            } else {
                for (StartTimeObject startTimeObject : parallelSTOContainer) {
                    if (StringUtils.equals(instanceId, startTimeObject.getInstanceId())) {
                        releatedStartTimeObjects.add(startTimeObject);
                        break;
                    }
                }
            }
        }
        if (null != unParallelSTOContainer && unParallelSTOContainer.size() > 0) {
            releatedStartTimeObjects.addAll(unParallelSTOContainer);
        }
        return releatedStartTimeObjects;
    }

    /**
     * 获取TimeoutConfig
     */
    private static TimeoutConfig getTimeoutConfig(ExecTimeoutCtrlConfig executionTimeoutControl, String serviceId, Stack<String> dataObjectPaths) {
        TimeoutConfig timeoutConfig = new TimeoutConfig();
        getTimeoutConfigByService(timeoutConfig, executionTimeoutControl, serviceId);
        getTimeoutConfigByDataObject(timeoutConfig, executionTimeoutControl, dataObjectPaths);
        return timeoutConfig;
    }

    /**
     * 通过service配置获取TimeoutConfig
     */
    private static void getTimeoutConfigByService(TimeoutConfig timeoutConfig, ExecTimeoutCtrlConfig executionTimeoutControl, String serviceId) {
        ServiceCtrl serviceCtrl = getServiceCtrl(executionTimeoutControl, serviceId);
        if (null != serviceCtrl) {
            long requestTimeoutMs = serviceCtrl.getRequestTimeoutMs();
            long dbTransactionTimeoutMs = serviceCtrl.getDbTransactionTimeoutMs();
            long dbExecutionTimeoutMs = serviceCtrl.getDbExecutionTimeoutMs();
            timeoutConfig.setServiceTimeoutMs(requestTimeoutMs);
            timeoutConfig.setTransactionTimeoutMs(dbTransactionTimeoutMs);
            if (0 == dbExecutionTimeoutMs) {
                dbExecutionTimeoutMs = defaultDbSqlTimeout;
            }
            timeoutConfig.setDbExectionTimeoutMs(dbExecutionTimeoutMs);
        }
    }

    /**
     * 通过DataObject配置获取TimeoutConfig
     */
    private static void getTimeoutConfigByDataObject(TimeoutConfig timeoutConfig, ExecTimeoutCtrlConfig executionTimeoutControl, Stack<String> dataObjectPaths) {
        if (null != dataObjectPaths && dataObjectPaths.size() > 0) {
            for (String dataObjectPath : dataObjectPaths) {
                DataObjectCtrl dataObjectCtrl = getDataObjectCtrl(executionTimeoutControl, dataObjectPath);
                if (null != dataObjectCtrl) {
                    long dataObjectTimeoutMs = dataObjectCtrl.getDataObjectTimeoutMs();
                    long dbExecutionTimeoutMs = dataObjectCtrl.getDbExecutionTimeoutMs();
                    Map<String, Long> dataObjectTimeoutMsMap = timeoutConfig.getDataObjectTimeoutMsMap();
                    if (null == dataObjectTimeoutMsMap) {
                        dataObjectTimeoutMsMap = new HashMap<>();
                        timeoutConfig.setDataObjectTimeoutMsMap(dataObjectTimeoutMsMap);
                    }
                    dataObjectTimeoutMsMap.put(dataObjectPath, dataObjectTimeoutMs);
                    if (0 == dbExecutionTimeoutMs) {
                        dbExecutionTimeoutMs = defaultDbSqlTimeout;
                    }
                    timeoutConfig.setDbExectionTimeoutMs(dbExecutionTimeoutMs);
                }
            }
        }
    }

    /**
     * 获取超时时间的异常信息
     */
    private static String getThrowExecutionTimeoutMsg(List<String> endTimeLogs) {
        String msg = "stop execution due to timeout,{NOW:" + DateFormatUtils.format(new Date(), DATE_PATTERN) + "}";
        for (String endTimeLog : endTimeLogs) {
            msg = msg + ",{" + endTimeLog + "}";
        }
        return msg;
    }


    /**
     * 获取超时时间
     */
    private static Long getTimeoutMs(TimeoutConfig timeoutConfig, StartTimeObject.StartTimeObjectType type, String executionPoint) {
        if (null != timeoutConfig) {
            switch (type) {
                case SERVICE:
                    return timeoutConfig.getServiceTimeoutMs();
                case TRANSACTION:
                    return timeoutConfig.getTransactionTimeoutMs();
                case DATAOBJECT:
                    Long dataObjectTimeoutMs = null;
                    Map<String, Long> dataObjectTimeoutMsMap = timeoutConfig.getDataObjectTimeoutMsMap();
                    if (null != dataObjectTimeoutMsMap) {
                        dataObjectTimeoutMs = dataObjectTimeoutMsMap.get(executionPoint);
                    }
                    return dataObjectTimeoutMs;
                case SQLEXECUTION:
                    return timeoutConfig.getDbExectionTimeoutMs();
            }
        }
        return 0L;
    }


    /**
     * 获取Service相关配置
     */
    private static ServiceCtrl getServiceCtrl(ExecTimeoutCtrlConfig executionTimeoutControl, String serviceId) {
        if (null == executionTimeoutControl || StringUtils.isBlank(serviceId)) {
            return null;
        }
        List<ServiceCtrl> serviceCtrls = executionTimeoutControl.getServiceCtrls();
        if (null == serviceCtrls) {
            serviceCtrls = new ArrayList<>();
        }
        List<ServiceGroup> serviceGroups = executionTimeoutControl.getServiceGroups();
        if (null == serviceGroups) {
            serviceGroups = new ArrayList<>();
        }

        Map<String, List<String>> patternsMap = new HashMap<>();
        for (ServiceGroup serviceGroup : serviceGroups) {
            List<String> mergePatterns = new ArrayList<>();
            String groupName = serviceGroup.getName();
            List<String> patterns = serviceGroup.getPatterns();
            if (null != patterns && patterns.size() > 0) {
                mergePatterns.addAll(patterns);
            }
            patternsMap.put(groupName, mergePatterns);
        }
        for (ServiceCtrl serviceCtrl : serviceCtrls) {
            String groupName = serviceCtrl.getServiceGroup();
            List<String> mergePatterns = patternsMap.get(groupName);
            if (null == mergePatterns) {
                return null;
            }
            if (null != serviceCtrl.getPatterns()) {
                mergePatterns.addAll(serviceCtrl.getPatterns());
            }
            for (String pattern : mergePatterns) {
                if (StringUtils.equals(pattern, serviceId) || Wildcard.matchPath(serviceId, pattern)) {
                    return serviceCtrl;
                }
            }
        }
        return null;
    }

    /**
     * 获取DataObject配置
     */
    private static DataObjectCtrl getDataObjectCtrl(ExecTimeoutCtrlConfig executionTimeoutControl, String dataObjectPath) {
        if (null == executionTimeoutControl || StringUtils.isBlank(dataObjectPath)) {
            return null;
        }
        List<DataObjectCtrl> dataObjectCtrls = executionTimeoutControl.getDataObjectCtrls();
        if (null == dataObjectCtrls) {
            dataObjectCtrls = new ArrayList<>();
        }
        List<DataObjectGroup> dataObjectGroups = executionTimeoutControl.getDataObjectGroups();
        if (null == dataObjectGroups) {
            dataObjectGroups = new ArrayList<>();
        }

        Map<String, List<String>> patternsMap = new HashMap<>();
        for (DataObjectGroup dataObjectGroup : dataObjectGroups) {
            List<String> mergePatterns = new ArrayList<>();
            String groupName = dataObjectGroup.getName();
            List<String> patterns = dataObjectGroup.getPatterns();
            if (null != patterns) {
                mergePatterns.addAll(patterns);
            }
            patternsMap.put(groupName, mergePatterns);
        }

        for (DataObjectCtrl dataObjectCtrl : dataObjectCtrls) {
            String groupName = dataObjectCtrl.getDataObjectGroup();
            List<String> mergePatterns = patternsMap.get(groupName);
            if (null == mergePatterns) {
                return null;
            }
            if (null != dataObjectCtrl.getPatterns()) {
                mergePatterns.addAll(dataObjectCtrl.getPatterns());
            }
            for (String pattern : mergePatterns) {
                if (StringUtils.equals(pattern, dataObjectPath) || Wildcard.match(dataObjectPath, pattern)) {
                    return dataObjectCtrl;
                }
            }
        }
        return null;
    }

}
