package com.bokesoft.distro.tech.bootsupport.starter.beans;

import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.exception.ExceptionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.io.Resource;
import org.springframework.core.io.ResourceLoader;
import org.springframework.http.CacheControl;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Service;
import org.springframework.util.Assert;
import org.springframework.util.MultiValueMap;

import com.bokesoft.distro.tech.bootsupport.starter.api.YigoClientPageVariableProvider;
import com.bokesoft.distro.tech.bootsupport.starter.api.ctx.PageBuildContext;
import com.bokesoft.distro.tech.bootsupport.starter.runtime.YigoInstanceManager;
import com.bokesoft.distro.tech.bootsupport.starter.runtime.model.InstanceStatus;
import com.bokesoft.distro.tech.commons.basis.dependency.DependencySortCore;
import com.bokesoft.distro.tech.yigosupport.extension.utils.yigo.SessionUtils;
import com.bokesoft.yigo.mid.base.DefaultContext;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;

@Service
public class PagesBuildService {

    private static Logger logger = LoggerFactory.getLogger(PagesBuildService.class);

    private ResourceLoader resourceLoader;
    private List<YigoClientPageVariableProvider> varProviders;

    @Autowired
    private YigoInstanceManager yigoInstanceManager;

    private ObjectMapper objectMapper;

    public PagesBuildService(ResourceLoader resourceLoader, List<YigoClientPageVariableProvider> varProviders) {
        this.resourceLoader = resourceLoader;
        this.varProviders = varProviders;

        this.objectMapper = new ObjectMapper();
        this.objectMapper.setSerializationInclusion (JsonInclude.Include.NON_NULL);
    }
    /**
     * 构建页面
     * @param pageResourceLocation 页面文件的资源地址
     * @param pageBuildContext 页面构建上下文
     * @param clientID 当前用户ClientID
     * @param locale 当前语言
     * @return 返回页面内容
     */
    public ResponseEntity<String> buildPage(String pageResourceLocation, PageBuildContext pageBuildContext, String clientID, String locale) {
        if(InstanceStatus.Status.READY.equals(yigoInstanceManager.getInstanceStatus().getStatus())) {
            return SessionUtils.processWithContext(clientID, ctx -> {
                ctx.getEnv().setLocale(locale);
                return getStringResponseEntity(pageResourceLocation, pageBuildContext, ctx);
            });
        }else{
            try {
                    return getStringResponseEntity(pageResourceLocation, pageBuildContext, null);
            }catch (Exception e){
                return ExceptionUtils.rethrow(e);
            }
        }
    }

    private ResponseEntity<String> getStringResponseEntity(String pageResourceLocation,
        PageBuildContext pageBuildContext, DefaultContext yigoCtx) {
        pageBuildContext.setDefaultContext(yigoCtx);
        return buildPage(pageResourceLocation, pageBuildContext);
    }

    private ResponseEntity<String> buildPage(String resLoc, PageBuildContext pvc)  {
        Resource resource = resourceLoader.getResource(resLoc);
        Assert.notNull(resource, "无法找到资源 - '"+resLoc+"'");

        try(InputStream is = resource.getInputStream()){
            String html = IOUtils.toString(is, StandardCharsets.UTF_8);
            Assert.isTrue(StringUtils.isNotBlank(html), "资源 '"+resLoc+"' 内容为空");

            //特别针对 Yigo 的几个 JSP 的处理 - 清除 <%%> JSP 标签
            int docTypeStartPos = html.toUpperCase().indexOf("<!doctype html>".toUpperCase());
            if (docTypeStartPos>0){
                html = html.substring(docTypeStartPos);
            }

            //处理注入的变量
            List<YigoClientPageVariableProvider> providers = DependencySortCore.sort(this.varProviders);
            Map<String, Map<String, Object>> pageVars = new LinkedHashMap<>();
            for (YigoClientPageVariableProvider p: providers){
                if(p.support(pvc)){
                    Map<String, Object> vars = p.getVariables(pvc);
                    if (null!=vars && vars.size()>0){
                        pageVars.put(p.getClass().getName(), vars);
                    }
                }
            }
            html = injectPageVarsJs(html, pageVars, resLoc);

            MultiValueMap<String, String> headers = new HttpHeaders();
            headers.set(HttpHeaders.CACHE_CONTROL, CacheControl.noCache().cachePrivate().mustRevalidate().getHeaderValue());

            return new ResponseEntity<>(html, headers, HttpStatus.OK);
        }catch (Exception e){
            logger.error(e.getMessage(),e);
            return ExceptionUtils.rethrow(e);
        }
    }

    private String injectPageVarsJs(String html, Map<String, Map<String, Object>> pageVars, String resLoc){
        try{
            List<String> result = new ArrayList<>();
            result.add("/** Created by "+this.getClass().getName()+" */");
            result.add("window.YigoClientPageVariable = {};");

            for (Map.Entry<String, Map<String, Object>> pcEn: pageVars.entrySet()){
                String providerName = pcEn.getKey();
                result.add("/** Provided by "+providerName+" */");
                for (Map.Entry<String, Object> en: pcEn.getValue().entrySet()){
                    String varName = en.getKey();
                    Object val = en.getValue();
                    String json = this.objectMapper.writerWithDefaultPrettyPrinter().writeValueAsString(val);

                    result.add("window.YigoClientPageVariable."+varName + "=" + json + ";");
                }
            }
            int scriptInsertPoint = calcScriptInsertPoint(html, resLoc);
            String before = html.substring(0, scriptInsertPoint);
            String after = html.substring(scriptInsertPoint);

            return before
                    + "\n\n<script>\n" + StringUtils.join(result, "\n") + "\n</script>\n\n"
                    + after;
        }catch(JsonProcessingException e){
            return ExceptionUtils.rethrow(e);
        }
    }

    private int calcScriptInsertPoint(String html,String resLoc) {
        String sHead = "<head>";
        int result = html.toUpperCase().indexOf(sHead.toUpperCase());
        if( result>0 ){
            result = result+sHead.length();
        }
        Assert.isTrue( result > 0 ,
                "页面 '"+resLoc+"' 无效 - 无法找到 <head></head> 标签作为可插入点");
        return result;
    }

}
